/*
 * Merged with mainline ieee80211.h in Aug 2004.  Original ieee802_11
 * remains copyright by the original authors
 *
 * Portions of the merged code are based on Host AP (software wireless
 * LAN access point) driver for Intersil Prism2/2.5/3.
 *
 * Copyright (c) 2001-2002, SSH Communications Security Corp and Jouni Malinen
 * <jkmaline@cc.hut.fi>
 * Copyright (c) 2002-2003, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * Adaption to a generic IEEE 802.11 stack by James Ketrenos
 * <jketreno@linux.intel.com>
 * Copyright (c) 2004-2005, Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 *
 * API Version History
 * 1.0.x -- Initial version
 * 1.1.x -- Added radiotap, QoS, TIM, ieee80211_geo APIs,
 *          various structure changes, and crypto API init method
 */
#ifndef IEEE80211_H
#define IEEE80211_H
#include <linux/autoconf.h>
#include <linux/if_ether.h>	/* ETH_ALEN */
#include <linux/kernel.h>	/* ARRAY_SIZE */
#include <linux/wireless.h>

#define IEEE80211_VERSION "git-1.1.13"

#define IEEE80211_DATA_LEN		2304
/* Maximum size for the MA-UNITDATA primitive, 802.11 standard section
   6.2.1.1.2.

   The figure in section 7.1.2 suggests a body size of up to 2312
   bytes is allowed, which is a bit confusing, I suspect this
   represents the 2304 bytes of real data, plus a possible 8 bytes of
   WEP IV and ICV. (this interpretation suggested by Ramiro Barreiro) */
#define IEEE80211_ADDR_LEN  6               /* size of 802.11 address. Used in Atheros driver */
/* is 802.11 address multicast/broadcast? */
#define IEEE80211_IS_MULTICAST(_a)      (*(_a) & 0x01)

#define IEEE80211_1ADDR_LEN 10
#define IEEE80211_2ADDR_LEN 16
#define IEEE80211_3ADDR_LEN 24
#define IEEE80211_4ADDR_LEN 30
#define IEEE80211_FCS_LEN    4
#define IEEE80211_HLEN			(IEEE80211_4ADDR_LEN)
#define IEEE80211_FRAME_LEN		(IEEE80211_DATA_LEN + IEEE80211_HLEN)

#define MIN_FRAG_THRESHOLD     256U
#define	MAX_FRAG_THRESHOLD     2346U

/* Frame control field constants */
#define IEEE80211_FCTL_VERS		0x0003
#define IEEE80211_FCTL_FTYPE		0x000c
#define IEEE80211_FCTL_STYPE		0x00f0
#define IEEE80211_FCTL_TODS		0x0100
#define IEEE80211_FCTL_FROMDS		0x0200
#define IEEE80211_FCTL_MOREFRAGS	0x0400
#define IEEE80211_FCTL_RETRY		0x0800
#define IEEE80211_FCTL_PM		0x1000
#define IEEE80211_FCTL_MOREDATA		0x2000
#define IEEE80211_FCTL_PROTECTED	0x4000
#define IEEE80211_FCTL_ORDER		0x8000

#define IEEE80211_FTYPE_MGMT		0x0000
#define IEEE80211_FTYPE_CTL		0x0004
#define IEEE80211_FTYPE_DATA		0x0008

/* These are from Atheros driver src */
#define IEEE80211_FC0_VERSION_MASK              IEEE80211_FCTL_VERS
#define IEEE80211_FC0_VERSION_SHIFT             0
#define IEEE80211_FC0_VERSION_0                 0x00
#define IEEE80211_FC0_TYPE_MASK                 IEEE80211_FCTL_FTYPE
#define IEEE80211_FC0_TYPE_SHIFT                2
#define IEEE80211_FC0_TYPE_MGT                  0x00
#define IEEE80211_FC0_TYPE_CTL                  IEEE80211_FTYPE_CTL
#define IEEE80211_FC0_TYPE_DATA                 IEEE80211_FTYPE_DATA

#define IEEE80211_FC0_SUBTYPE_MASK              IEEE80211_FCTL_STYPE
#define IEEE80211_FC0_SUBTYPE_SHIFT             4
/* End of import from Atheros driver src */

/* management */
#define IEEE80211_STYPE_ASSOC_REQ	0x0000
#define IEEE80211_STYPE_ASSOC_RESP 	0x0010
#define IEEE80211_STYPE_REASSOC_REQ	0x0020
#define IEEE80211_STYPE_REASSOC_RESP	0x0030
#define IEEE80211_STYPE_PROBE_REQ	0x0040
#define IEEE80211_STYPE_PROBE_RESP	0x0050
#define IEEE80211_STYPE_BEACON		0x0080
#define IEEE80211_STYPE_ATIM		0x0090
#define IEEE80211_STYPE_DISASSOC	0x00A0
#define IEEE80211_STYPE_AUTH		0x00B0
#define IEEE80211_STYPE_DEAUTH		0x00C0
#define IEEE80211_STYPE_ACTION		0x00D0

/* control */
#define IEEE80211_STYPE_PSPOLL		0x00A0
#define IEEE80211_STYPE_RTS		0x00B0
#define IEEE80211_STYPE_CTS		0x00C0
#define IEEE80211_STYPE_ACK		0x00D0
#define IEEE80211_STYPE_CFEND		0x00E0
#define IEEE80211_STYPE_CFENDACK	0x00F0

/* data */
#define IEEE80211_STYPE_DATA		0x0000
#define IEEE80211_STYPE_DATA_CFACK	0x0010
#define IEEE80211_STYPE_DATA_CFPOLL	0x0020
#define IEEE80211_STYPE_DATA_CFACKPOLL	0x0030
#define IEEE80211_STYPE_NULLFUNC	0x0040
#define IEEE80211_STYPE_CFACK		0x0050
#define IEEE80211_STYPE_CFPOLL		0x0060
#define IEEE80211_STYPE_CFACKPOLL	0x0070
#define IEEE80211_STYPE_QOS_DATA        0x0080

#define IEEE80211_SCTL_FRAG		0x000F
#define IEEE80211_SCTL_SEQ		0xFFF0

/* QOS control */
#define IEEE80211_QCTL_TID		0x000F

/* These are from Atheros driver src */
#define WME_AC_TO_TID(_ac) (       \
        ((_ac) == WME_AC_VO) ? 6 : \
        ((_ac) == WME_AC_VI) ? 5 : \
        ((_ac) == WME_AC_BK) ? 1 : \
        0)

#define TID_TO_WME_AC(_tid) (      \
	(((_tid) == 0) || ((_tid) == 3)) ? WME_AC_BE : \
	(((_tid) == 1) || ((_tid) == 2)) ? WME_AC_BK : \
	(((_tid) == 4) || ((_tid) == 5)) ? WME_AC_VI : \
	WME_AC_VO)

#define IEEE80211_CRC_LEN               CRC_LENGTH

#define IEEE80211_TX_CHAINMASK_MIN      1
#define IEEE80211_TX_CHAINMASK_MAX      7

#define IEEE80211_RX_CHAINMASK_MIN      1
#define IEEE80211_RX_CHAINMASK_MAX      7

#define IEEE80211_CHANSWITCHANN_BYTES 5
#define IEEE80211_EXTCHANSWITCHANN_BYTES 6

/* for TYPE_MGT */
#define IEEE80211_FC0_SUBTYPE_ASSOC_REQ         IEEE80211_STYPE_ASSOC_REQ
#define IEEE80211_FC0_SUBTYPE_ASSOC_RESP        IEEE80211_STYPE_ASSOC_RESP
#define IEEE80211_FC0_SUBTYPE_REASSOC_REQ       IEEE80211_STYPE_REASSOC_REQ
#define IEEE80211_FC0_SUBTYPE_REASSOC_RESP      IEEE80211_STYPE_REASSOC_RESP
#define IEEE80211_FC0_SUBTYPE_PROBE_REQ         IEEE80211_STYPE_PROBE_REQ
#define IEEE80211_FC0_SUBTYPE_PROBE_RESP        IEEE80211_STYPE_PROBE_RESP
#define IEEE80211_FC0_SUBTYPE_BEACON            IEEE80211_STYPE_BEACON
#define IEEE80211_FC0_SUBTYPE_ATIM              IEEE80211_STYPE_ATIM
#define IEEE80211_FC0_SUBTYPE_DISASSOC          IEEE80211_STYPE_DISASSOC
#define IEEE80211_FC0_SUBTYPE_AUTH              IEEE80211_STYPE_AUTH
#define IEEE80211_FC0_SUBTYPE_DEAUTH            IEEE80211_STYPE_DEAUTH
#define IEEE80211_FC0_SUBTYPE_ACTION            IEEE80211_STYPE_ACTION
/* for TYPE_CTL */
#define IEEE80211_FC0_SUBTYPE_BAR               0x80
#define IEEE80211_FC0_SUBTYPE_PS_POLL           IEEE80211_STYPE_PSPOLL
#define IEEE80211_FC0_SUBTYPE_RTS               IEEE80211_STYPE_RTS
#define IEEE80211_FC0_SUBTYPE_CTS               IEEE80211_STYPE_CTS
#define IEEE80211_FC0_SUBTYPE_ACK               IEEE80211_STYPE_ACK
#define IEEE80211_FC0_SUBTYPE_CF_END            IEEE80211_STYPE_CFEND
#define IEEE80211_FC0_SUBTYPE_CF_END_ACK        IEEE80211_STYPE_CFENDACK
/* for TYPE_DATA (bit combination) */
#define IEEE80211_FC0_SUBTYPE_DATA              IEEE80211_STYPE_DATA
#define IEEE80211_FC0_SUBTYPE_CF_ACK            IEEE80211_STYPE_DATA_CFACK
#define IEEE80211_FC0_SUBTYPE_CF_POLL           IEEE80211_STYPE_DATA_CFPOLL
#define IEEE80211_FC0_SUBTYPE_CF_ACPL           IEEE80211_STYPE_DATA_CFACKPOLL
#define IEEE80211_FC0_SUBTYPE_NODATA            IEEE80211_STYPE_NULLFUNC
#define IEEE80211_FC0_SUBTYPE_CFACK             IEEE80211_STYPE_CFACK
#define IEEE80211_FC0_SUBTYPE_CFPOLL            IEEE80211_STYPE_CFPOLL
#define IEEE80211_FC0_SUBTYPE_CF_ACK_CF_ACK     IEEE80211_STYPE_CFACKPOLL
#define IEEE80211_FC0_SUBTYPE_QOS               IEEE80211_STYPE_QOS_DATA
#define IEEE80211_FC0_SUBTYPE_QOS_NULL          0xc0

#define IEEE80211_FC1_DIR_MASK                  0x03
#define IEEE80211_FC1_DIR_NODS                  0x00/* STA->STA */
#define IEEE80211_FC1_DIR_TODS                  0x01/* STA->AP  */
#define IEEE80211_FC1_DIR_FROMDS                0x02/* AP ->STA */
#define IEEE80211_FC1_DIR_DSTODS                0x03/* AP ->AP  */

#define IEEE80211_FC1_MORE_FRAG                 0x04
#define IEEE80211_FC1_RETRY                  	0x08 
#define IEEE80211_FC1_PWR_MGT                   0x10
#define IEEE80211_FC1_MORE_DATA                 0x20
#define IEEE80211_FC1_WEP                       0x40
#define IEEE80211_FC1_ORDER                     0x80
/* End import from Atheros driver src */

/* debug macros */

#ifdef CONFIG_IEEE80211_DEBUG
extern u32 ieee80211_debug_level;
#define IEEE80211_DEBUG(level, fmt, args...) \
do { if (ieee80211_debug_level & (level)) \
  printk(KERN_DEBUG "ieee80211: %c %s " fmt, \
         in_interrupt() ? 'I' : 'U', __FUNCTION__ , ## args); } while (0)
#else
#define IEEE80211_DEBUG(level, fmt, args...) do {} while (0)
#endif				/* CONFIG_IEEE80211_DEBUG */

/* debug macros not dependent on CONFIG_IEEE80211_DEBUG */

#define MAC_FMT "%02x:%02x:%02x:%02x:%02x:%02x"
#define MAC_ARG(x) ((u8*)(x))[0],((u8*)(x))[1],((u8*)(x))[2],((u8*)(x))[3],((u8*)(x))[4],((u8*)(x))[5]

/* escape_essid() is intended to be used in debug (and possibly error)
 * messages. It should never be used for passing essid to user space. */
const char *escape_essid(const char *essid, u8 essid_len);

/*
 * To use the debug system:
 *
 * If you are defining a new debug classification, simply add it to the #define
 * list here in the form of:
 *
 * #define IEEE80211_DL_xxxx VALUE
 *
 * shifting value to the left one bit from the previous entry.  xxxx should be
 * the name of the classification (for example, WEP)
 *
 * You then need to either add a IEEE80211_xxxx_DEBUG() macro definition for your
 * classification, or use IEEE80211_DEBUG(IEEE80211_DL_xxxx, ...) whenever you want
 * to send output to that classification.
 *
 * To add your debug level to the list of levels seen when you perform
 *
 * % cat /proc/net/ieee80211/debug_level
 *
 * you simply need to add your entry to the ieee80211_debug_level array.
 *
 * If you do not see debug_level in /proc/net/ieee80211 then you do not have
 * CONFIG_IEEE80211_DEBUG defined in your kernel configuration
 *
 */

#define IEEE80211_DL_INFO          (1<<0)
#define IEEE80211_DL_WX            (1<<1)
#define IEEE80211_DL_SCAN          (1<<2)
#define IEEE80211_DL_STATE         (1<<3)
#define IEEE80211_DL_MGMT          (1<<4)
#define IEEE80211_DL_FRAG          (1<<5)
#define IEEE80211_DL_DROP          (1<<7)

#define IEEE80211_DL_TX            (1<<8)
#define IEEE80211_DL_RX            (1<<9)
#define IEEE80211_DL_QOS           (1<<31)

#define IEEE80211_ERROR(f, a...) printk(KERN_ERR "ieee80211: " f, ## a)
#define IEEE80211_WARNING(f, a...) printk(KERN_WARNING "ieee80211: " f, ## a)
#define IEEE80211_DEBUG_INFO(f, a...)   IEEE80211_DEBUG(IEEE80211_DL_INFO, f, ## a)

#define IEEE80211_DEBUG_WX(f, a...)     IEEE80211_DEBUG(IEEE80211_DL_WX, f, ## a)
#define IEEE80211_DEBUG_SCAN(f, a...)   IEEE80211_DEBUG(IEEE80211_DL_SCAN, f, ## a)
#define IEEE80211_DEBUG_STATE(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_STATE, f, ## a)
#define IEEE80211_DEBUG_MGMT(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_MGMT, f, ## a)
#define IEEE80211_DEBUG_FRAG(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_FRAG, f, ## a)
#define IEEE80211_DEBUG_DROP(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_DROP, f, ## a)
#define IEEE80211_DEBUG_TX(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_TX, f, ## a)
#define IEEE80211_DEBUG_RX(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_RX, f, ## a)
#define IEEE80211_DEBUG_QOS(f, a...)  IEEE80211_DEBUG(IEEE80211_DL_QOS, f, ## a)
#include <linux/netdevice.h>
#include <linux/wireless.h>
#include <linux/if_arp.h>	/* ARPHRD_ETHER */

#ifndef WIRELESS_SPY
#define WIRELESS_SPY		/* enable iwspy support */
#endif
#include <net/iw_handler.h>	/* new driver API */

#ifndef ETH_P_PAE
#define ETH_P_PAE 0x888E	/* Port Access Entity (IEEE 802.1X) */
#endif				/* ETH_P_PAE */

#define ETH_P_PREAUTH 0x88C7	/* IEEE 802.11i pre-authentication */

#ifndef ETH_P_80211_RAW
#define ETH_P_80211_RAW (ETH_P_ECONET + 1)
#endif

/* IEEE 802.11 defines */

#define P80211_OUI_LEN 3

struct ieee80211_snap_hdr {

	u8 dsap;		/* always 0xAA */
	u8 ssap;		/* always 0xAA */
	u8 ctrl;		/* always 0x03 */
	u8 oui[P80211_OUI_LEN];	/* organizational universal id */

} __attribute__ ((packed));

#define SNAP_SIZE sizeof(struct ieee80211_snap_hdr)

#define WLAN_FC_GET_VERS(fc) ((fc) & IEEE80211_FCTL_VERS)
#define WLAN_FC_GET_TYPE(fc) ((fc) & IEEE80211_FCTL_FTYPE)
#define WLAN_FC_GET_STYPE(fc) ((fc) & IEEE80211_FCTL_STYPE)

#define WLAN_GET_SEQ_FRAG(seq) ((seq) & IEEE80211_SCTL_FRAG)
#define WLAN_GET_SEQ_SEQ(seq)  (((seq) & IEEE80211_SCTL_SEQ) >> 4)

/* Authentication algorithms */
#define WLAN_AUTH_OPEN 0
#define WLAN_AUTH_SHARED_KEY 1
#define WLAN_AUTH_LEAP 2

#define WLAN_AUTH_CHALLENGE_LEN 128

#define WLAN_CAPABILITY_ESS (1<<0)
#define WLAN_CAPABILITY_IBSS (1<<1)
#define WLAN_CAPABILITY_CF_POLLABLE (1<<2)
#define WLAN_CAPABILITY_CF_POLL_REQUEST (1<<3)
#define WLAN_CAPABILITY_PRIVACY (1<<4)
#define WLAN_CAPABILITY_SHORT_PREAMBLE (1<<5)
#define WLAN_CAPABILITY_PBCC (1<<6)
#define WLAN_CAPABILITY_CHANNEL_AGILITY (1<<7)
#define WLAN_CAPABILITY_SPECTRUM_MGMT (1<<8)
#define WLAN_CAPABILITY_QOS (1<<9)
#define WLAN_CAPABILITY_SHORT_SLOT_TIME (1<<10)
#define WLAN_CAPABILITY_DSSS_OFDM (1<<13)

/* 802.11g ERP information element */
#define WLAN_ERP_NON_ERP_PRESENT (1<<0)
#define WLAN_ERP_USE_PROTECTION (1<<1)
#define WLAN_ERP_BARKER_PREAMBLE (1<<2)

/* Status codes */
enum ieee80211_statuscode {
	WLAN_STATUS_SUCCESS = 0,
	WLAN_STATUS_UNSPECIFIED_FAILURE = 1,
	WLAN_STATUS_CAPS_UNSUPPORTED = 10,
	WLAN_STATUS_REASSOC_NO_ASSOC = 11,
	WLAN_STATUS_ASSOC_DENIED_UNSPEC = 12,
	WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG = 13,
	WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION = 14,
	WLAN_STATUS_CHALLENGE_FAIL = 15,
	WLAN_STATUS_AUTH_TIMEOUT = 16,
	WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA = 17,
	WLAN_STATUS_ASSOC_DENIED_RATES = 18,
	/* 802.11b */
	WLAN_STATUS_ASSOC_DENIED_NOSHORTPREAMBLE = 19,
	WLAN_STATUS_ASSOC_DENIED_NOPBCC = 20,
	WLAN_STATUS_ASSOC_DENIED_NOAGILITY = 21,
	/* 802.11h */
	WLAN_STATUS_ASSOC_DENIED_NOSPECTRUM = 22,
	WLAN_STATUS_ASSOC_REJECTED_BAD_POWER = 23,
	WLAN_STATUS_ASSOC_REJECTED_BAD_SUPP_CHAN = 24,
	/* 802.11g */
	WLAN_STATUS_ASSOC_DENIED_NOSHORTTIME = 25,
	WLAN_STATUS_ASSOC_DENIED_NODSSSOFDM = 26,
	/* 802.11i */
	WLAN_STATUS_INVALID_IE = 40,
	WLAN_STATUS_INVALID_GROUP_CIPHER = 41,
	WLAN_STATUS_INVALID_PAIRWISE_CIPHER = 42,
	WLAN_STATUS_INVALID_AKMP = 43,
	WLAN_STATUS_UNSUPP_RSN_VERSION = 44,
	WLAN_STATUS_INVALID_RSN_IE_CAP = 45,
	WLAN_STATUS_CIPHER_SUITE_REJECTED = 46,
};

/* Reason codes */
enum ieee80211_reasoncode {
	WLAN_REASON_UNSPECIFIED = 1,
	WLAN_REASON_PREV_AUTH_NOT_VALID = 2,
	WLAN_REASON_DEAUTH_LEAVING = 3,
	WLAN_REASON_DISASSOC_DUE_TO_INACTIVITY = 4,
	WLAN_REASON_DISASSOC_AP_BUSY = 5,
	WLAN_REASON_CLASS2_FRAME_FROM_NONAUTH_STA = 6,
	WLAN_REASON_CLASS3_FRAME_FROM_NONASSOC_STA = 7,
	WLAN_REASON_DISASSOC_STA_HAS_LEFT = 8,
	WLAN_REASON_STA_REQ_ASSOC_WITHOUT_AUTH = 9,
	/* 802.11h */
	WLAN_REASON_DISASSOC_BAD_POWER = 10,
	WLAN_REASON_DISASSOC_BAD_SUPP_CHAN = 11,
	/* 802.11i */
	WLAN_REASON_INVALID_IE = 13,
	WLAN_REASON_MIC_FAILURE = 14,
	WLAN_REASON_4WAY_HANDSHAKE_TIMEOUT = 15,
	WLAN_REASON_GROUP_KEY_HANDSHAKE_TIMEOUT = 16,
	WLAN_REASON_IE_DIFFERENT = 17,
	WLAN_REASON_INVALID_GROUP_CIPHER = 18,
	WLAN_REASON_INVALID_PAIRWISE_CIPHER = 19,
	WLAN_REASON_INVALID_AKMP = 20,
	WLAN_REASON_UNSUPP_RSN_VERSION = 21,
	WLAN_REASON_INVALID_RSN_IE_CAP = 22,
	WLAN_REASON_IEEE8021X_FAILED = 23,
	WLAN_REASON_CIPHER_SUITE_REJECTED = 24,
};

/* Action categories - 802.11h */
enum ieee80211_actioncategories {
	WLAN_ACTION_SPECTRUM_MGMT = 0,
	/* Reserved 1-127  */
	/* Error    128-255 */
};

/* Action details - 802.11h */
enum ieee80211_actiondetails {
	WLAN_ACTION_CATEGORY_MEASURE_REQUEST = 0,
	WLAN_ACTION_CATEGORY_MEASURE_REPORT = 1,
	WLAN_ACTION_CATEGORY_TPC_REQUEST = 2,
	WLAN_ACTION_CATEGORY_TPC_REPORT = 3,
	WLAN_ACTION_CATEGORY_CHANNEL_SWITCH = 4,
	/* 5 - 255 Reserved */
};

#define IEEE80211_STATMASK_SIGNAL (1<<0)
#define IEEE80211_STATMASK_RSSI (1<<1)
#define IEEE80211_STATMASK_NOISE (1<<2)
#define IEEE80211_STATMASK_RATE (1<<3)
#define IEEE80211_STATMASK_WEMASK 0x7

#define IEEE80211_CCK_MODULATION    (1<<0)
#define IEEE80211_OFDM_MODULATION   (1<<1)

#define IEEE80211_24GHZ_BAND     (1<<0)
#define IEEE80211_52GHZ_BAND     (1<<1)

#define IEEE80211_CCK_RATE_1MB		        0x02
#define IEEE80211_CCK_RATE_2MB		        0x04
#define IEEE80211_CCK_RATE_5MB		        0x0B
#define IEEE80211_CCK_RATE_11MB		        0x16
#define IEEE80211_OFDM_RATE_6MB		        0x0C
#define IEEE80211_OFDM_RATE_9MB		        0x12
#define IEEE80211_OFDM_RATE_12MB		0x18
#define IEEE80211_OFDM_RATE_18MB		0x24
#define IEEE80211_OFDM_RATE_24MB		0x30
#define IEEE80211_OFDM_RATE_36MB		0x48
#define IEEE80211_OFDM_RATE_48MB		0x60
#define IEEE80211_OFDM_RATE_54MB		0x6C
#define IEEE80211_BASIC_RATE_MASK		0x80

#define IEEE80211_CCK_RATE_1MB_MASK		(1<<0)
#define IEEE80211_CCK_RATE_2MB_MASK		(1<<1)
#define IEEE80211_CCK_RATE_5MB_MASK		(1<<2)
#define IEEE80211_CCK_RATE_11MB_MASK		(1<<3)
#define IEEE80211_OFDM_RATE_6MB_MASK		(1<<4)
#define IEEE80211_OFDM_RATE_9MB_MASK		(1<<5)
#define IEEE80211_OFDM_RATE_12MB_MASK		(1<<6)
#define IEEE80211_OFDM_RATE_18MB_MASK		(1<<7)
#define IEEE80211_OFDM_RATE_24MB_MASK		(1<<8)
#define IEEE80211_OFDM_RATE_36MB_MASK		(1<<9)
#define IEEE80211_OFDM_RATE_48MB_MASK		(1<<10)
#define IEEE80211_OFDM_RATE_54MB_MASK		(1<<11)

#define IEEE80211_CCK_RATES_MASK	        0x0000000F
#define IEEE80211_CCK_BASIC_RATES_MASK	(IEEE80211_CCK_RATE_1MB_MASK | \
	IEEE80211_CCK_RATE_2MB_MASK)
#define IEEE80211_CCK_DEFAULT_RATES_MASK	(IEEE80211_CCK_BASIC_RATES_MASK | \
        IEEE80211_CCK_RATE_5MB_MASK | \
        IEEE80211_CCK_RATE_11MB_MASK)

#define IEEE80211_OFDM_RATES_MASK		0x00000FF0
#define IEEE80211_OFDM_BASIC_RATES_MASK	(IEEE80211_OFDM_RATE_6MB_MASK | \
	IEEE80211_OFDM_RATE_12MB_MASK | \
	IEEE80211_OFDM_RATE_24MB_MASK)
#define IEEE80211_OFDM_DEFAULT_RATES_MASK	(IEEE80211_OFDM_BASIC_RATES_MASK | \
	IEEE80211_OFDM_RATE_9MB_MASK  | \
	IEEE80211_OFDM_RATE_18MB_MASK | \
	IEEE80211_OFDM_RATE_36MB_MASK | \
	IEEE80211_OFDM_RATE_48MB_MASK | \
	IEEE80211_OFDM_RATE_54MB_MASK)
#define IEEE80211_DEFAULT_RATES_MASK (IEEE80211_OFDM_DEFAULT_RATES_MASK | \
                                IEEE80211_CCK_DEFAULT_RATES_MASK)

#define IEEE80211_NUM_OFDM_RATES	    8
#define IEEE80211_NUM_CCK_RATES	            4
#define IEEE80211_OFDM_SHIFT_MASK_A         4

/* NOTE: This data is for statistical purposes; not all hardware provides this
 *       information for frames received.
 *       For ieee80211_rx_mgt, you need to set at least the 'len' parameter.
 */
struct ieee80211_rx_stats {
	u32 mac_time;
	s8 rssi;
	u8 signal;
	u8 noise;
	u16 rate;		/* in 100 kbps */
	u8 received_channel;
	u8 control;
	u8 mask;
	u8 freq;
	u16 len;
	u64 tsf;
	u32 beacon_time;
};

/* IEEE 802.11 requires that STA supports concurrent reception of at least
 * three fragmented frames. This define can be increased to support more
 * concurrent frames, but it should be noted that each entry can consume about
 * 2 kB of RAM and increasing cache size will slow down frame reassembly. */
#define IEEE80211_FRAG_CACHE_LEN 4

struct ieee80211_frag_entry {
	unsigned long first_frag_time;
	unsigned int seq;
	unsigned int last_frag;
	struct sk_buff *skb;
	u8 src_addr[ETH_ALEN];
	u8 dst_addr[ETH_ALEN];
};

struct ieee80211_stats {
	unsigned int tx_unicast_frames;
	unsigned int tx_multicast_frames;
	unsigned int tx_fragments;
	unsigned int tx_unicast_octets;
	unsigned int tx_multicast_octets;
	unsigned int tx_deferred_transmissions;
	unsigned int tx_single_retry_frames;
	unsigned int tx_multiple_retry_frames;
	unsigned int tx_retry_limit_exceeded;
	unsigned int tx_discards;
	unsigned int rx_unicast_frames;
	unsigned int rx_multicast_frames;
	unsigned int rx_fragments;
	unsigned int rx_unicast_octets;
	unsigned int rx_multicast_octets;
	unsigned int rx_fcs_errors;
	unsigned int rx_discards_no_buffer;
	unsigned int tx_discards_wrong_sa;
	unsigned int rx_discards_undecryptable;
	unsigned int rx_message_in_msg_fragments;
	unsigned int rx_message_in_bad_msg_fragments;
};

struct ieee80211_device;

#include "ieee80211_crypt.h"

#define SEC_KEY_1		(1<<0)
#define SEC_KEY_2		(1<<1)
#define SEC_KEY_3		(1<<2)
#define SEC_KEY_4		(1<<3)
#define SEC_ACTIVE_KEY		(1<<4)
#define SEC_AUTH_MODE		(1<<5)
#define SEC_UNICAST_GROUP	(1<<6)
#define SEC_LEVEL		(1<<7)
#define SEC_ENABLED		(1<<8)
#define SEC_ENCRYPT		(1<<9)

#define SEC_LEVEL_0		0	/* None */
#define SEC_LEVEL_1		1	/* WEP 40 and 104 bit */
#define SEC_LEVEL_2		2	/* Level 1 + TKIP */
#define SEC_LEVEL_2_CKIP	3	/* Level 1 + CKIP */
#define SEC_LEVEL_3		4	/* Level 2 + CCMP */

#define SEC_ALG_NONE		0
#define SEC_ALG_WEP		1
#define SEC_ALG_TKIP		2
#define SEC_ALG_CCMP		3

#define WEP_KEYS		4
#define WEP_KEY_LEN		13
#define SCM_KEY_LEN		32
#define SCM_TEMPORAL_KEY_LENGTH	16

struct ieee80211_security {
	u16 active_key:2,
	    enabled:1,
	    auth_mode:2, auth_algo:4, unicast_uses_group:1, encrypt:1;
	u8 encode_alg[WEP_KEYS];
	u8 key_sizes[WEP_KEYS];
	u8 keys[WEP_KEYS][SCM_KEY_LEN];
	u8 level;
	u16 flags;
} __attribute__ ((packed));

/*

 802.11 data frame from AP

      ,-------------------------------------------------------------------.
Bytes |  2   |  2   |    6    |    6    |    6    |  2   | 0..2312 |   4  |
      |------|------|---------|---------|---------|------|---------|------|
Desc. | ctrl | dura |  DA/RA  |   TA    |    SA   | Sequ |  frame  |  fcs |
      |      | tion | (BSSID) |         |         | ence |  data   |      |
      `-------------------------------------------------------------------'

Total: 28-2340 bytes

*/

#define BEACON_PROBE_SSID_ID_POSITION 12

/* Management Frame Information Element Types */
enum ieee80211_mfie {
	MFIE_TYPE_SSID = 0,
	MFIE_TYPE_RATES = 1,
	MFIE_TYPE_FH_SET = 2,
	MFIE_TYPE_DS_SET = 3,
	MFIE_TYPE_CF_SET = 4,
	MFIE_TYPE_TIM = 5,
	MFIE_TYPE_IBSS_SET = 6,
	MFIE_TYPE_COUNTRY = 7,
	MFIE_TYPE_HOP_PARAMS = 8,
	MFIE_TYPE_HOP_TABLE = 9,
	MFIE_TYPE_REQUEST = 10,
	MFIE_TYPE_CHALLENGE = 16,
	MFIE_TYPE_POWER_CONSTRAINT = 32,
	MFIE_TYPE_POWER_CAPABILITY = 33,
	MFIE_TYPE_TPC_REQUEST = 34,
	MFIE_TYPE_TPC_REPORT = 35,
	MFIE_TYPE_SUPP_CHANNELS = 36,
	MFIE_TYPE_CSA = 37,
	MFIE_TYPE_MEASURE_REQUEST = 38,
	MFIE_TYPE_MEASURE_REPORT = 39,
	MFIE_TYPE_QUIET = 40,
	MFIE_TYPE_IBSS_DFS = 41,
	MFIE_TYPE_ERP_INFO = 42,
	MFIE_TYPE_RSN = 48,
	MFIE_TYPE_RATES_EX = 50,
	MFIE_TYPE_GENERIC = 221,
	MFIE_TYPE_QOS_PARAMETER = 222,
};

/* Minimal header; can be used for passing 802.11 frames with sufficient
 * information to determine what type of underlying data type is actually
 * stored in the data. */
struct ieee80211_hdr {
	__le16 frame_ctl;
	__le16 duration_id;
	u8 payload[0];
} __attribute__ ((packed));

struct ieee80211_hdr_1addr {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_dur[2];
	};
	u8 addr1[ETH_ALEN];
#define i_addr1 addr1
	u8 payload[0];
} __attribute__ ((packed));
#define ieee80211_frame_ack ieee80211_hdr_1addr

struct ieee80211_hdr_2addr {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_aidordur[2];
		u8 i_dur[2];
	};
	u8 addr1[ETH_ALEN];
#define i_addr1 addr1
	u8 addr2[ETH_ALEN];
#define i_addr2 addr2
	u8 payload[0];
} __attribute__ ((packed));
#define ieee80211_frame_min ieee80211_hdr_2addr
#define ieee80211_ctlframe_addr2 ieee80211_hdr_2addr

struct ieee80211_frame_bar {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_dur[2];
	};
        u8 i_ra[ETH_ALEN];
        u8 i_ta[ETH_ALEN];
	__le16 i_ctl;
	__le16 i_seq;
        /* FCS */
} __attribute__ ((packed));

struct ieee80211_hdr_3addr {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_dur[2];
	};
	u8 addr1[ETH_ALEN];
#define i_addr1 addr1
	u8 addr2[ETH_ALEN];
#define i_addr2 addr2
	u8 addr3[ETH_ALEN];
#define i_addr3 addr3
	union {
		__le16 seq_ctl;
		u8 i_seq[2];
	};
	u8 payload[0];
} __attribute__ ((packed));
#define ieee80211_frame ieee80211_hdr_3addr

struct ieee80211_hdr_4addr {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_dur[2];
	};
	u8 addr1[ETH_ALEN];
#define i_addr1 addr1
	u8 addr2[ETH_ALEN];
#define i_addr2 addr2
	u8 addr3[ETH_ALEN];
#define i_addr3 addr3
	union {
		__le16 seq_ctl;
		u8 i_seq[2];
	};
	u8 addr4[ETH_ALEN];
#define i_addr4 addr4
	u8 payload[0];
} __attribute__ ((packed));
#define ieee80211_frame_addr4 ieee80211_hdr_4addr

struct ieee80211_hdr_3addrqos {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_dur[2];
	};
	u8 addr1[ETH_ALEN];
	u8 addr2[ETH_ALEN];
	u8 addr3[ETH_ALEN];
	union {
		__le16 seq_ctl;
		u8 i_seq[2];
	};
	u8 payload[0];
	union {
		__le16 qos_ctl;
		u8 i_qos[2];
	};
} __attribute__ ((packed));
#define ieee80211_qosframe ieee80211_hdr_3addrqos
struct ieee80211_hdr_4addrqos {
	union {
		__le16 frame_ctl;
		u8 i_fc[2];
	};
	union {
		__le16 duration_id;
		u8 i_dur[2];
	};
	u8 addr1[ETH_ALEN];
	u8 addr2[ETH_ALEN];
	u8 addr3[ETH_ALEN];
	union {
		__le16 seq_ctl;
		u8 i_seq[2];
	};
	u8 addr4[ETH_ALEN];
	u8 payload[0];
	union {
		__le16 qos_ctl;
		u8 i_qos[2];
	};
} __attribute__ ((packed));
#define ieee80211_qosframe_addr4 ieee80211_hdr_4addrqos

struct ieee80211_info_element {
	u8 id;
	u8 len;
	u8 data[0];
} __attribute__ ((packed));

/*
 * These are the data types that can make up management packets
 *
	u16 auth_algorithm;
	u16 auth_sequence;
	u16 beacon_interval;
	u16 capability;
	u8 current_ap[ETH_ALEN];
	u16 listen_interval;
	struct {
		u16 association_id:14, reserved:2;
	} __attribute__ ((packed));
	u32 time_stamp[2];
	u16 reason;
	u16 status;
*/

struct ieee80211_auth {
	struct ieee80211_hdr_3addr header;
	__le16 algorithm;
	__le16 transaction;
	__le16 status;
	/* challenge */
	struct ieee80211_info_element info_element[0];
} __attribute__ ((packed));

struct ieee80211_channel_switch {
	u8 id;
	u8 len;
	u8 mode;
	u8 channel;
	u8 count;
} __attribute__ ((packed));

struct ieee80211_action {
	struct ieee80211_hdr_3addr header;
	u8 category;
	u8 action;
#define ia_category category
#define ia_action action
	union {
		struct ieee80211_action_exchange {
			u8 token;
			struct ieee80211_info_element info_element[0];
		} exchange;
		struct ieee80211_channel_switch channel_switch;

	} format;
} __attribute__ ((packed));

struct ieee80211_disassoc {
	struct ieee80211_hdr_3addr header;
	__le16 reason;
} __attribute__ ((packed));

/* Alias deauth for disassoc */
#define ieee80211_deauth ieee80211_disassoc

struct ieee80211_probe_request {
	struct ieee80211_hdr_3addr header;
	/* SSID, supported rates */
	struct ieee80211_info_element info_element[0];
} __attribute__ ((packed));

struct ieee80211_probe_response {
	struct ieee80211_hdr_3addr header;
	u32 time_stamp[2];
	__le16 beacon_interval;
	__le16 capability;
	/* SSID, supported rates, FH params, DS params,
	 * CF params, IBSS params, TIM (if beacon), RSN */
	struct ieee80211_info_element info_element[0];
} __attribute__ ((packed));

/* Alias beacon for probe_response */
#define ieee80211_beacon ieee80211_probe_response

struct ieee80211_assoc_request {
	struct ieee80211_hdr_3addr header;
	__le16 capability;
	__le16 listen_interval;
	/* SSID, supported rates, RSN */
	struct ieee80211_info_element info_element[0];
} __attribute__ ((packed));

struct ieee80211_reassoc_request {
	struct ieee80211_hdr_3addr header;
	__le16 capability;
	__le16 listen_interval;
	u8 current_ap[ETH_ALEN];
	struct ieee80211_info_element info_element[0];
} __attribute__ ((packed));

struct ieee80211_assoc_response {
	struct ieee80211_hdr_3addr header;
	__le16 capability;
	__le16 status;
	__le16 aid;
	/* supported rates */
	struct ieee80211_info_element info_element[0];
} __attribute__ ((packed));

struct ieee80211_txb {
	u8 nr_frags;
	u8 encrypted;
	u8 rts_included;
	u8 reserved;
	__le16 frag_size;
	__le16 payload_size;
	struct sk_buff *fragments[0];
};

/* Import from Atheros driver */
/*
 * Reason codes
 *
 * Unlisted codes are reserved
 */
enum {
        IEEE80211_REASON_UNSPECIFIED            = 1,
        IEEE80211_REASON_AUTH_EXPIRE            = 2,
        IEEE80211_REASON_AUTH_LEAVE             = 3,
        IEEE80211_REASON_ASSOC_EXPIRE           = 4,
        IEEE80211_REASON_ASSOC_TOOMANY          = 5,
        IEEE80211_REASON_NOT_AUTHED             = 6,
        IEEE80211_REASON_NOT_ASSOCED            = 7,
        IEEE80211_REASON_ASSOC_LEAVE            = 8,
        IEEE80211_REASON_ASSOC_NOT_AUTHED       = 9,

        IEEE80211_REASON_RSN_REQUIRED           = 11,
        IEEE80211_REASON_RSN_INCONSISTENT       = 12,
        IEEE80211_REASON_IE_INVALID             = 13,
        IEEE80211_REASON_MIC_FAILURE            = 14,

        IEEE80211_STATUS_SUCCESS                = 0,
        IEEE80211_STATUS_UNSPECIFIED            = 1,
        IEEE80211_STATUS_CAPINFO                = 10,
        IEEE80211_STATUS_NOT_ASSOCED            = 11,
        IEEE80211_STATUS_OTHER                  = 12,
        IEEE80211_STATUS_ALG                    = 13,
        IEEE80211_STATUS_SEQUENCE               = 14,
        IEEE80211_STATUS_CHALLENGE              = 15,
        IEEE80211_STATUS_TIMEOUT                = 16,
        IEEE80211_STATUS_TOOMANY                = 17,
        IEEE80211_STATUS_BASIC_RATE             = 18,
        IEEE80211_STATUS_SP_REQUIRED            = 19,
        IEEE80211_STATUS_PBCC_REQUIRED          = 20,
        IEEE80211_STATUS_CA_REQUIRED            = 21,
        IEEE80211_STATUS_TOO_MANY_STATIONS      = 22,
        IEEE80211_STATUS_RATES                  = 23,
        IEEE80211_STATUS_SHORTSLOT_REQUIRED     = 25,
        IEEE80211_STATUS_DSSSOFDM_REQUIRED      = 26,
        IEEE80211_STATUS_REFUSED                = 37,
        IEEE80211_STATUS_INVALID_PARAM          = 38,
};

/* Transmit Burst Limit */
enum {
        IEEE80211_HTINFO_TXBURST_UNLIMITED, /* Transmit Burst is unlimited */
        IEEE80211_HTINFO_TXBURST_LIMITED, /* Transmit Burst is limited */
};

/* OBSS Non-HT STAs present */
enum {
        IEEE80211_HTINFO_OBBSS_NONHT_NOT_PRESENT, /* OBSS Non-HT STAs not present */
        IEEE80211_HTINFO_OBBSS_NONHT_PRESENT, /* OBSS Non-HT STAs present */
};

/* categories */
#define IEEE80211_ACTION_CAT_QOS                0       /* qos */
#define IEEE80211_ACTION_CAT_BA                 3       /* BA */
#define IEEE80211_ACTION_CAT_HT                 7       /* HT per IEEE802.11n-D1.06 */

#define WME_NUM_AC              		4       /* 4 AC categories */
#define WME_PARAM_ACI           0x60    /* Mask for ACI field */
#define WME_PARAM_ACI_S         5       /* Shift for ACI field */
#define WME_PARAM_ACM           0x10    /* Mask for ACM bit */
#define WME_PARAM_ACM_S         4       /* Shift for ACM bit */
#define WME_PARAM_AIFSN         0x0f    /* Mask for aifsn field */
#define WME_PARAM_AIFSN_S       0       /* Shift for aifsn field */
#define WME_PARAM_LOGCWMIN      0x0f    /* Mask for CwMin field (in log) */
#define WME_PARAM_LOGCWMIN_S    0       /* Shift for CwMin field */
#define WME_PARAM_LOGCWMAX      0xf0    /* Mask for CwMax field (in log) */
#define WME_PARAM_LOGCWMAX_S    4       /* Shift for CwMax field */

#define IEEE80211_NWID_LEN                      32

/* Import from Atheros driver src */
/* WME stream classes */
#define WME_AC_BE       0               /* best effort */
#define WME_AC_BK       1               /* background */
#define WME_AC_VI       2               /* video */
#define WME_AC_VO       3               /* voice */

#define WME_OUI                 0xf25000
#define WME_OUI_TYPE            0x02
#define WME_INFO_OUI_SUBTYPE    0x00
#define WME_PARAM_OUI_SUBTYPE   0x01
#define WME_VERSION             1

/*
 * The 802.11 spec says at most 2007 stations may be
 * associated at once.  For most AP's this is way more
 * than is feasible so we use a default of 128.  This
 * number may be overridden by the driver and/or by
 * user configuration.
 */
//ernest 2007-09-03
#define IEEE80211_AID_MAX               2007
#define IEEE80211_AID_DEF               128

#define IEEE80211_AID(b)        ((b) &~ 0xc000)

#define WPA_ASE_NONE            0x00
#define WPA_ASE_8021X_UNSPEC    0x01
#define WPA_ASE_8021X_PSK       0x02
#define IEEE80211_CAPINFO_ESS                   0x0001
#define IEEE80211_CAPINFO_IBSS                  0x0002
#define IEEE80211_CAPINFO_CF_POLLABLE           0x0004
#define IEEE80211_CAPINFO_CF_POLLREQ            0x0008
#define IEEE80211_CAPINFO_PRIVACY               0x0010
#define IEEE80211_CAPINFO_SHORT_PREAMBLE        0x0020
#define IEEE80211_CAPINFO_PBCC                  0x0040
#define IEEE80211_CAPINFO_CHNL_AGILITY          0x0080
/* bits 8-9 are reserved (8 now for specturm management) */
#define IEEE80211_CAPINFO_SPECTRUM_MGMT         0x0100
#define IEEE80211_CAPINFO_SHORT_SLOTTIME        0x0400
#define IEEE80211_CAPINFO_RSN                   0x0800 /* bit 12 is reserved */
#define IEEE80211_CAPINFO_DSSSOFDM              0x2000 /* bits 14-15 are reserved */
#define IEEE80211_SEQ_FRAG_MASK                 0x000f
#define IEEE80211_SEQ_FRAG_SHIFT                0
#define IEEE80211_SEQ_SEQ_MASK                  0xfff0
#define IEEE80211_SEQ_SEQ_SHIFT                 4

#define IEEE80211_SEQ_LEQ(a,b)  ((int)((a)-(b)) <= 0)
/* Atheros capabilities */
#define IEEE80211_ATHC_TURBOP   0x0001          /* Turbo Prime */
#define IEEE80211_ATHC_COMP     0x0002          /* Compression */
#define IEEE80211_ATHC_FF       0x0004          /* Fast Frames */
#define IEEE80211_ATHC_XR       0x0008          /* Xtended Range support */
#define IEEE80211_ATHC_AR       0x0010          /* Advanced Radar support */
#define IEEE80211_ATHC_BURST    0x0020          /* Bursting - not negotiated */
#define IEEE80211_ATHC_WME      0x0040          /* CWMin tuning */
#define IEEE80211_ATHC_BOOST    0x0080          /* Boost */
/* Atheros extended capabilities */
#define IEEE80211_ATH_EXTCAP_OWL        0x00000001      /* OWL device */

#define IEEE80211_QOS_TXOP                      0x00ff

#define IEEE80211_QOS_AMSDU                     0x80
#define IEEE80211_QOS_AMSDU_S                   7
#define IEEE80211_QOS_ACKPOLICY                 0x60
#define IEEE80211_QOS_ACKPOLICY_S               5
#define IEEE80211_QOS_EOSP                      0x10
#define IEEE80211_QOS_EOSP_S                    4
#define IEEE80211_QOS_TID                       0x0f

/* End import from Atheros driver src */

/* SWEEP TABLE ENTRIES NUMBER */
#define MAX_SWEEP_TAB_ENTRIES		  42
#define MAX_SWEEP_TAB_ENTRIES_PER_PACKET  7
/* MAX_RATES_LENGTH needs to be 12.  The spec says 8, and many APs
 * only use 8, and then use extended rates for the remaining supported
 * rates.  Other APs, however, stick all of their supported rates on the
 * main rates information element... */
#define MAX_RATES_LENGTH                  ((u8)12)
#define MAX_RATES_EX_LENGTH               ((u8)16)
#define MAX_NETWORK_COUNT                  128

#define CRC_LENGTH                 4U

#define MAX_WPA_IE_LEN 64

#define NETWORK_EMPTY_ESSID    (1<<0)
#define NETWORK_HAS_OFDM       (1<<1)
#define NETWORK_HAS_CCK        (1<<2)

/*
 * 802.11i defines an extended IV for use with non-WEP ciphers.
 * When the EXTIV bit is set in the key id byte an additional
 * 4 bytes immediately follow the IV for TKIP.  For CCMP the
 * EXTIV bit is likewise set but the 8 bytes represent the
 * CCMP header rather than IV+extended-IV.
 */
#define IEEE80211_WEP_EXTIV             0x20
#define IEEE80211_WEP_EXTIVLEN          4       /* extended IV length */
#define IEEE80211_WEP_MICLEN            8       /* trailing MIC */

/*
 * Maximum acceptable MTU is:
 *      IEEE80211_MAX_LEN - WEP overhead - CRC -
 *              QoS overhead - RSN/WPA overhead
 * Min is arbitrarily chosen > IEEE80211_MIN_LEN.  The default
 * mtu is Ethernet-compatible; it's set by ether_ifattach.
 */
#define IEEE80211_MTU_MAX               2290
#define IEEE80211_MTU_MIN               32

#define IEEE80211_MAX_LEN               (2300 + IEEE80211_CRC_LEN + \
    (IEEE80211_WEP_IVLEN + IEEE80211_WEP_KIDLEN + IEEE80211_WEP_CRCLEN))
#define IEEE80211_ACK_LEN \
        (sizeof(struct ieee80211_frame_ack) + IEEE80211_CRC_LEN)
#define IEEE80211_MIN_LEN \
        (sizeof(struct ieee80211_frame_min) + IEEE80211_CRC_LEN)

/* An 802.11 data frame can be one of three types:
 * 1. An unaggregated frame: The maximum length of an unaggregated data frame is 2324 bytes + headers.
 * 2. A data frame that is part of an AMPDU: The maximum length of an AMPDU may be upto 65535 bytes, but data frame is limited to 2324 bytes + header.
 * 3. An AMSDU: The maximum length of an AMSDU is eihther 3839 or 7095 bytes.
 * The maximum frame length supported by hardware is 4095 bytes.
 * A length of 3839 bytes is chosen here to support unaggregated data frames, any size AMPDUs and 3839 byte AMSDUs.
 * */
#define IEEE80211N_MAX_FRAMELEN  3839
#define IEEE80211N_MAX_LEN (IEEE80211N_MAX_FRAMELEN + IEEE80211_CRC_LEN + \
    (IEEE80211_WEP_IVLEN + IEEE80211_WEP_KIDLEN + IEEE80211_WEP_CRCLEN))

/*
 * RTS frame length parameters.  The default is specified in
 * the 802.11 spec.  The max may be wrong for jumbo frames.
 */
#define IEEE80211_RTS_DEFAULT           512
#define IEEE80211_RTS_MIN               1
#define IEEE80211_RTS_MAX               2346

/*
 * Regulatory extention identifier for country IE.
 */
#define IEEE80211_REG_EXT_ID            201

/* QoS structure */
#define NETWORK_HAS_QOS_PARAMETERS      (1<<3)
#define NETWORK_HAS_QOS_INFORMATION     (1<<4)
#define NETWORK_HAS_QOS_MASK            (NETWORK_HAS_QOS_PARAMETERS | \
					 NETWORK_HAS_QOS_INFORMATION)

/* 802.11h */
#define NETWORK_HAS_POWER_CONSTRAINT    (1<<5)
#define NETWORK_HAS_CSA                 (1<<6)
#define NETWORK_HAS_QUIET               (1<<7)
#define NETWORK_HAS_IBSS_DFS            (1<<8)
#define NETWORK_HAS_TPC_REPORT          (1<<9)

#define NETWORK_HAS_ERP_VALUE           (1<<10)

#define QOS_QUEUE_NUM                   4
#define QOS_OUI_LEN                     3
#define QOS_OUI_TYPE                    2
#define QOS_ELEMENT_ID                  221
#define QOS_OUI_INFO_SUB_TYPE           0
#define QOS_OUI_PARAM_SUB_TYPE          1
#define QOS_VERSION_1                   1
#define QOS_AIFSN_MIN_VALUE             2

/* Import from Atheros driver src */
#define IEEE80211_WEP_KEYLEN            5       /* 40bit */
#define IEEE80211_WEP_IVLEN             3       /* 24bit */
#define IEEE80211_WEP_KIDLEN            1       /* 1 octet */
#define IEEE80211_WEP_CRCLEN            4       /* CRC-32 */
#define IEEE80211_WEP_NKID              4       /* number of key ids */

#ifdef CONFIG_CPU_BIG_ENDIAN
#define __BYTE_ORDER __BIG_ENDIAN
#elif defined(CONFIG_CPU_LITTLE_ENDIAN
#define __BYTE_ORDER __LITTLE_ENDIAN
#else
#error CPU Endiannes undefined!!!
#endif

/* operating flags */
#define IEEE80211_HTINFO_OPMODE_PURE                0x00 /* no protection */
#define IEEE80211_HTINFO_OPMODE_MIXED_PROT_OPT  0x01 /* prot optional (legacy device maybe present) */
#define IEEE80211_HTINFO_OPMODE_MIXED_PROT_40   0x02 /* prot required (20 MHz) */
#define IEEE80211_HTINFO_OPMODE_MIXED_PROT_ALL  0x03 /* prot required (legacy devices present) */
#define IEEE80211_HTINFO_OPMODE_MASK            0x03 /* For protection 0x00-0x03 */

/* BA actions */
#define IEEE80211_ACTION_BA_ADDBA_REQUEST       0   /* ADDBA request */
#define IEEE80211_ACTION_BA_ADDBA_RESPONSE      1   /* ADDBA response */
#define IEEE80211_ACTION_BA_DELBA               2   /* DELBA */

#define  IEEE80211_BA_POLICY_DELAYED      0
#define  IEEE80211_BA_POLICY_IMMEDIATE    1

/*
 * BAR frame format
 */
#define IEEE80211_BAR_CTL_TID_M     0xF000      /* tid mask             */
#define IEEE80211_BAR_CTL_TID_S         12      /* tid shift            */
#define IEEE80211_BAR_CTL_NOACK     0x0001      /* no-ack policy        */
#define IEEE80211_BAR_CTL_COMBA     0x0004      /* compressed block-ack */

struct ieee80211_tim_ie {
        u_int8_t        tim_ie;                 /* IEEE80211_ELEMID_TIM */
        u_int8_t        tim_len;
        u_int8_t        tim_count;              /* DTIM count */
        u_int8_t        tim_period;             /* DTIM period */
        u_int8_t        tim_bitctl;             /* bitmap control */
        u_int8_t        tim_bitmap[1];          /* variable-length bitmap */
} __attribute__((packed));

/*
 * Country information element.
 */
/*
 *  * Country/Region Codes from MS WINNLS.H
 *   * Numbering from ISO 3166
 *    * XXX belongs elsewhere
 *     */
enum CountryCode {
    CTRY_ALBANIA              = 8,       /* Albania */
    CTRY_ALGERIA              = 12,      /* Algeria */
    CTRY_ARGENTINA            = 32,      /* Argentina */
    CTRY_ARMENIA              = 51,      /* Armenia */
    CTRY_AUSTRALIA            = 36,      /* Australia */
    CTRY_AUSTRIA              = 40,      /* Austria */
    CTRY_AZERBAIJAN           = 31,      /* Azerbaijan */
    CTRY_BAHRAIN              = 48,      /* Bahrain */
    CTRY_BELARUS              = 112,     /* Belarus */
    CTRY_BELGIUM              = 56,      /* Belgium */
    CTRY_BELIZE               = 84,      /* Belize */
    CTRY_BOLIVIA              = 68,      /* Bolivia */
    CTRY_BOSNIA_HERZEGOWINA   = 70,      /* Bosnia and Herzegowina */
    CTRY_BRAZIL               = 76,      /* Brazil */
    CTRY_BRUNEI_DARUSSALAM    = 96,      /* Brunei Darussalam */
    CTRY_BULGARIA             = 100,     /* Bulgaria */
    CTRY_CANADA               = 124,     /* Canada */
    CTRY_CHILE                = 152,     /* Chile */
    CTRY_CHINA                = 156,     /* People's Republic of China */
    CTRY_COLOMBIA             = 170,     /* Colombia */
    CTRY_COSTA_RICA           = 188,     /* Costa Rica */
    CTRY_CROATIA              = 191,     /* Croatia */
    CTRY_CYPRUS               = 196,	 /* Cyprus */
    CTRY_CZECH                = 203,     /* Czech Republic */
    CTRY_DENMARK              = 208,     /* Denmark */
    CTRY_DOMINICAN_REPUBLIC   = 214,     /* Dominican Republic */
    CTRY_ECUADOR              = 218,     /* Ecuador */
    CTRY_EGYPT                = 818,     /* Egypt */
    CTRY_EL_SALVADOR          = 222,     /* El Salvador */
    CTRY_ESTONIA              = 233,     /* Estonia */
    CTRY_FAEROE_ISLANDS       = 234,     /* Faeroe Islands */
    CTRY_FINLAND              = 246,     /* Finland */
    CTRY_FRANCE               = 250,     /* France */
    CTRY_FRANCE2              = 255,     /* France2 */
    CTRY_GEORGIA              = 268,     /* Georgia */
    CTRY_GERMANY              = 276,     /* Germany */
    CTRY_GREECE               = 300,     /* Greece */
    CTRY_GUATEMALA            = 320,     /* Guatemala */
    CTRY_HONDURAS             = 340,     /* Honduras */
    CTRY_HONG_KONG            = 344,     /* Hong Kong S.A.R., P.R.C. */
    CTRY_HUNGARY              = 348,     /* Hungary */
    CTRY_ICELAND              = 352,     /* Iceland */
    CTRY_INDIA                = 356,     /* India */
    CTRY_INDONESIA            = 360,     /* Indonesia */
    CTRY_IRAN                 = 364,     /* Iran */
    CTRY_IRAQ                 = 368,     /* Iraq */
    CTRY_IRELAND              = 372,     /* Ireland */
    CTRY_ISRAEL               = 376,     /* Israel */
    CTRY_ISRAEL2              = 377,     /* Israel */
    CTRY_ITALY                = 380,     /* Italy */
    CTRY_JAMAICA              = 388,     /* Jamaica */
    CTRY_JAPAN                = 392,     /* Japan */
    CTRY_JAPAN1               = 393,     /* Japan (JP1) */
    CTRY_JAPAN2               = 394,     /* Japan (JP0) */
    CTRY_JAPAN3               = 395,     /* Japan (JP1-1) */
    CTRY_JAPAN4               = 396,     /* Japan (JE1) */
    CTRY_JAPAN5               = 397,     /* Japan (JE2) */
    CTRY_JAPAN6               = 399,     /* Japan (JP6) */
    CTRY_JAPAN7               = 4007,    /* Japan */
    CTRY_JAPAN8               = 4008,    /* Japan */
    CTRY_JAPAN9               = 4009,    /* Japan */
    CTRY_JAPAN10              = 4010,    /* Japan */
    CTRY_JAPAN11              = 4011,    /* Japan */
    CTRY_JAPAN12              = 4012,    /* Japan */
    CTRY_JAPAN13              = 4013,    /* Japan */
    CTRY_JAPAN14              = 4014,    /* Japan */
    CTRY_JAPAN15              = 4015,    /* Japan */
    CTRY_JAPAN16              = 4016,    /* Japan */
    CTRY_JAPAN17              = 4017,    /* Japan */
    CTRY_JAPAN18              = 4018,    /* Japan */
    CTRY_JAPAN19              = 4019,    /* Japan */
    CTRY_JAPAN20              = 4020,    /* Japan */
    CTRY_JAPAN21              = 4021,    /* Japan */
    CTRY_JAPAN22              = 4022,    /* Japan */
    CTRY_JAPAN23              = 4023,    /* Japan */
    CTRY_JAPAN24              = 4024,    /* Japan */
    CTRY_JAPAN25              = 4025,    /* Japan (J25) */
    CTRY_JAPAN26              = 4026,    /* Japan (J26) */
    CTRY_JAPAN27              = 4027,    /* Japan (J27) */

    CTRY_JAPAN28              = 4028,    /* Japan (J28) */
    CTRY_JAPAN29              = 4029,    /* Japan (J29) */
    CTRY_JAPAN30              = 4030,    /* Japan (J30) */

    CTRY_JAPAN31              = 4031,    /* Japan (J31) */
    CTRY_JAPAN32              = 4032,    /* Japan (J32) */
    CTRY_JAPAN33              = 4033,    /* Japan (J33) */

    CTRY_JAPAN34              = 4034,    /* Japan (J34) */
    CTRY_JAPAN35              = 4035,    /* Japan (J35) */
    CTRY_JAPAN36              = 4036,    /* Japan (J36) */

    CTRY_JAPAN37              = 4037,    /* Japan (J37) */
    CTRY_JAPAN38              = 4038,    /* Japan (J38) */
    CTRY_JAPAN39              = 4039,    /* Japan (J39) */

    CTRY_JAPAN40              = 4040,    /* Japan (J40) */
    CTRY_JAPAN41              = 4041,    /* Japan (J41) */
    CTRY_JAPAN42              = 4042,    /* Japan (J42) */
    CTRY_JAPAN43              = 4043,    /* Japan (J43) */
    CTRY_JAPAN44              = 4044,    /* Japan (J44) */
    CTRY_JAPAN45              = 4045,    /* Japan (J45) */
    CTRY_JAPAN46              = 4046,    /* Japan (J46) */
    CTRY_JAPAN47              = 4047,    /* Japan (J47) */
    CTRY_JAPAN48              = 4048,    /* Japan (J48) */
    CTRY_JAPAN49              = 4049,    /* Japan (J49) */

    CTRY_JAPAN50              = 4050,    /* Japan (J50) */
    CTRY_JAPAN51              = 4051,    /* Japan (J51) */
    CTRY_JAPAN52              = 4052,    /* Japan (J52) */
    CTRY_JAPAN53              = 4053,    /* Japan (J53) */
    CTRY_JAPAN54              = 4054,    /* Japan (J54) */

    CTRY_AUSTRALIA2           = 5000,    /* Australia */
    CTRY_CANADA2              = 5001,    /* Canada */

    CTRY_JORDAN               = 400,     /* Jordan */
    CTRY_KAZAKHSTAN           = 398,     /* Kazakhstan */
    CTRY_KENYA                = 404,     /* Kenya */
    CTRY_KOREA_NORTH          = 408,     /* North Korea */
    CTRY_KOREA_ROC            = 410,     /* South Korea */
    CTRY_KOREA_ROC2           = 411,     /* South Korea */
    CTRY_KOREA_ROC3           = 412,     /* South Korea */
    CTRY_KUWAIT               = 414,     /* Kuwait */
    CTRY_LATVIA               = 428,     /* Latvia */
    CTRY_LEBANON              = 422,     /* Lebanon */
    CTRY_LIBYA                = 434,     /* Libya */
    CTRY_LIECHTENSTEIN        = 438,     /* Liechtenstein */
    CTRY_LITHUANIA            = 440,     /* Lithuania */
    CTRY_LUXEMBOURG           = 442,     /* Luxembourg */
    CTRY_MACAU                = 446,     /* Macau */
    CTRY_MACEDONIA            = 807,     /* the Former Yugoslav Republic of Macedonia */
    CTRY_MALAYSIA             = 458,     /* Malaysia */
    CTRY_MALTA                = 470,     /* Malta */
    CTRY_MEXICO               = 484,     /* Mexico */
    CTRY_MONACO               = 492,     /* Principality of Monaco */
    CTRY_MOROCCO              = 504,     /* Morocco */
    CTRY_NETHERLANDS          = 528,     /* Netherlands */
    CTRY_NETHERLANDS_ANT      = 530,     /* Netherlands-Antellis */
    CTRY_NEW_ZEALAND          = 554,     /* New Zealand */
    CTRY_NICARAGUA            = 558,     /* Nicaragua */
    CTRY_NORWAY               = 578,     /* Norway */
    CTRY_OMAN                 = 512,     /* Oman */
    CTRY_PAKISTAN             = 586,     /* Islamic Republic of Pakistan */
    CTRY_PANAMA               = 591,     /* Panama */
    CTRY_PARAGUAY             = 600,     /* Paraguay */
    CTRY_PERU                 = 604,     /* Peru */
    CTRY_PHILIPPINES          = 608,     /* Republic of the Philippines */
    CTRY_POLAND               = 616,     /* Poland */
    CTRY_PORTUGAL             = 620,     /* Portugal */
    CTRY_PUERTO_RICO          = 630,     /* Puerto Rico */
    CTRY_QATAR                = 634,     /* Qatar */
    CTRY_ROMANIA              = 642,     /* Romania */
    CTRY_RUSSIA               = 643,     /* Russia */
    CTRY_SAUDI_ARABIA         = 682,     /* Saudi Arabia */
    CTRY_SERBIA_MONT          = 891,     /* Serbia and Montenegro */
    CTRY_SINGAPORE            = 702,     /* Singapore */
    CTRY_SLOVAKIA             = 703,     /* Slovak Republic */
    CTRY_SLOVENIA             = 705,     /* Slovenia */
    CTRY_SOUTH_AFRICA         = 710,     /* South Africa */
    CTRY_SPAIN                = 724,     /* Spain */
    CTRY_SRI_LANKA            = 144,     /* Sri Lanka */
    CTRY_SWEDEN               = 752,     /* Sweden */
    CTRY_SWITZERLAND          = 756,     /* Switzerland */
    CTRY_SYRIA                = 760,     /* Syria */
    CTRY_TAIWAN               = 158,     /* Taiwan */
    CTRY_THAILAND             = 764,     /* Thailand */
    CTRY_TRINIDAD_Y_TOBAGO    = 780,     /* Trinidad y Tobago */
    CTRY_TUNISIA              = 788,     /* Tunisia */
    CTRY_TURKEY               = 792,     /* Turkey */
    CTRY_UAE                  = 784,     /* U.A.E. */
    CTRY_UKRAINE              = 804,     /* Ukraine */
    CTRY_UNITED_KINGDOM       = 826,     /* United Kingdom */
    CTRY_UNITED_STATES        = 840,     /* United States */
    CTRY_UNITED_STATES_FCC49  = 842,     /* United States (Public Safety)*/
    CTRY_URUGUAY              = 858,     /* Uruguay */
    CTRY_UZBEKISTAN           = 860,     /* Uzbekistan */
    CTRY_VENEZUELA            = 862,     /* Venezuela */
    CTRY_VIET_NAM             = 704,     /* Viet Nam */
    CTRY_YEMEN                = 887,     /* Yemen */
    CTRY_ZIMBABWE             = 716      /* Zimbabwe */
};

#define IEEE80211_COUNTRY_MAX_TRIPLETS (83)
struct ieee80211_ie_country {
        u_int8_t        country_id;
        u_int8_t        country_len;
        u_int8_t        country_str[3];
        u_int8_t        country_triplet[IEEE80211_COUNTRY_MAX_TRIPLETS*3];
} __attribute__ ((packed));

struct ieee80211_ba_seqctrl {
#if __BYTE_ORDER == __BIG_ENDIAN
        u_int16_t       startseqnum     : 12,    /* B4-15  starting sequence number */
                        fragnum         : 4;     /* B0-3  fragment number */
#else
        u_int16_t       fragnum         : 4,     /* B0-3  fragment number */
                        startseqnum     : 12;    /* B4-15  starting sequence number */
#endif
} __attribute__ ((packed));

struct ieee80211_ba_parameterset {
#if __BYTE_ORDER == __BIG_ENDIAN
        u_int16_t       buffersize      : 10,   /* B6-15  buffer size */
                        tid             : 4,    /* B2-5   TID */
                        bapolicy        : 1,    /* B1   block ack policy */
                        reserved0       : 1;    /* B0   reserved */
#else
        u_int16_t       reserved0       : 1,    /* B0   reserved */
                        bapolicy        : 1,    /* B1   block ack policy */
                        tid             : 4,    /* B2-5   TID */
                        buffersize      : 10;   /* B6-15  buffer size */
#endif
} __attribute__ ((packed));

struct ieee80211_delba_parameterset {
#if __BYTE_ORDER == __BIG_ENDIAN
        u_int16_t       tid             : 4,     /* B12-15  tid */
                        initiator       : 1,     /* B11     initiator */
                        reserved0       : 11;    /* B0-10   reserved */
#else
        u_int16_t       reserved0       : 11,    /* B0-10   reserved */
                        initiator       : 1,     /* B11     initiator */
                        tid             : 4;     /* B12-15  tid */
#endif
} __attribute__ ((packed));

struct ieee80211_ie_htcap_cmn {
        u_int16_t       hc_cap;                 /* HT capabilities */
#if __BYTE_ORDER == __BIG_ENDIAN
        u_int8_t        hc_reserved     : 3,    /* B5-7 reserved */
                hc_mpdudensity  : 3,    /* B2-4 MPDU density (aka Minimum MPDU Start Spacing) */
                hc_maxampdu     : 2;    /* B0-1 maximum rx A-MPDU factor */
#else
        u_int8_t        hc_maxampdu     : 2,    /* B0-1 maximum rx A-MPDU factor */
                hc_mpdudensity  : 3,    /* B2-4 MPDU density (aka Minimum MPDU Start Spacing) */
                hc_reserved     : 3;    /* B5-7 reserved */
#endif
        u_int8_t        hc_mcsset[16];          /* supported MCS set */
        u_int16_t       hc_extcap;              /* extended HT capabilities */
        u_int32_t       hc_txbf;                /* txbf capabilities */
        u_int8_t        hc_antenna;             /* antenna capabilities */
} __attribute__((packed));

/*
 * 802.11i/WPA information element (maximally sized).
 */
struct ieee80211_ie_wpa {
        u_int8_t        wpa_id;         /* IEEE80211_ELEMID_VENDOR */
        u_int8_t        wpa_len;        /* length in bytes */
        u_int8_t        wpa_oui[3];     /* 0x00, 0x50, 0xf2 */
        u_int8_t        wpa_type;       /* OUI type */
        u_int16_t       wpa_version;    /* spec revision */
        u_int32_t       wpa_mcipher[1]; /* multicast/group key cipher */
        u_int16_t       wpa_uciphercnt; /* # pairwise key ciphers */
        u_int32_t       wpa_uciphers[8];/* ciphers */
        u_int16_t       wpa_authselcnt; /* authentication selector cnt*/
        u_int32_t       wpa_authsels[8];/* selectors */
        u_int16_t       wpa_caps;       /* 802.11i capabilities */
        u_int16_t       wpa_pmkidcnt;   /* 802.11i pmkid count */
        u_int16_t       wpa_pmkids[8];  /* 802.11i pmkids */
} __attribute__((packed));

#define IEEE80211_HTCAP_MAXRXAMPDU_FACTOR   13

/* EPR information element flags */
#define IEEE80211_ERP_NON_ERP_PRESENT   0x01
#define IEEE80211_ERP_USE_PROTECTION    0x02
#define IEEE80211_ERP_LONG_PREAMBLE     0x04

/* HT actions */
#define IEEE80211_ACTION_HT_TXCHWIDTH           0       /* recommended transmission channel width */
#define IEEE80211_ACTION_HT_SMPOWERSAVE         1       /* Spatial Multiplexing (SM) Power Save */

/* HT capability flags */
#define IEEE80211_HTCAP_C_ADVCODING                   0x0001
#define IEEE80211_HTCAP_C_CHWIDTH40               0x0002
#define IEEE80211_HTCAP_C_SMPOWERSAVE_STATIC      0x0000 /* Capable of SM Power Save (Static) */
#define IEEE80211_HTCAP_C_SMPOWERSAVE_DYNAMIC     0x0004 /* Capable of SM Power Save (Dynamic) */
#define IEEE80211_HTCAP_C_SM_RESERVED             0x0008 /* Reserved */
#define IEEE80211_HTCAP_C_SM_ENABLED              0x000c /* SM enabled, no SM Power Save */
#define IEEE80211_HTCAP_C_GREENFIELD          0x0010
#define IEEE80211_HTCAP_C_SHORTGI20                   0x0020
#define IEEE80211_HTCAP_C_SHORTGI40               0x0040
#define IEEE80211_HTCAP_C_TXSTBC                  0x0080
#define IEEE80211_HTCAP_C_RXSTBC                  0x0100  /* 2 bits */
#define IEEE80211_HTCAP_C_DELAYEDBLKACK           0x0400
#define IEEE80211_HTCAP_C_MAXAMSDUSIZE            0x0800  /* 1 = 8K, 0 = 3839B */
#define IEEE80211_HTCAP_C_DSSSCCK40               0x1000
#define IEEE80211_HTCAP_C_PSMP                    0x2000
#define IEEE80211_HTCAP_C_INTOLERANT40            0x4000
#define IEEE80211_HTCAP_C_LSIGTXOPPROT            0x8000

#define IEEE80211_HTCAP_C_SM_MASK                 0x000c /* Spatial Multiplexing (SM) capabitlity bitmask */

/* B0-1 maximum rx A-MPDU factor 2^(13+Max Rx A-MPDU Factor) */
enum {
        IEEE80211_HTCAP_MAXRXAMPDU_8192,        /* 2 ^ 13 */
        IEEE80211_HTCAP_MAXRXAMPDU_16384,   /* 2 ^ 14 */
        IEEE80211_HTCAP_MAXRXAMPDU_32768,   /* 2 ^ 15 */
        IEEE80211_HTCAP_MAXRXAMPDU_65536,   /* 2 ^ 16 */
};

/* B2-4 MPDU density (usec) */
enum {
        IEEE80211_HTCAP_MPDUDENSITY_NA,         /* No time restriction */
        IEEE80211_HTCAP_MPDUDENSITY_0_25,       /* 1/4 usec */
        IEEE80211_HTCAP_MPDUDENSITY_0_5,    /* 1/2 usec */
        IEEE80211_HTCAP_MPDUDENSITY_1,          /* 1 usec */
        IEEE80211_HTCAP_MPDUDENSITY_2,          /* 2 usec */
        IEEE80211_HTCAP_MPDUDENSITY_4,          /* 4 usec */
        IEEE80211_HTCAP_MPDUDENSITY_8,          /* 8 usec */
        IEEE80211_HTCAP_MPDUDENSITY_16,         /* 16 usec */
};

/* HT extended capability flags */
#define IEEE80211_HTCAP_EXTC_PCO                        0x0001
#define IEEE80211_HTCAP_EXTC_TRANS_TIME_RSVD    0x0000
#define IEEE80211_HTCAP_EXTC_TRANS_TIME_400     0x0002 /* 20-40 switch time */
#define IEEE80211_HTCAP_EXTC_TRANS_TIME_1500    0x0004 /* in us             */
#define IEEE80211_HTCAP_EXTC_TRANS_TIME_5000    0x0006
#define IEEE80211_HTCAP_EXTC_RSVD_1             0x00f8
#define IEEE80211_HTCAP_EXTC_MCS_FEEDBACK_NONE  0x0000
#define IEEE80211_HTCAP_EXTC_MCS_FEEDBACK_RSVD  0x0100
#define IEEE80211_HTCAP_EXTC_MCS_FEEDBACK_UNSOL 0x0200
#define IEEE80211_HTCAP_EXTC_MCS_FEEDBACK_FULL  0x0300
#define IEEE80211_HTCAP_EXTC_RSVD_2             0xfc00

struct ieee80211_ie_htinfo_cmn {
        u_int8_t        hi_ctrlchannel;         /* control channel */
#if __BYTE_ORDER == __BIG_ENDIAN
        u_int8_t        hi_serviceinterval : 3,    /* B5-7 svc interval granularity */
                hi_ctrlaccess      : 1,    /* B4   controlled access only */
                hi_rifsmode        : 1,    /* B3   rifs mode */
                hi_txchwidth       : 1,    /* B2   recommended xmiss width set */
                hi_extchoff        : 2;    /* B0-1 extension channel offset */
#else
        u_int8_t    hi_extchoff        : 2,    /* B0-1 extension channel offset */
                hi_txchwidth       : 1,    /* B2   recommended xmiss width set */
                hi_rifsmode        : 1,    /* B3   rifs mode */
                hi_ctrlaccess      : 1,    /* B4   controlled access only */
                hi_serviceinterval : 3;    /* B5-7 svc interval granularity */
#endif
#if __BYTE_ORDER == __BIG_ENDIAN
    u_int8_t   hi_reserved0       : 3,   /* B5-7 Reserved */
                hi_obssnonhtpresent: 1,    /* B4    OBSS Non-HT STAs Present */
                hi_txburstlimit    : 1,    /* B3    Transmit Burst Limit */
                hi_nongfpresent    : 1,    /* B2    Non-greenfield STAs present */
                hi_opmode          : 2;    /* B0-1  Operating Mode */
#else
    u_int8_t   hi_opmode          : 2,    /* B0-1  Operating Mode */
                hi_nongfpresent    : 1,    /* B2    Non-greenfield STAs present */
                hi_txburstlimit    : 1,    /* B3    Transmit Burst Limit */
                hi_obssnonhtpresent: 1,    /* B4    OBSS Non-HT STAs Present */
                hi_reserved0       : 3;   /* B5-7 Reserved */
#endif
    u_int8_t    hi_reserved1;
    u_int16_t   hi_miscflags;

        u_int8_t        hi_basicmcsset[16];     /* basic MCS set */
} __attribute__((packed));

/*
 * 802.11n HT Information IE
 */
struct ieee80211_ie_htinfo {
        u_int8_t                            hi_id;                      /* element ID */
        u_int8_t                            hi_len;                     /* length in bytes */
        struct ieee80211_ie_htinfo_cmn  hi_ie;
} __attribute__((packed));

/*
 * Atheros Advanced Capability information element.
 */
struct ieee80211_ie_athAdvCap {
        u_int8_t        athAdvCap_id;           /* IEEE80211_ELEMID_VENDOR */
        u_int8_t        athAdvCap_len;          /* length in bytes */
        u_int8_t        athAdvCap_oui[3];       /* 0x00, 0x03, 0x7f */
        u_int8_t        athAdvCap_type;         /* OUI type */
        u_int8_t        athAdvCap_subtype;      /* OUI subtype */
        u_int8_t        athAdvCap_version;      /* spec revision */
        u_int8_t        athAdvCap_capability;   /* Capability info */
        u_int16_t       athAdvCap_defKeyIndex;
} __attribute__((packed));

/*
 *  * Atheros Extended Capability information element.
 *   */
struct ieee80211_ie_ath_extcap {
        u_int8_t        ath_extcap_id;  /* IEEE80211_ELEMID_VENDOR */
        u_int8_t        ath_extcap_len; /* length in bytes */
        u_int8_t        ath_extcap_oui[3];      /* 0x00, 0x03, 0x7f */
        u_int8_t        ath_extcap_type;        /* OUI type */
        u_int8_t        ath_extcap_subtype;     /* OUI subtype */
        u_int8_t        ath_extcap_version;     /* spec revision */
        u_int32_t       ath_extcap_data;        /* Data */
} __attribute__((packed));

/*
 *  * Atheros XR information element.
 *   */
struct ieee80211_xr_param {
        u_int8_t        param_id;
        u_int8_t        param_len;
        u_int8_t        param_oui[3];
        u_int8_t        param_oui_type;
        u_int8_t        param_oui_sybtype;
        u_int8_t        param_version;
        u_int8_t        param_Info;
        u_int8_t        param_base_bssid[IEEE80211_ADDR_LEN];
        u_int8_t        param_xr_bssid[IEEE80211_ADDR_LEN];
        u_int16_t       param_xr_beacon_interval;
        u_int8_t        param_base_ath_capability;
        u_int8_t        param_xr_ath_capability;
} __attribute__((packed));

/*
 * WME Parameter Element
 */
struct ieee80211_wme_acparams {
        u_int8_t        acp_aci_aifsn;
        u_int8_t        acp_logcwminmax;
        u_int16_t       acp_txop;
} __attribute__((packed));

struct ieee80211_wme_param {
        u_int8_t        param_id;
        u_int8_t        param_len;
        u_int8_t        param_oui[3];
        u_int8_t        param_oui_type;
        u_int8_t        param_oui_sybtype;
        u_int8_t        param_version;
        u_int8_t        param_qosInfo;
        u_int8_t        param_reserved;
        struct ieee80211_wme_acparams   params_acParams[WME_NUM_AC];
} __attribute__((packed));

/*
 * Temporary vendor private HT Capability IE
 */
struct vendor_ie_htcap {
        u_int8_t                            hc_id;                      /* element ID */
        u_int8_t                            hc_len;                     /* length in bytes */
        u_int8_t                        hc_oui[3];
        u_int8_t                        hc_ouitype;
        struct ieee80211_ie_htcap_cmn   hc_ie;
} __attribute__((packed));

/*
 * Temporary vendor private HT Information IE
 */
struct vendor_ie_htinfo {
        u_int8_t                            hi_id;                      /* element ID */
        u_int8_t                        hi_len;                 /* length in bytes */
    u_int8_t                        hi_oui[3];
    u_int8_t                        hi_ouitype;
    struct ieee80211_ie_htinfo_cmn  hi_ie;
} __attribute__((packed));

/*
 * 802.11n HT Capability IE
 */
struct ieee80211_ie_htcap {
        u_int8_t                            hc_id;                      /* element ID */
        u_int8_t                            hc_len;                     /* length in bytes */
        struct ieee80211_ie_htcap_cmn   hc_ie;
} __attribute__((packed));

/* BA - ADDBA request */
struct ieee80211_action_ba_addbarequest {
        struct ieee80211_action                 rq_header;
        u_int8_t                                rq_dialogtoken;
        struct ieee80211_ba_parameterset        rq_baparamset;
        u_int16_t                               rq_batimeout;   /* in TUs */
        struct ieee80211_ba_seqctrl             rq_basequencectrl;
} __attribute__((packed));

/* BA - ADDBA response */
struct ieee80211_action_ba_addbaresponse {
        struct ieee80211_action                 rs_header;
        u_int8_t                                rs_dialogtoken;
        u_int16_t                               rs_statuscode;
        struct ieee80211_ba_parameterset        rs_baparamset;
        u_int16_t                               rs_batimeout;          /* in TUs */
} __attribute__((packed));

/* BA - DELBA */
struct ieee80211_action_ba_delba {
        struct ieee80211_action                 dl_header;
        struct ieee80211_delba_parameterset     dl_delbaparamset;
        u_int16_t                               dl_reasoncode;
} __attribute__((packed));

/* HT - recommended transmission channel width */
struct ieee80211_action_ht_txchwidth {
        struct ieee80211_action         at_header;
        u_int8_t                        at_chwidth;
} __attribute__((packed));

/* HT - Spatial Multiplexing (SM) Power Save */
struct ieee80211_action_ht_smpowersave {
        struct ieee80211_action         as_header;
        u_int8_t                        as_control;
} __attribute__((packed));

/*
 * WME/802.11e information element.
 */
struct ieee80211_ie_wme {
        u_int8_t        wme_id;         /* IEEE80211_ELEMID_VENDOR */
        u_int8_t        wme_len;        /* length in bytes */
        u_int8_t        wme_oui[3];     /* 0x00, 0x50, 0xf2 */
        u_int8_t        wme_type;       /* OUI type */
        u_int8_t        wme_subtype;    /* OUI subtype */
        u_int8_t        wme_version;    /* spec revision */
        u_int8_t        wme_info;       /* QoS info */
} __attribute__((packed));

/* values defined for 'as_control' field per 802.11n-D1.06 */
#define IEEE80211_A_HT_SMPOWERSAVE_DISABLED     0x00   /* SM Power Save Disabled, SM packets ok  */
#define IEEE80211_A_HT_SMPOWERSAVE_ENABLED      0x01   /* SM Power Save Enabled bit  */
#define IEEE80211_A_HT_SMPOWERSAVE_MODE         0x02   /* SM Power Save Mode bit */
#define IEEE80211_A_HT_SMPOWERSAVE_RESERVED     0xFC   /* SM Power Save Reserved bits */

/* extension channel offset (2 bit signed number) */
enum {
        IEEE80211_HTINFO_EXTOFFSET_NA    = 0,   /* 0  no extension channel is present */
        IEEE80211_HTINFO_EXTOFFSET_ABOVE = 1,   /* +1 extension channel above control channel */
        IEEE80211_HTINFO_EXTOFFSET_UNDEF = 2,   /* -2 undefined */
        IEEE80211_HTINFO_EXTOFFSET_BELOW = 3    /* -1 extension channel below control channel*/
};

/* recommended transmission width set */
enum {
        IEEE80211_HTINFO_TXWIDTH_20,
        IEEE80211_HTINFO_TXWIDTH_2040
};

/* HT capability flags */
#define IEEE80211_HTCAP_C_ADVCODING                   0x0001
#define IEEE80211_HTCAP_C_CHWIDTH40               0x0002
#define IEEE80211_HTCAP_C_SMPOWERSAVE_STATIC      0x0000 /* Capable of SM Power Save (Static) */
#define IEEE80211_HTCAP_C_SMPOWERSAVE_DYNAMIC     0x0004 /* Capable of SM Power Save (Dynamic) */
#define IEEE80211_HTCAP_C_SM_RESERVED             0x0008 /* Reserved */
#define IEEE80211_HTCAP_C_SM_ENABLED              0x000c /* SM enabled, no SM Power Save */
#define IEEE80211_HTCAP_C_GREENFIELD          0x0010
#define IEEE80211_HTCAP_C_SHORTGI20                   0x0020
#define IEEE80211_HTCAP_C_SHORTGI40               0x0040
#define IEEE80211_HTCAP_C_TXSTBC                  0x0080
#define IEEE80211_HTCAP_C_RXSTBC                  0x0100  /* 2 bits */
#define IEEE80211_HTCAP_C_DELAYEDBLKACK           0x0400
#define IEEE80211_HTCAP_C_MAXAMSDUSIZE            0x0800  /* 1 = 8K, 0 = 3839B */
#define IEEE80211_HTCAP_C_DSSSCCK40               0x1000
#define IEEE80211_HTCAP_C_PSMP                    0x2000
#define IEEE80211_HTCAP_C_INTOLERANT40            0x4000
#define IEEE80211_HTCAP_C_LSIGTXOPPROT            0x8000

#define IEEE80211_HTCAP_C_SM_MASK                 0x000c /* Spatial Multiplexing (SM) capabitlity bitmask */

/*
 * WME U-APSD qos info field defines
 */
#define WME_CAPINFO_UAPSD_EN                    0x00000080
#define WME_CAPINFO_UAPSD_VO                    0x00000001
#define WME_CAPINFO_UAPSD_VI                    0x00000002
#define WME_CAPINFO_UAPSD_BK                    0x00000004
#define WME_CAPINFO_UAPSD_BE                    0x00000008
#define WME_CAPINFO_UAPSD_ACFLAGS_SHIFT         0
#define WME_CAPINFO_UAPSD_ACFLAGS_MASK          0xF
#define WME_CAPINFO_UAPSD_MAXSP_SHIFT           5
#define WME_CAPINFO_UAPSD_MAXSP_MASK            0x3
#define WME_CAPINFO_IE_OFFSET                   8
#define WME_UAPSD_MAXSP(_qosinfo) (((_qosinfo) >> WME_CAPINFO_UAPSD_MAXSP_SHIFT) & WME_CAPINFO_UAPSD_MAXSP_MASK)
#define WME_UAPSD_AC_ENABLED(_ac, _qosinfo) ( (1<<(3 - (_ac))) &   \
                (((_qosinfo) >> WME_CAPINFO_UAPSD_ACFLAGS_SHIFT) & WME_CAPINFO_UAPSD_ACFLAGS_MASK) )

#define WME_QOSINFO_COUNT       0x0f  /* Mask for Param Set Count field */

#define IEEE80211_CHALLENGE_LEN         128

#define IEEE80211_SUPPCHAN_LEN          26

#define IEEE80211_RATE_BASIC            0x80
#define IEEE80211_RATE_VAL              0x7f

#define IEEE80211_AUTH_ALG_OPEN         0x0000
#define IEEE80211_AUTH_ALG_SHARED       0x0001
#define IEEE80211_AUTH_ALG_LEAP         0x0080

#define IEEE80211_A_HT_TXCHWIDTH_20     0
#define IEEE80211_A_HT_TXCHWIDTH_2040   1

enum {
        IEEE80211_AUTH_OPEN_REQUEST             = 1,
        IEEE80211_AUTH_OPEN_RESPONSE            = 2,
};

enum {
        IEEE80211_AUTH_SHARED_REQUEST           = 1,
        IEEE80211_AUTH_SHARED_CHALLENGE         = 2,
        IEEE80211_AUTH_SHARED_RESPONSE          = 3,
        IEEE80211_AUTH_SHARED_PASS              = 4,
};

/*
 *  * Management information element payloads.
 *   */

enum {
        IEEE80211_ELEMID_SSID           = 0,
        IEEE80211_ELEMID_RATES          = 1,
        IEEE80211_ELEMID_FHPARMS        = 2,
        IEEE80211_ELEMID_DSPARMS        = 3,
        IEEE80211_ELEMID_CFPARMS        = 4,
        IEEE80211_ELEMID_TIM            = 5,
        IEEE80211_ELEMID_IBSSPARMS      = 6,
        IEEE80211_ELEMID_COUNTRY        = 7,
        IEEE80211_ELEMID_REQINFO        = 10,
        IEEE80211_ELEMID_CHALLENGE      = 16,
        /* 17-31 reserved for challenge text extension */
        IEEE80211_ELEMID_PWRCNSTR       = 32,
        IEEE80211_ELEMID_PWRCAP         = 33,
        IEEE80211_ELEMID_TPCREQ         = 34,
        IEEE80211_ELEMID_TPCREP         = 35,
        IEEE80211_ELEMID_SUPPCHAN       = 36,
        IEEE80211_ELEMID_CHANSWITCHANN  = 37,
        IEEE80211_ELEMID_MEASREQ        = 38,
        IEEE80211_ELEMID_MEASREP        = 39,
        IEEE80211_ELEMID_QUIET          = 40,
        IEEE80211_ELEMID_IBSSDFS        = 41,
        IEEE80211_ELEMID_ERP            = 42,
        IEEE80211_ELEMID_HTCAP_ANA      = 45,
        IEEE80211_ELEMID_RSN            = 48,
        IEEE80211_ELEMID_XRATES         = 50,
        IEEE80211_ELEMID_HTCAP          = 51,
        IEEE80211_ELEMID_HTINFO         = 52,
        IEEE80211_ELEMID_EXTCHANSWITCHANN = 60, /* Fix this later as per ANA definition */
        IEEE80211_ELEMID_HTINFO_ANA     = 61,
        IEEE80211_ELEMID_TPC            = 150,
        IEEE80211_ELEMID_CCKM           = 156,
        IEEE80211_ELEMID_VENDOR         = 221,  /* vendor private */
};

/* RIFS mode */
enum {
        IEEE80211_HTINFO_RIFSMODE_PROHIBITED,   /* use of rifs prohibited */
        IEEE80211_HTINFO_RIFSMODE_ALLOWED,      /* use of rifs permitted */
};

#define ATH_OUI                 0x7f0300                /* Atheros OUI */
#define ATH_OUI_TYPE            0x01
#define ATH_OUI_SUBTYPE         0x01
#define ATH_OUI_VERSION         0x00
#define ATH_OUI_TYPE_XR         0x03
#define ATH_OUI_VER_XR          0x01
#define ATH_OUI_EXTCAP_TYPE     0x04    /* Atheros Extended Cap Type */
#define ATH_OUI_EXTCAP_SUBTYPE  0x01    /* Atheros Extended Cap Sub-type */
#define ATH_OUI_EXTCAP_VERSION  0x00    /* Atheros Extended Cap Version */

#define WPA_OUI                 0xf25000
#define WPA_OUI_TYPE            0x01
#define WPA_VERSION             1               /* current supported version */

#define WSC_OUI                 0x0050f204

#define WPA_CSE_NULL            0x00
#define WPA_CSE_WEP40           0x01
#define WPA_CSE_TKIP            0x02
#define WPA_CSE_CCMP            0x04
#define WPA_CSE_WEP104          0x05

#define WPA_ASE_NONE            0x00
#define WPA_ASE_8021X_UNSPEC    0x01
#define WPA_ASE_8021X_PSK       0x02

#define RSN_OUI                 0xac0f00
#define RSN_VERSION             1               /* current supported version */

#define RSN_CSE_NULL            0x00
#define RSN_CSE_WEP40           0x01
#define RSN_CSE_TKIP            0x02
#define RSN_CSE_WRAP            0x03
#define RSN_CSE_CCMP            0x04
#define RSN_CSE_WEP104          0x05

#define RSN_ASE_NONE            0x00
#define RSN_ASE_8021X_UNSPEC    0x01
#define RSN_ASE_8021X_PSK       0x02

#define RSN_CAP_PREAUTH         0x01

/* does frame have QoS sequence control data */
#define IEEE80211_QOS_HAS_SEQ(wh) \
        (((wh)->frame_ctl & \
          (IEEE80211_FC0_TYPE_MASK | IEEE80211_FC0_SUBTYPE_QOS)) == \
          (IEEE80211_FC0_TYPE_DATA | IEEE80211_FC0_SUBTYPE_QOS))
/* End import from Atheros driver src */

struct ieee80211_qos_information_element {
	u8 elementID;
	u8 length;
	u8 qui[QOS_OUI_LEN];
	u8 qui_type;
	u8 qui_subtype;
	u8 version;
	u8 ac_info;
} __attribute__ ((packed));

struct ieee80211_qos_ac_parameter {
	u8 aci_aifsn;
	u8 ecw_min_max;
	__le16 tx_op_limit;
} __attribute__ ((packed));

struct ieee80211_qos_parameter_info {
	struct ieee80211_qos_information_element info_element;
	u8 reserved;
	struct ieee80211_qos_ac_parameter ac_params_record[QOS_QUEUE_NUM];
} __attribute__ ((packed));

struct ieee80211_qos_parameters {
	__le16 cw_min[QOS_QUEUE_NUM];
	__le16 cw_max[QOS_QUEUE_NUM];
	u8 aifs[QOS_QUEUE_NUM];
	u8 flag[QOS_QUEUE_NUM];
	__le16 tx_op_limit[QOS_QUEUE_NUM];
} __attribute__ ((packed));

struct ieee80211_qos_data {
	struct ieee80211_qos_parameters parameters;
	int active;
	int supported;
	u8 param_count;
	u8 old_param_count;
};

struct ieee80211_tim_parameters {
	u8 tim_count;
	u8 tim_period;
} __attribute__ ((packed));

/*******************************************************/

enum {				/* ieee80211_basic_report.map */
	IEEE80211_BASIC_MAP_BSS = (1 << 0),
	IEEE80211_BASIC_MAP_OFDM = (1 << 1),
	IEEE80211_BASIC_MAP_UNIDENTIFIED = (1 << 2),
	IEEE80211_BASIC_MAP_RADAR = (1 << 3),
	IEEE80211_BASIC_MAP_UNMEASURED = (1 << 4),
	/* Bits 5-7 are reserved */

};
struct ieee80211_basic_report {
	u8 channel;
	__le64 start_time;
	__le16 duration;
	u8 map;
} __attribute__ ((packed));

enum {				/* ieee80211_measurement_request.mode */
	/* Bit 0 is reserved */
	IEEE80211_MEASUREMENT_ENABLE = (1 << 1),
	IEEE80211_MEASUREMENT_REQUEST = (1 << 2),
	IEEE80211_MEASUREMENT_REPORT = (1 << 3),
	/* Bits 4-7 are reserved */
};

enum {
	IEEE80211_REPORT_BASIC = 0,	/* required */
	IEEE80211_REPORT_CCA = 1,	/* optional */
	IEEE80211_REPORT_RPI = 2,	/* optional */
	/* 3-255 reserved */
};

struct ieee80211_measurement_params {
	u8 channel;
	__le64 start_time;
	__le16 duration;
} __attribute__ ((packed));

struct ieee80211_measurement_request {
	struct ieee80211_info_element ie;
	u8 token;
	u8 mode;
	u8 type;
	struct ieee80211_measurement_params params[0];
} __attribute__ ((packed));

struct ieee80211_measurement_report {
	struct ieee80211_info_element ie;
	u8 token;
	u8 mode;
	u8 type;
	union {
		struct ieee80211_basic_report basic[0];
	} u;
} __attribute__ ((packed));

struct ieee80211_tpc_report {
	u8 transmit_power;
	u8 link_margin;
} __attribute__ ((packed));

struct ieee80211_channel_map {
	u8 channel;
	u8 map;
} __attribute__ ((packed));

struct ieee80211_ibss_dfs {
	struct ieee80211_info_element ie;
	u8 owner[ETH_ALEN];
	u8 recovery_interval;
	struct ieee80211_channel_map channel_map[0];
};

struct ieee80211_csa {
	u8 mode;
	u8 channel;
	u8 count;
} __attribute__ ((packed));

struct ieee80211_quiet {
	u8 count;
	u8 period;
	u8 duration;
	u8 offset;
} __attribute__ ((packed));

struct ieee80211_network {
	/* These entries are used to identify a unique network */
	u8 bssid[ETH_ALEN];
	u8 channel;
	/* Ensure null-terminated for any debug msgs */
	u8 ssid[IW_ESSID_MAX_SIZE + 1];
	u8 ssid_len;

	struct ieee80211_qos_data qos_data;

	/* These are network statistics */
	struct ieee80211_rx_stats stats;
	u16 capability;
	u8 rates[MAX_RATES_LENGTH];
	u8 rates_len;
	u8 rates_ex[MAX_RATES_EX_LENGTH];
	u8 rates_ex_len;
	unsigned long last_scanned;
	u8 mode;
	u32 flags;
	u32 last_associate;
	u32 time_stamp[2];
	u16 beacon_interval;
	u16 listen_interval;
	u16 atim_window;
	u8 erp_value;
	u8 wpa_ie[MAX_WPA_IE_LEN];
	size_t wpa_ie_len;
	u8 rsn_ie[MAX_WPA_IE_LEN];
	size_t rsn_ie_len;
	struct ieee80211_tim_parameters tim;

	/* 802.11h info */

	/* Power Constraint - mandatory if spctrm mgmt required */
	u8 power_constraint;

	/* TPC Report - mandatory if spctrm mgmt required */
	struct ieee80211_tpc_report tpc_report;

	/* IBSS DFS - mandatory if spctrm mgmt required and IBSS
	 * NOTE: This is variable length and so must be allocated dynamically */
	struct ieee80211_ibss_dfs *ibss_dfs;

	/* Channel Switch Announcement - optional if spctrm mgmt required */
	struct ieee80211_csa csa;

	/* Quiet - optional if spctrm mgmt required */
	struct ieee80211_quiet quiet;

	struct list_head list;
};

enum ieee80211_state {
	IEEE80211_UNINITIALIZED = 0, /* was 0 */
	IEEE80211_INITIALIZED   = 1,
	IEEE80211_ASSOCIATING   = 2,
	IEEE80211_ASSOCIATED    = 3,
	IEEE80211_AUTHENTICATING= 4,
	IEEE80211_AUTHENTICATED = 5,
	IEEE80211_SHUTDOWN	= 6,
/* Below are required by Atheros driver */
	IEEE80211_S_INIT	= 0,
	IEEE80211_S_SCAN 	= 1,
	IEEE80211_S_JOIN 	= 2,
	IEEE80211_S_AUTH 	= 3,
	IEEE80211_S_ASSOC 	= 4,
	IEEE80211_S_RUN 	= 5
};
#ifdef CONFIG_IEEE80211_ATHEROS
#define IEEE80211_S_MAX (IEEE80211_S_RUN+1)
#else
#define IEEE80211_S_MAX (IEEE80211_SHUTDOWN+1)
#endif

#define DEFAULT_MAX_SCAN_AGE (15 * HZ)
#define DEFAULT_FTS 2346

#define CFG_IEEE80211_RESERVE_FCS (1<<0)
#define CFG_IEEE80211_COMPUTE_FCS (1<<1)
#define CFG_IEEE80211_RTS (1<<2)

#define IEEE80211_24GHZ_MIN_CHANNEL 1
#define IEEE80211_24GHZ_MAX_CHANNEL 14
#define IEEE80211_24GHZ_CHANNELS (IEEE80211_24GHZ_MAX_CHANNEL - \
				  IEEE80211_24GHZ_MIN_CHANNEL + 1)

#define IEEE80211_52GHZ_MIN_CHANNEL 34
#define IEEE80211_52GHZ_MAX_CHANNEL 165
#define IEEE80211_52GHZ_CHANNELS (IEEE80211_52GHZ_MAX_CHANNEL - \
				  IEEE80211_52GHZ_MIN_CHANNEL + 1)

enum {
	IEEE80211_CH_PASSIVE_ONLY = (1 << 0),
	IEEE80211_CH_80211H_RULES = (1 << 1),
	IEEE80211_CH_B_ONLY = (1 << 2),
	IEEE80211_CH_NO_IBSS = (1 << 3),
	IEEE80211_CH_UNIFORM_SPREADING = (1 << 4),
	IEEE80211_CH_RADAR_DETECT = (1 << 5),
	IEEE80211_CH_INVALID = (1 << 6),
};

#if 1
/* Also Defined in _ieee80211.h */
struct ieee80211_channel {
	u32 freq;	/* in MHz */
	u8 channel;
	u32 flags;	/* was 'u8' */
	u8 max_power;	/* in dBm */
	/* Below are imports from _ieee80211.h required by Atheros driver */
//        u_int16_t       ic_freq;        /* setting in Mhz */
//        u_int32_t       ic_flags;       /* see below */
#define ic_freq freq
#define ic_flags flags
//        u_int8_t        ic_ieee;        /* IEEE channel number */
#define ic_ieee channel
        int8_t          ic_maxregpower; /* maximum regulatory tx power in dBm */
//        int8_t          ic_maxpower;    /* maximum tx power in dBm */
#define ic_maxpower max_power
        int8_t          ic_minpower;    /* minimum tx power in dBm */
        u_int8_t        ic_regclassid;  /* regulatory class id */
};
#endif

struct ieee80211_geo {
	u8 name[4];
	u8 bg_channels;
	u8 a_channels;
	struct ieee80211_channel bg[IEEE80211_24GHZ_CHANNELS];
	struct ieee80211_channel a[IEEE80211_52GHZ_CHANNELS];
};

struct ieee80211_device {
	struct net_device *dev;
	struct ieee80211_security sec;

	/* Bookkeeping structures */
	struct net_device_stats stats;
	struct ieee80211_stats ieee_stats;

	struct ieee80211_geo geo;

	/* Probe / Beacon management */
	struct list_head network_free_list;
	struct list_head network_list;
	struct ieee80211_network *networks;
	int scans;
	int scan_age;

	int iw_mode;		/* operating mode (IW_MODE_*) */
	struct iw_spy_data spy_data;	/* iwspy support */

	spinlock_t lock;

	int tx_headroom;	/* Set to size of any additional room needed at front
				 * of allocated Tx SKBs */
	u32 config;

	/* WEP and other encryption related settings at the device level */
	int open_wep;		/* Set to 1 to allow unencrypted frames */

	int reset_on_keychange;	/* Set to 1 if the HW needs to be reset on
				 * WEP key changes */

	/* If the host performs {en,de}cryption, then set to 1 */
	int host_encrypt;
	int host_encrypt_msdu;
	int host_decrypt;
	/* host performs multicast decryption */
	int host_mc_decrypt;

	/* host should strip IV and ICV from protected frames */
	/* meaningful only when hardware decryption is being used */
	int host_strip_iv_icv;

	int host_open_frag;
	int host_build_iv;
	int ieee802_1x;		/* is IEEE 802.1X used */

	/* WPA data */
	int wpa_enabled;
	int drop_unencrypted;
	int privacy_invoked;
	size_t wpa_ie_len;
	u8 *wpa_ie;

	struct list_head crypt_deinit_list;
	struct ieee80211_crypt_data *crypt[WEP_KEYS];
	int tx_keyidx;		/* default TX key index (crypt[tx_keyidx]) */
	struct timer_list crypt_deinit_timer;
	int crypt_quiesced;

	int bcrx_sta_key;	/* use individual keys to override default keys even
				 * with RX of broad/multicast frames */

	/* Fragmentation structures */
	struct ieee80211_frag_entry frag_cache[IEEE80211_FRAG_CACHE_LEN];
	unsigned int frag_next_idx;
	u16 fts;		/* Fragmentation Threshold */
	u16 rts;		/* RTS threshold */

	/* Association info */
	u8 bssid[ETH_ALEN];

	enum ieee80211_state state;

	int mode;		/* A, B, G */
	int modulation;		/* CCK, OFDM */
	int freq_band;		/* 2.4Ghz, 5.2Ghz, Mixed */
	int abg_true;		/* ABG flag              */

	int perfect_rssi;
	int worst_rssi;

	u16 prev_seq_ctl;	/* used to drop duplicate frames */

	/* Callback functions */
	void (*set_security) (struct net_device * dev,
			      struct ieee80211_security * sec);
	int (*hard_start_xmit) (struct ieee80211_txb * txb,
				struct net_device * dev, int pri);
	int (*reset_port) (struct net_device * dev);
	int (*is_queue_full) (struct net_device * dev, int pri);

	int (*handle_management) (struct net_device * dev,
				  struct ieee80211_network * network, u16 type);
	int (*is_qos_active) (struct net_device *dev, struct sk_buff *skb);

	/* Typical STA methods */
	int (*handle_auth) (struct net_device * dev,
			    struct ieee80211_auth * auth);
	int (*handle_deauth) (struct net_device * dev,
			      struct ieee80211_deauth * auth);
	int (*handle_action) (struct net_device * dev,
			      struct ieee80211_action * action,
			      struct ieee80211_rx_stats * stats);
	int (*handle_disassoc) (struct net_device * dev,
				struct ieee80211_disassoc * assoc);
	int (*handle_beacon) (struct net_device * dev,
			      struct ieee80211_beacon * beacon,
			      struct ieee80211_network * network);
	int (*handle_probe_response) (struct net_device * dev,
				      struct ieee80211_probe_response * resp,
				      struct ieee80211_network * network);
	int (*handle_probe_request) (struct net_device * dev,
				     struct ieee80211_probe_request * req,
				     struct ieee80211_rx_stats * stats);
	int (*handle_assoc_response) (struct net_device * dev,
				      struct ieee80211_assoc_response * resp,
				      struct ieee80211_network * network);

	/* Typical AP methods */
	int (*handle_assoc_request) (struct net_device * dev);
	int (*handle_reassoc_request) (struct net_device * dev,
				       struct ieee80211_reassoc_request * req);

	/* This must be the last item so that it points to the data
	 * allocated beyond this structure by alloc_ieee80211 */
	u8 priv[0];
};

#define IEEE_A            (1<<0)
#define IEEE_B            (1<<1)
#define IEEE_G            (1<<2)
#define IEEE_MODE_MASK    (IEEE_A|IEEE_B|IEEE_G)

static inline void *ieee80211_priv(struct net_device *dev)
{
	return ((struct ieee80211_device *)netdev_priv(dev))->priv;
}

static inline int ieee80211_is_empty_essid(const char *essid, int essid_len)
{
	/* Single white space is for Linksys APs */
	if (essid_len == 1 && essid[0] == ' ')
		return 1;

	/* Otherwise, if the entire essid is 0, we assume it is hidden */
	while (essid_len) {
		essid_len--;
		if (essid[essid_len] != '\0')
			return 0;
	}

	return 1;
}

static inline int ieee80211_is_valid_mode(struct ieee80211_device *ieee,
					  int mode)
{
	/*
	 * It is possible for both access points and our device to support
	 * combinations of modes, so as long as there is one valid combination
	 * of ap/device supported modes, then return success
	 *
	 */
	if ((mode & IEEE_A) &&
	    (ieee->modulation & IEEE80211_OFDM_MODULATION) &&
	    (ieee->freq_band & IEEE80211_52GHZ_BAND))
		return 1;

	if ((mode & IEEE_G) &&
	    (ieee->modulation & IEEE80211_OFDM_MODULATION) &&
	    (ieee->freq_band & IEEE80211_24GHZ_BAND))
		return 1;

	if ((mode & IEEE_B) &&
	    (ieee->modulation & IEEE80211_CCK_MODULATION) &&
	    (ieee->freq_band & IEEE80211_24GHZ_BAND))
		return 1;

	return 0;
}

static inline int ieee80211_get_hdrlen(u16 fc)
{
	int hdrlen = IEEE80211_3ADDR_LEN;
	u16 stype = WLAN_FC_GET_STYPE(fc);

	switch (WLAN_FC_GET_TYPE(fc)) {
	case IEEE80211_FTYPE_DATA:
		if ((fc & IEEE80211_FCTL_FROMDS) && (fc & IEEE80211_FCTL_TODS))
			hdrlen = IEEE80211_4ADDR_LEN;
		if (stype & IEEE80211_STYPE_QOS_DATA)
			hdrlen += 2;
		break;
	case IEEE80211_FTYPE_CTL:
		switch (WLAN_FC_GET_STYPE(fc)) {
		case IEEE80211_STYPE_CTS:
		case IEEE80211_STYPE_ACK:
			hdrlen = IEEE80211_1ADDR_LEN;
			break;
		default:
			hdrlen = IEEE80211_2ADDR_LEN;
			break;
		}
		break;
	}

	return hdrlen;
}

static inline u8 *ieee80211_get_payload(struct ieee80211_hdr *hdr)
{
	switch (ieee80211_get_hdrlen(le16_to_cpu(hdr->frame_ctl))) {
	case IEEE80211_1ADDR_LEN:
		return ((struct ieee80211_hdr_1addr *)hdr)->payload;
	case IEEE80211_2ADDR_LEN:
		return ((struct ieee80211_hdr_2addr *)hdr)->payload;
	case IEEE80211_3ADDR_LEN:
		return ((struct ieee80211_hdr_3addr *)hdr)->payload;
	case IEEE80211_4ADDR_LEN:
		return ((struct ieee80211_hdr_4addr *)hdr)->payload;
	}
	return NULL;
}

static inline int ieee80211_is_ofdm_rate(u8 rate)
{
	switch (rate & ~IEEE80211_BASIC_RATE_MASK) {
	case IEEE80211_OFDM_RATE_6MB:
	case IEEE80211_OFDM_RATE_9MB:
	case IEEE80211_OFDM_RATE_12MB:
	case IEEE80211_OFDM_RATE_18MB:
	case IEEE80211_OFDM_RATE_24MB:
	case IEEE80211_OFDM_RATE_36MB:
	case IEEE80211_OFDM_RATE_48MB:
	case IEEE80211_OFDM_RATE_54MB:
		return 1;
	}
	return 0;
}

static inline int ieee80211_is_cck_rate(u8 rate)
{
	switch (rate & ~IEEE80211_BASIC_RATE_MASK) {
	case IEEE80211_CCK_RATE_1MB:
	case IEEE80211_CCK_RATE_2MB:
	case IEEE80211_CCK_RATE_5MB:
	case IEEE80211_CCK_RATE_11MB:
		return 1;
	}
	return 0;
}

/* ieee80211.c */
extern void free_ieee80211(struct net_device *dev);
extern struct net_device *alloc_ieee80211(int sizeof_priv);

extern int ieee80211_set_encryption(struct ieee80211_device *ieee);

/* ieee80211_tx.c */
extern int ieee80211_xmit(struct sk_buff *skb, struct net_device *dev);
extern void ieee80211_txb_free(struct ieee80211_txb *);
extern int ieee80211_tx_frame(struct ieee80211_device *ieee,
			      struct ieee80211_hdr *frame, int hdr_len,
			      int total_len, int encrypt_mpdu);

/* ieee80211_rx.c */
extern void ieee80211_rx_any(struct ieee80211_device *ieee,
		     struct sk_buff *skb, struct ieee80211_rx_stats *stats);
extern int ieee80211_rx(struct ieee80211_device *ieee, struct sk_buff *skb,
			struct ieee80211_rx_stats *rx_stats);
/* make sure to set stats->len */
extern void ieee80211_rx_mgt(struct ieee80211_device *ieee,
			     struct ieee80211_hdr_4addr *header,
			     struct ieee80211_rx_stats *stats);
extern void ieee80211_network_reset(struct ieee80211_network *network);

/* ieee80211_geo.c */
extern const struct ieee80211_geo *ieee80211_get_geo(struct ieee80211_device
						     *ieee);
extern int ieee80211_set_geo(struct ieee80211_device *ieee,
			     const struct ieee80211_geo *geo);

extern int ieee80211_is_valid_channel(struct ieee80211_device *ieee,
				      u8 channel);
extern int ieee80211_channel_to_index(struct ieee80211_device *ieee,
				      u8 channel);
extern u8 ieee80211_freq_to_channel(struct ieee80211_device *ieee, u32 freq);
extern u8 ieee80211_get_channel_flags(struct ieee80211_device *ieee,
				      u8 channel);
extern const struct ieee80211_channel *ieee80211_get_channel(struct
							     ieee80211_device
							     *ieee, u8 channel);

/* ieee80211_wx.c */
extern int ieee80211_wx_get_scan(struct ieee80211_device *ieee,
				 struct iw_request_info *info,
				 union iwreq_data *wrqu, char *key);
extern int ieee80211_wx_set_encode(struct ieee80211_device *ieee,
				   struct iw_request_info *info,
				   union iwreq_data *wrqu, char *key);
extern int ieee80211_wx_get_encode(struct ieee80211_device *ieee,
				   struct iw_request_info *info,
				   union iwreq_data *wrqu, char *key);
extern int ieee80211_wx_set_encodeext(struct ieee80211_device *ieee,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu, char *extra);
extern int ieee80211_wx_get_encodeext(struct ieee80211_device *ieee,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu, char *extra);
extern int ieee80211_wx_set_auth(struct net_device *dev,
				 struct iw_request_info *info,
				 union iwreq_data *wrqu,
				 char *extra);
extern int ieee80211_wx_get_auth(struct net_device *dev,
				 struct iw_request_info *info,
				 union iwreq_data *wrqu,
				 char *extra);

static inline void ieee80211_increment_scans(struct ieee80211_device *ieee)
{
	ieee->scans++;
}

static inline int ieee80211_get_scans(struct ieee80211_device *ieee)
{
	return ieee->scans;
}

#endif				/* IEEE80211_H */
