/******************************************************************************
**
** FILE NAME    : amazon_se_sdio.h
** PROJECT      : IFX 
** MODULES      : MMC
**
** DATE         : 07 Jan 2008
** AUTHOR       : Reddy Mallikarjuna
** DESCRIPTION  : IFX MMC  Driver
** COPYRIGHT    :       Copyright (c) 2008
**                      Infineon Technologies AG
**                      Am Campeon 1-12, 85579 Neubiberg, Germany
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
** HISTORY
** $Version $Date      $Author     $Comment
*******************************************************************************/
#ifndef AMAZON_SE_SDIO_H
#define AMAZON_SE_SDIO_H

#include <linux/mmc/protocol.h>

#include <linux/mmc/protocol.h>
#include <linux/highmem.h>
#include <asm/cacheflush.h>
#include <asm/div64.h>
#include <asm/io.h>
#include <asm/scatterlist.h>


/**
  \file amazon_se_sdio.h
  \brief header file for amazon_se sdio driver
*/

/*!
  \defgroup AMAZON_SE_SDIO_DEFINITIONS Amazon_S sdio definitions
  \ingroup AMAZON_SE_SDIO
  \brief definitions for amazon_se sdio driver
* @{
*/


#define RCV_INT          1
#define TX_BUF_FULL_INT  2
#define TRANSMIT_CPT_INT 4

#define DEVICE_NAME "AMAZON_SE_SDIO"

#undef SDIODEBUG
#ifdef SDIO_DEBUG
#define SDIODBGMSG(fmt, args ...) printk(KERN_DEBUG "sdio:" fmt, ##args)
#define SDIOERRMSG(fmt, args ...) printk(KERN_ERR "sdio:" fmt, ##args)
#else
#define SDIODBGMSG(fmt, args ...)
#define SDIOERRMSG(fmt, args ...) printk(KERN_ERR "sdio:" fmt, ##args)
#endif


#define MMC_WRITE_REG32(data,addr)      do{ *((volatile u32*)(addr)) = (u32)(data); asm("SYNC");} while (0)
#define MMC_READ_REG32(addr)    (*((volatile u32*)(addr)))

/*change the GPIO pins based on HW changes*/
/*MCLCMD       GPIO 19*/
#define MCLCMD          19   
/*MCLCLK       GPIO 15*/
#define MCLCLK          15
/*MCLDATA0     GPIO 12*/
#define MCLDATA0        12
/*MCLDATA1     GPIO 26*/
#define MCLDATA1        26
/*MCLDATA2     GPIO 28*/
#define MCLDATA2     28 
/*MCLDATA3     GPIO 20*/
#define MCLDATA3    20 

#define MAX_PIN_PER_PORT      16
/* Every port has 16 pins, up to 4 ports from 0~3 */
#define PIN2PORT(pin)      ((((pin) >> 4) & 0x3))
#define PIN2PORTPIN(pin)   ((pin) % (MAX_PIN_PER_PORT))

#define IFX_SD_PIN_RESERVE(pin) \
	bsp_port_reserve_pin((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_DIR_OUT(pin) \
	bsp_port_set_dir_out((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_DIR_IN(pin) \
	bsp_port_set_dir_in((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_OUTPUT_SET(pin) \
	bsp_port_set_output((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_OUTPUT_CLR(pin) \
	bsp_port_clear_output((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL0_SET(pin) \
	bsp_port_set_altsel0((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL0_CLR(pin) \
	bsp_port_clear_altsel0((PIN2PORT(pin)), (PIN2PORTPIN(pin)),(PORT_MODULE_SDIO))

#define IFX_SD_OD_SET(pin) \
	bsp_port_set_open_drain((PIN2PORT(pin)), (PIN2PORTPIN(pin)),(PORT_MODULE_SDIO))

#define IFX_SD_OD_CLR(pin) \
	bsp_port_clear_open_drain((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL1_SET(pin) \
	bsp_port_set_altsel1((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL1_CLR(pin) \
	bsp_port_clear_altsel1((PIN2PORT(pin)), (PIN2PORTPIN(pin)),(PORT_MODULE_SDIO))

#define IFX_SD_PUDSEL_SET(pin) \
	bsp_port_set_pudsel((PIN2PORT(pin)), (PIN2PORTPIN(pin)),(PORT_MODULE_SDIO))

#define IFX_SD_PUDEN_SET(pin) \
	bsp_port_set_puden((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define BIT0            (1<<0)
#define BIT1            (1<<1)
#define BIT2            (1<<2)
#define BIT3            (1<<3)
#define BIT4            (1<<4)
#define BIT5            (1<<5)
#define BIT6            (1<<6)
#define BIT7            (1<<7)
#define BIT8            (1<<8)
#define BIT9            (1<<9)
#define BIT10           (1<<10)
#define BIT11           (1<<11)
#define BIT12           (1<<12)
#define BIT13           (1<<13)
#define BIT14           (1<<14)
#define BIT15           (1<<15)
#define BIT16           (1<<16)
#define BIT17           (1<<17)
#define BIT18           (1<<18)
#define BIT19           (1<<19)
#define BIT20           (1<<20)
#define BIT21           (1<<21)
#define BIT22           (1<<22)
#define BIT23           (1<<23)
#define BIT24           (1<<24)
#define BIT25           (1<<25)
#define BIT26           (1<<26)
#define BIT27           (1<<27)
#define BIT28           (1<<28)
#define BIT29           (1<<29)
#define BIT30           (1<<30)
#define BIT31           (1<<31)

/*!
* \brief SDIO base address 
*/
/*registers address space */
#define MMCI_ADDR_SPACE         0xBE102000

// registers address
// MCI Registers
/*!
* \brief MCI power control register 
*/
#define MCI_PWR                 (MMCI_ADDR_SPACE + 0x0000)
/* MCI_PWR REGISTER */
/*!
* \brief MCI power off
*/
#define MCI_PWR_OFF             0x0
/*!
* \brief MCI power up command 
*/
#define MCI_PWR_UP              0x2
/*!
* \brief MCI power up on 
*/
#define MCI_PWR_ON              0x3

/*!
* \brief MCI clock control register 
*/
#define MCI_CLK                 (MMCI_ADDR_SPACE + 0x0004)
/*!
* \brief MCI clock enabled 
*/

#define MCI_CLK_ENABLE         BIT8 
/*!
* \brief MCI clock power save 
*/
#define MCI_CLK_PWRSAVE        BIT9 
/*!
* \brief MCI clock bypass 
*/
#define MCI_CLK_BYPASS              BIT10
/*!
* \brief MCI wide bus mode (4-bit mode) 
*/
#define MCI_WIDE_BUS		BIT11
/*!
* \brief MCI wide bus mode  
*/
#define MCI_CLK_WD              BIT11

/*!
* \brief MCI argument register 
*/
#define MCI_ARG                 (MMCI_ADDR_SPACE + 0x0008)

/*!
* \brief MCI command register 
*/
#define MCI_CMD                 (MMCI_ADDR_SPACE + 0x000C)
/*!
* \brief MCI command enabled 
*/
#define MCI_CPSM_ENABLE         BIT10	

/*!
* \brief MCI command pending enabled
*/
#define MCI_CPSM_PENDING        BIT9
/*!
* \brief MCI command interrupt enabled
*/
#define MCI_CPSM_INTERRUPT      BIT8
/*!
* \brief MCI command long response  enabled
*/
#define MCI_CPSM_LONGRSP        BIT7
/*!
* \brief MCI command response  enabled
*/
#define MCI_CPSM_RESPONSE       BIT6
/*!
* \brief MCI command no response flag
*/
#define MCI_CMD_NO_RSP          0
/*!
* \brief MCI command short response flag 
*/
#define MCI_CMD_SHORT_RSP       (MCI_CPSM_RESPONSE)
/*!
* \brief MCI command long response flag 
*/
#define MCI_CMD_LONG_RSP        (MCI_CPSM_RESPONSE | MCI_CPSM_LONGRSP)

/*!
* \brief MCI response command register 
*/
#define MCI_REPCMD              (MMCI_ADDR_SPACE + 0x0010)
/*!
* \brief MCI response 0 register 
*/
#define MCI_REP0                (MMCI_ADDR_SPACE + 0x0014)
/*!
* \brief MCI response 1 register 
*/
#define MCI_REP1                (MMCI_ADDR_SPACE + 0x0018)
/*!
* \brief MCI response 2 register 
*/
#define MCI_REP2                (MMCI_ADDR_SPACE + 0x001C)
/*!
* \brief MCI response 3 register 
*/
#define MCI_REP3                (MMCI_ADDR_SPACE + 0x0020)
/*!
* \brief MCI data timer register 
*/
#define MCI_DTIM                (MMCI_ADDR_SPACE + 0x0024)
/*!
* \brief MCI data length register 
*/
#define MCI_DLGTH               (MMCI_ADDR_SPACE + 0x0028)
/*!
* \brief MCI data control register 
*/
#define MCI_DCTRL               (MMCI_ADDR_SPACE + 0x002C)
/*!
* \brief MCI data control dma mode 
*/
#define MCI_DPSM_DMAENABLE      BIT3
/*!
* \brief MCI data transfer stream mode 
*/
#define MCI_DPSM_MODE           BIT2
/*!
* \brief MCI data direction from card to controller 
*/
#define MCI_DPSM_DIRECTION      BIT1
/*!
* \brief MCI data transfer enabled 
*/
#define MCI_DPSM_ENABLE         BIT0
/*!
* \brief MCI data counter register (returns the remaining data bytes) 
*/
#define MCI_DCNT                (MMCI_ADDR_SPACE + 0x0030)
/*!
* \brief MCI status register 
*/
#define MCI_STAT                (MMCI_ADDR_SPACE + 0x0034)
/*!
* \brief Receive data available
*/
#define MCI_RXDATAAVLBL         BIT21 
/*!
* \brief Transmit data available
*/
#define MCI_TXDATAAVLBL         BIT20
/*!
* \brief Receive FIFO empty
*/
#define MCI_RXFIFOEMPTY         BIT19
/*!
* \brief Transmit FIFO empty
*/
#define MCI_TXFIFOEMPTY         BIT18 
/*!
* \brief Receive FIFO full 
*/
#define MCI_RXFIFOFULL          BIT17
/*!
* \brief Transmit FIFO full
*/
#define MCI_TXFIFOFULL          BIT16
/*!
* \brief Receive FIFO half full 
*/
#define MCI_RXFIFOHALFFULL      BIT15
/*!
* \brief Transmit FIFO half full
*/
#define MCI_TXFIFOHALFEMPTY     BIT14
/*!
* \brief Receive active 
*/
#define MCI_RXACTIVE            BIT13
/*!
* \brief Transmit active 
*/
#define MCI_TXACTIVE            BIT12
/*!
* \brief command active
*/
#define MCI_CMDACTIVE           BIT11
/*!
* \brief data block end
*/
#define MCI_DATABLOCKEND        BIT10
/*!
* \brief start bit error
*/
#define MCI_DATASTARTBITERR             BIT9
/*!
* \brief data end
*/
#define MCI_DATAEND             BIT8
/*!
* \brief command sent
*/
#define MCI_CMDSENT             BIT7
/*!
* \brief command response end 
*/
#define MCI_CMDRESPEND          BIT6
/*!
* \brief Receive overflow
*/
#define MCI_RXOVERRUN           BIT5
/*!
* \brief Transmit under run
*/
#define MCI_TXUNDERRUN          BIT4
/*!
* \brief Data timeout 
*/
#define MCI_DATATIMEOUT         BIT3
/*!
* \brief  Command timeout 
*/
#define MCI_CMDTIMEOUT          BIT2
/*!
* \brief Data CRC fail 
*/
#define MCI_DATACRCFAIL         BIT1
/*!
* \brief Command CRC fail 
*/
#define MCI_CMDCRCFAIL          BIT0

/*!
* \brief MCI interrupt mask clear register 
*/
#define MCI_CL                  (MMCI_ADDR_SPACE + 0x0038)
/*!
* \brief Data block end clear 
*/
#define MCI_DATABLOCKENDCLR     BIT10
/*!
* \brief Start bit error clear
*/
#define MCI_STARTBITERRCLR             BIT9
/*!
* \brief Data end clear
*/
#define MCI_DATAENDCLR          BIT8
/*!
* \brief Command sent clear
*/
#define MCI_CMDSENTCLR          BIT7
/*!
* \brief Command response end clear
*/
#define MCI_CMDRESPENDCLR       BIT6
/*!
* \brief Receive over run clear
*/
#define MCI_RXOVERRUNCLR        BIT5
/*!
* \brief Transmit under run clear
*/
#define MCI_TXUNDERRUNCLR       BIT4
/*!
* \brief Data timeout clear
*/
#define MCI_DATATIMEOUTCLR      BIT3
/*!
* \brief Command timeout clear
*/
#define MCI_CMDTIMEOUTCLR       BIT2
/*!
* \brief Data CRC fail clear
*/
#define MCI_DATACRCFAILCLR      BIT1
/*!
* \brief Command CRC fail clear
*/
#define MCI_CMDCRCFAILCLR       BIT0
/*!
* \brief MCI IM0 interrupt mask Register
*/
#define MCI_IM0                 (MMCI_ADDR_SPACE + 0x003C)

/*!
* \brief MCI IM1 interrupt mask Register
*/
#define MCI_IM1                 (MMCI_ADDR_SPACE + 0x0040)
/*!
* \brief Receive data available interrupt mask
*/
#define MCI_RXDATAAVLBLMASK     BIT21
/*!
* \brief Transmit data available interrupt mask
*/
#define MCI_TXDATAAVLBLMASK     BIT20
/*!
* \brief Receive FIFO empty interrupt mask
*/
#define MCI_RXFIFOEMPTYMASK     BIT19
/*! 
* \brief Transmit FIFO empty interrupt mask
*/
#define MCI_TXFIFOEMPTYMASK     BIT18
/*!
* \brief Receive FIFO full interrupt mask
*/
#define MCI_RXFIFOFULLMASK      BIT17
/*!
* \brief Transmit FIFO full interrupt mask
*/
#define MCI_TXFIFOFULLMASK      BIT16
/*!
* \brief Receive FIFO half full interrupt mask
*/
#define MCI_RXFIFOHALFFULLMASK  BIT15
/*!
* \brief Transmit FIFO half full interrupt mask
*/
#define MCI_TXFIFOHALFEMPTYMASK  BIT14
/*!
* \brief Receive active interrupt mask
*/
#define MCI_RXACTIVEMASK        BIT13 
/*!
* \brief Transmit active interrupt mask
*/
#define MCI_TXACTIVEMASK        BIT12
/*!
* \brief command active interrupt mask
*/
#define MCI_CMDACTIVEMASK       BIT11
/*!
* \brief data block end interrupt mask
*/
#define MCI_DATABLOCKENDMASK    BIT10
/*!
* \brief start bit error interrupt mask
*/
#define MCI_STARTBITERRMASK     BIT9
/*!
* \brief data end interrupt mask
*/
#define MCI_DATAENDMASK               BIT8
/*!
* \brief command sent interrupt mask
*/
#define MCI_CMDSENTMASK         BIT7
/*!
* \brief command response end interrupt mask
*/
#define MCI_CMDRESPENDMASK      BIT6
/*!
* \brief Receive overflow interrupt mask
*/
#define MCI_RXOVERRUNMASK       BIT5
/*!
* \brief Transmit under run interrupt mask
*/
#define MCI_TXUNDERRUNMASK      BIT4
/*!
* \brief Data timeout interrupt mask
*/
#define MCI_DATATIMEOUTMASK     BIT3
/*!
* \brief  Command timeout interrupt mask
*/
#define MCI_CMDTIMEOUTMASK      BIT2
/*!
* \brief Data CRC fail interrupt mask
*/
#define MCI_DATACRCFAILMASK     BIT1
/*!
* \brief Command CRC fail interrupt mask
*/
#define MCI_CMDCRCFAILMASK      BIT0
/*!
* \brief MCI SD memory card select address Register
*/
#define MCI_SDMCS               (MMCI_ADDR_SPACE + 0x0044)
/*!
* \brief MCI FIFO counter register
*/
#define MCI_FC                  (MMCI_ADDR_SPACE + 0x0048)
/*!
* \brief MCI DATA FIFO 0 register
*/
#define MCI_DF0                 (MMCI_ADDR_SPACE + 0x0080)
/*!
* \brief MCI DATA FIFO 1 register
*/
#define MCI_DF1                 (MMCI_ADDR_SPACE + 0x0084)
/*!
* \brief MCI DATA FIFO 2 register
*/
#define MCI_DF2                 (MMCI_ADDR_SPACE + 0x0088)
/*!
* \brief MCI DATA FIFO 3 register
*/
#define MCI_DF3                 (MMCI_ADDR_SPACE + 0x008C)
/*!
* \brief MCI DATA FIFO 4 register
*/
#define MCI_DF4                 (MMCI_ADDR_SPACE + 0x0090)
/*!
* \brief MCI DATA FIFO 5 register
*/
#define MCI_DF5                 (MMCI_ADDR_SPACE + 0x0094)
/*!
* \brief MCI DATA FIFO 6 register
*/
#define MCI_DF6                 (MMCI_ADDR_SPACE + 0x0098)
/*!
* \brief MCI DATA FIFO 7 register
*/
#define MCI_DF7                 (MMCI_ADDR_SPACE + 0x009C)
/*!
* \brief MCI DATA FIFO 8 register
*/
#define MCI_DF8                 (MMCI_ADDR_SPACE + 0x00A0)
/*!
* \brief MCI DATA FIFO 9 register
*/
#define MCI_DF9                 (MMCI_ADDR_SPACE + 0x00A4)
/*!
* \brief MCI DATA FIFO 10 register
*/
#define MCI_DF10                (MMCI_ADDR_SPACE + 0x00A8)
/*!
* \brief MCI DATA FIFO 11register
*/
#define MCI_DF11                (MMCI_ADDR_SPACE + 0x00AC)
/*!
* \brief MCI DATA FIFO 12 register
*/
#define MCI_DF12                (MMCI_ADDR_SPACE + 0x00B0)
/*!
* \brief MCI DATA FIFO 13 register
*/
#define MCI_DF13                (MMCI_ADDR_SPACE + 0x00B4)
/*!
* \brief MCI DATA FIFO 14 register
*/
#define MCI_DF14                (MMCI_ADDR_SPACE + 0x00B8)
/*!
* \brief MCI DATA FIFO 15 register
*/
#define MCI_DF15                (MMCI_ADDR_SPACE + 0x00BC)

/*!
* \brief MCI peripheral ID register 0
*/
#define MCI_PID0                (MMCI_ADDR_SPACE + 0x0FE0)
/*!
* \brief MCI peripheral ID register 1
*/
#define MCI_PID1                (MMCI_ADDR_SPACE + 0x0FE4)
/*!
* \brief MCI peripheral ID register 2
*/
#define MCI_PID2                (MMCI_ADDR_SPACE + 0x0FE8)
/*!
* \brief MCI peripheral ID register 3
*/
#define MCI_PID3                (MMCI_ADDR_SPACE + 0x0FEC)
/*!
* \brief MCI core identification register 0
*/
#define MCI_PCID0               (MMCI_ADDR_SPACE + 0x0FF0)
/*!
* \brief MCI core identification register 1
*/
#define MCI_PCID1               (MMCI_ADDR_SPACE + 0x0FF4)
/*!
* \brief MCI core identification register 2
*/
#define MCI_PCID2               (MMCI_ADDR_SPACE + 0x0FF8)
/*!
* \brief MCI core identification register 3
*/
#define MCI_PCID3               (MMCI_ADDR_SPACE + 0x0FFC)
/*!
* \brief SDIO clock control register
*/
#define SDIO_CLC                (MMCI_ADDR_SPACE + 0x1000)
/*!
* \brief SDIO clock divider 
*/
#define SDIO_CLC_RMC_BYPASS     BIT8
/*!
* \brief SDIO clock fast sgut off enable bit 
*/
#define SDIO_CLC_FSOE           BIT5
/*!
* \brief SDIO clock suspend bit write enable for OCDS
*/
#define SDIO_CLC_SBWE           BIT4
/*!
* \brief SDIO clock external request disable
*/
#define SDIO_CLC_EDIS           BIT3
/*!
* \brief SDIO clock suspend enable bit for OCDS
*/
#define SDIO_CLC_SPEN           BIT2
/*!
* \brief SDIO clock disable status bit
*/
#define SDIO_CLC_DISS           BIT1
/*!
* \brief SDIO clock disable request  bit
*/
#define SDIO_CLC_DISR           BIT0
/*!
* \brief SDIO Identification register 
*/
#define SDIO_ID                 (MMCI_ADDR_SPACE + 0x1004)
/*!
* \brief SDIO control register 
*/
#define SDIO_CTRL               (MMCI_ADDR_SPACE + 0x1008)
/*!
* \brief SDIO Read/Wait control
*/
#define SDIO_CTRL_RDWT          BIT4
/*!
* \brief SDIO enable 
*/
#define SDIO_CTRL_SDIOEN        BIT0
/*!
* \brief SDIO interrupt mask control registr
*/
#define SDIO_IMC                (MMCI_ADDR_SPACE + 0x1010)      
/*!
* \brief SDIO interrupt mask 
*/
#define SDIO_IMC_SDIO           BIT2
/*!
* \brief Interrupt 1  mask 
*/
#define SDIO_IMC_INTR1          BIT1
/*!
* \brief Interrupt 0  mask 
*/
#define SDIO_IMC_INTR0          BIT0

/*!
* \brief SDIO raw interrupt status register
*/
#define SDIO_RIS                (MMCI_ADDR_SPACE + 0x100C)       
/*!
* \brief SDIO  raw interrupt status
*/
#define SDIO_RIS_SDIO           BIT2
/*!
* \brief Interrupt 1  status
*/
#define SDIO_RIS_INTR1          BIT1
/*!
* \brief Interrupt 0  status
*/
#define SDIO_RIS_INTR0          BIT0
/*!
* \brief SDIO masked interrupt status register
*/
#define SDIO_MIS                (MMCI_ADDR_SPACE + 0x1014)
/*!
* \brief SDIO masked interrupt status
*/
#define SDIO_MIS_SDIO           BIT2
/*!
* \brief Interrupt 1 masked  status
*/
#define SDIO_MIS_INTR1          BIT1
/*!
* \brief Interrupt 0 masked  status
*/
#define SDIO_MIS_INTR0          BIT0
/*!
* \brief SDIO interrupt clear register
*/
#define SDIO_ICR                (MMCI_ADDR_SPACE + 0x1018)
/*!
* \brief SDIO interrupt clear 
*/
#define SDIO_ICR_SDIO           BIT2
/*!
* \brief Interrupt 1 clear 
*/
#define SDIO_ICR_INTR1          BIT1
/*!
* \brief Interrupt 0 clear 
*/
#define SDIO_ICR_INTR0          BIT0
/*!
* \brief Interrupt set register
*/
#define SDIO_ISR                (MMCI_ADDR_SPACE + 0x101C)
/*!
* \brief SDIO Interrupt set 
*/
#define SDIO_ISR_SDIO           BIT2
/*!
* \brief INTR0 interrupt 1  set 
*/
#define SDIO_ISR_INTR1          BIT1
/*!
* \brief INTR0 interrupt 0  set 
*/
#define SDIO_ISR_INTR0          BIT0
/*!
* \brief SDIO DMA control register
*/
#define SDIO_DMACON             (MMCI_ADDR_SPACE + 0x1020)
/*!
* \brief DMA Tx path on
*/
#define SDIO_DMACON_TXON        BIT1
/*!
* \brief DMA Rx path on
*/
#define SDIO_DMACON_RXON        BIT0
/*!
* \brief DMA class
*/
#define SDIO_DMA_CLASS          BIT2

#define SDIO_RSVD_ADDR0         (MMCI_ADDR_SPACE + 0x10F0)
/*!
* \brief Success
*/
#define OK 			0
/*!
* \brief Error code 
*/
#define ERROR_NOMEM		-1	// out of memory
/*!
* \brief Error code 
*/
#define ERROR_CRC_ERROR		-2
/*!
* \brief Error code 
*/
#define ERROR_TIMEOUT		-3
/*!
* \brief Error code 
*/
#define ERROR_WRONG_CARD_STATE 	-4
/*!
* \brief Error code 
*/
#define ERROR_WRONG_RESPONSE_TYPE -5
/*!
* \brief Error code 
*/
#define ERROR_WRONG_RESPONSE_CMD  -6
/*!
* \brief Error code 
*/
#define ERROR_DATA_ERROR	-7

#define MCI_IRQENABLE   \
        (MCI_CMDCRCFAILMASK|MCI_DATACRCFAILMASK|MCI_CMDTIMEOUTMASK|     \
        MCI_DATATIMEOUTMASK|MCI_TXUNDERRUNMASK|MCI_RXOVERRUNMASK|       \
        MCI_CMDRESPENDMASK|MCI_CMDSENTMASK|MCI_DATABLOCKENDMASK | MCI_STARTBITERRMASK)

/*!
* \brief Interrupt mode
*/
#define CMD_INTERRUPT_MODE	1
/*!
* \brief Polling mode 
*/
#define CMD_POLLING_MODE 	0

/*!
* \brief Set 1-bit Bus mode
*/
#define SCR_SD_BUS_WIDTH_1BIT (1<<0)
/*!
* \brief Set 4-bit Bus mode
*/
#define SCR_SD_BUS_WIDTH_4BIT (1<<2)
#define SD_SET_VDD       1
/*!
* \brief SDIO operation command value
*/
#define SD_SET_FREQENCY  2
/*!
* \brief SDIO operation command value
*/
#define SD_SET_BUS_WIDTH 3
/*!
* \brief SDIO operation command value
*/
#define SD_SET_BLOCK_LEN 4

/*!
* \brief Ioctl command 
*/
#define AMAZON_SE_SDIO_SEND_CMD           1
#define AMAZON_SE_SDIO_SEND_DATA_SYNC     2
#define AMAZON_SE_SDIO_SEND_DATA_ASYNC    3
#define AMAZON_SE_SDIO_READ_DATA_SYNC     4
#define AMAZON_SE_SDIO_READ_DATA_ASYNC    5
/*!
* \brief Ioctl command 
*/
#define AMAZON_SE_SDIO_SET_OPS_WBUS       6
/*!
* \brief Ioctl command 
*/
#define AMAZON_SE_SDIO_SET_OPS_FREQUENCY  7
/*!
* \brief Ioctl command 
*/
#define AMAZON_SE_SDIO_GET_OPS_WBUS       8
/*!
* \brief Ioctl command 
*/
#define AMAZON_SE_SDIO_GET_OPS_FREQUENCY  9

/*!
* \brief clock 
*/
#define SD_CLK_400K	400000

#define SD_BUS_1BITS	1
#define SD_BUS_4BITS	4

#define IFX_DMA_CTRL                        AMAZON_SE_DMA_CTRL
#define IFX_DMA_CPOLL                       AMAZON_SE_DMA_CPOLL
#define IFX_DMA_CS                          AMAZON_SE_DMA_CS(0)
#define IFX_DMA_CCTRL                       AMAZON_SE_DMA_CCTRL(0)
#define IFX_DMA_CDBA                        AMAZON_SE_DMA_CDBA(0)
#define IFX_DMA_CDLEN                       AMAZON_SE_DMA_CDLEN(0)
#define IFX_DMA_CIS                         AMAZON_SE_DMA_CIS(0)
#define IFX_DMA_CIE                         AMAZON_SE_DMA_CIE(0)
#define IFX_DMA_PS                          AMAZON_SE_DMA_PS(0)
#define IFX_DMA_PCTRL                       AMAZON_SE_DMA_PCTRL(0)
#define IFX_DMA_IRNEN                       AMAZON_SE_DMA_IRNEN
#if 0
#define MAX_PIN_PER_PORT      16
/* Every port has 16 pins, up to 4 ports from 0~3 */
#define PIN2PORT(pin)      ((((pin) >> 4) & 0x3))
#define PIN2PORTPIN(pin)   ((pin) % (MAX_PIN_PER_PORT))

#define IFX_SD_PIN_RESERVE(pin) \
            bsp_port_reserve_pin((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_DIR_OUT(pin) \
            bsp_port_set_dir_out((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_DIR_IN(pin) \
            bsp_port_set_dir_in((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_OUTPUT_SET(pin) \
            bsp_port_set_output((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_OUTPUT_CLR(pin) \
            bsp_port_clear_output((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL0_SET(pin) \
            bsp_port_set_altsel0((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL0_CLR(pin) \
            bsp_port_clear_altsel0((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_OD_SET(pin) \
            bsp_port_set_open_drain((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_OD_CLR(pin) \
            bsp_port_clear_open_drain((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL1_SET(pin) \
            bsp_port_set_altsel1((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_ALTSEL1_CLR(pin) \
            bsp_port_clear_altsel1((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_PUDSEL_SET(pin) \
            bsp_port_set_pudsel((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))

#define IFX_SD_PUDEN_SET(pin) \
            bsp_port_set_puden((PIN2PORT(pin)), (PIN2PORTPIN(pin)), (PORT_MODULE_SDIO))
#endif
#define ifx_mmc_card_present(c)     ((c)->state & MMC_STATE_PRESENT)
#define ifx_mmc_card_dead(c)        ((c)->state & MMC_STATE_DEAD)
#define ifx_mmc_card_bad(c)         ((c)->state & MMC_STATE_BAD)
#define ifx_mmc_card_sd(c)          ((c)->state & MMC_STATE_SDCARD)
#define ifx_mmc_card_readonly(c)    ((c)->state & MMC_STATE_READONLY)
#define ifx_mmc_card_highspeed(c)   ((c)->state & MMC_STATE_HIGHSPEED)

#define ifx_mmc_card_set_present(c) ((c)->state |= MMC_STATE_PRESENT)
#define ifx_mmc_card_set_dead(c)    ((c)->state |= MMC_STATE_DEAD)
#define ifx_mmc_card_set_bad(c)     ((c)->state |= MMC_STATE_BAD)
#define ifx_mmc_card_set_sd(c)      ((c)->state |= MMC_STATE_SDCARD)
#define ifx_mmc_card_set_readonly(c) ((c)->state |= MMC_STATE_READONLY)
#define ifx_mmc_card_set_highspeed(c) ((c)->state |= MMC_STATE_HIGHSPEED)

#define ifx_mmc_card_name(c)        ((c)->cid.prod_name)
#define ifx_mmc_card_id(c)          ((c)->dev.bus_id)

#define ifx_mmc_get_drvdata(c)      dev_get_drvdata(&(c)->dev)
#define ifx_mmc_set_drvdata(c,d)    dev_set_drvdata(&(c)->dev, d)


/** @} */
/*
 * The size of the FIFO in bytes.
 */
#define MCI_FIFOSIZE    (16*4)

#define MCI_FIFOHALFSIZE (MCI_FIFOSIZE / 2)

#define NR_SG           16

/*!
* \brief struct sdio command
*/
typedef struct sdio_cmd {
	uint32_t op_code; /*!< command opcode */
	uint32_t args; /*!< command argument */
	uint32_t response_type;	/*!< command response type*/
	uint32_t response[4]; /*!< command response */
	int error;	/*!< error codes*/
//	int mode;	/*!< command mode (Interrupt(1)/ Polling(0)) */
}x_IFX_sdio_cmd_t;

/*!
* \brief struct sdio block request 
*/
typedef struct {
        uint8_t block_size_shift; /*!< block size shift bits*/
        int nBlocks;    /*!< no. of blocks*/
        uint8_t *pData_buffer;  /*!< pointer to data */
} x_IFX_sdio_block_request_t;

/*!
* \brief struct ifx sdio host  
*/
struct ifx_sdio_host {
        void __iomem            *base;  /*!< io memory map */
        struct mmc_request      *mrq;  /*!< struct mmc request */
        struct mmc_command      *cmd;  /*!< struct mmc command */
        struct mmc_data         *data; /*!< struct mmc data */
        struct mmc_host         *mmc;  /*!< struct mmc host */

        unsigned int            data_xfered; /*!< transfered bytes */

        spinlock_t              lock; /*!< host busy lock*/

        unsigned int            mclk; /*!< max host support clock */
        unsigned int            cclk; /*!< current clock */
        u32                     pwr;  /*!< power */
        struct mmc_platform_data *plat;  /*!< struct mmc platform data */

        struct timer_list       timer; /*!< struct timer */
        unsigned int            oldstat; /*!< old status */

        unsigned int            sg_len;  /*!< scatter buffer length*/

        /* pio stuff */
        struct scatterlist      *sg_ptr; /*!< struct scatterlist */
        unsigned int            sg_off; /*!< scatter offset */
        unsigned int            size; /*!< size of packet*/
};

static inline void ifx_sdio_init_sg(struct ifx_sdio_host *host, struct mmc_data *data)
{
        /*
         * Ideally, we want the higher levels to pass us a scatter list.
         */
        host->sg_len = data->sg_len;
        host->sg_ptr = data->sg;
        host->sg_off = 0;
}

static inline int ifx_sdio_next_sg(struct ifx_sdio_host *host)
{
        host->sg_ptr++;
        host->sg_off = 0;
        return --host->sg_len;
}

static inline char *ifx_sdio_kmap_atomic(struct ifx_sdio_host *host, unsigned long *flags)
{
        struct scatterlist *sg = host->sg_ptr;

        local_irq_save(*flags);
//      return kmap_atomic(sg->page, KM_BIO_SRC_IRQ) + sg->offset;
        return kmap(sg->page ) + sg->offset;
}

static inline void ifx_sdio_kunmap_atomic(struct ifx_sdio_host *host, void *buffer, unsigned long *flags)
{
//      kunmap_atomic(buffer, KM_BIO_SRC_IRQ);
        kunmap(buffer);
        local_irq_restore(*flags);
}


struct mmc_ios {
        unsigned int    clock;                  /* clock rate */
        unsigned short  vdd;

#define MMC_VDD_150     0
#define MMC_VDD_155     1
#define MMC_VDD_160     2
#define MMC_VDD_165     3
#define MMC_VDD_170     4
#define MMC_VDD_180     5
#define MMC_VDD_190     6
#define MMC_VDD_200     7
#define MMC_VDD_210     8
#define MMC_VDD_220     9
#define MMC_VDD_230     10
#define MMC_VDD_240     11
#define MMC_VDD_250     12
#define MMC_VDD_260     13
#define MMC_VDD_270     14
#define MMC_VDD_280     15
#define MMC_VDD_290     16
#define MMC_VDD_300     17
#define MMC_VDD_310     18
#define MMC_VDD_320     19
#define MMC_VDD_330     20
#define MMC_VDD_340     21
#define MMC_VDD_350     22
#define MMC_VDD_360     23

        unsigned char   bus_mode;               /* command output mode */

#define MMC_BUSMODE_OPENDRAIN   1
#define MMC_BUSMODE_PUSHPULL    2

        unsigned char   chip_select;            /* SPI chip select */

#define MMC_CS_DONTCARE         0
#define MMC_CS_HIGH             1
#define MMC_CS_LOW              2

        unsigned char   power_mode;             /* power supply mode */

#define MMC_POWER_OFF           0
#define MMC_POWER_UP            1
#define MMC_POWER_ON            2

        unsigned char   bus_width;              /* data bus width */
#define MMC_BUS_WIDTH_1         0
#define MMC_BUS_WIDTH_4         2
};

struct mmc_host_ops {
        void    (*request)(struct mmc_host *host, struct mmc_request *req);
        void    (*set_ios)(struct mmc_host *host, struct mmc_ios *ios);
        int     (*get_ro)(struct mmc_host *host);
};

struct mmc_card;
struct device;

struct mmc_host {
        struct device           *parent;
        struct device           class_dev;
        int                     index;
        const struct mmc_host_ops *ops;
        unsigned int            f_min;
        unsigned int            f_max;
        u32                     ocr_avail;

        unsigned long           caps;           /* Host capabilities */

#define MMC_CAP_4_BIT_DATA      (1 << 0)        /* Can the host do 4 bit transfers */
#define MMC_CAP_MULTIWRITE      (1 << 1)        /* Can accurately report bytes sent to card on error */
#define MMC_CAP_BYTEBLOCK       (1 << 2)        /* Can do non-log2 block sizes */

        /* host specific block data */
        unsigned int            max_seg_size;   /* see blk_queue_max_segment_size */
        unsigned short          max_hw_segs;    /* see blk_queue_max_hw_segments */
        unsigned short          max_phys_segs;  /* see blk_queue_max_phys_segments */
        unsigned short          max_sectors;    /* see blk_queue_max_sectors */
        unsigned short          unused;

        /* private data */
        struct mmc_ios          ios;            /* current io bus settings */
        u32                     ocr;            /* the current OCR setting */

        unsigned int            mode;           /* current card mode of host */
#define MMC_MODE_MMC            0
#define MMC_MODE_SD             1

        struct list_head        cards;          /* devices attached to this host */

        wait_queue_head_t       wq;
        spinlock_t              lock;           /* card_busy lock */
        struct mmc_card         *card_busy;     /* the MMC card claiming host */
        struct mmc_card         *card_selected; /* the selected MMC card */

        struct delayed_work     detect;

        unsigned long           private[0] ____cacheline_aligned;
};

static inline void *ifx_mmc_priv(struct mmc_host *host)
{
        return (void *)host->private;
}

#define ifx_mmc_dev(x)      ((x)->parent)
#define ifx_mmc_hostname(x) ((x)->class_dev.bus_id)

extern struct mmc_host *ifx_mmc_alloc_host(struct mmc_host *host, struct device *dev);
extern int ifx_mmc_add_host(struct mmc_host *);
extern void ifx_mmc_remove_host(struct mmc_host *);
extern void ifx_mmc_free_host(struct mmc_host *);
extern int ifx_mmc_suspend_host(struct mmc_host *, pm_message_t);
extern int ifx_mmc_resume_host(struct mmc_host *);

extern void ifx_mmc_detect_change(struct mmc_host *, unsigned long delay);
extern void ifx_mmc_request_done(struct mmc_host *, struct mmc_request *);

extern int ifx_sdio_send_cmd (x_IFX_sdio_cmd_t *pCmd );
extern int ifx_sdio_controller_set_ops (int type, uint32_t data);
extern int ifx_sdio_write_stream_data_fifo_mode(x_IFX_sdio_cmd_t *pCmd, void *buffer, int length, unsigned long *status, unsigned long timeout);
extern int ifx_sdio_read_stream_data_fifo_mode(x_IFX_sdio_cmd_t *pCmd, void *buffer, int length, unsigned long *status, unsigned long timeout);
extern int ifx_sdio_write_block_data_fifo_mode( x_IFX_sdio_cmd_t *pCmd, void *buffer, int block_size_shift,unsigned long *status, unsigned long timeout);
extern int ifx_sdio_read_blcok_data_fifo_mode(x_IFX_sdio_cmd_t *pCmd, void *buffer, int block_size_shift,unsigned long *status, unsigned long timeout);
extern int ifx_sdio_write_stream_data_dma_mode (x_IFX_sdio_cmd_t *pCmd,  void *buffer,   int len);
extern int ifx_sdio_read_stream_data_dma_mode (x_IFX_sdio_cmd_t *pCmd,  void *buffer,   int len);
extern int ifx_sdio_write_block_data_dma_mode (x_IFX_sdio_cmd_t *pCmd,  x_IFX_sdio_block_request_t * pRequest,   unsigned long timeout);
extern int ifx_sdio_read_block_data_dma_mode (x_IFX_sdio_cmd_t *pCmd,  x_IFX_sdio_block_request_t * pRequest,   unsigned long timeout);

#endif /* AMAZON_SE_SDIO_H */

