/******************************************************************************
**
** FILE NAME    : amazon_se_sflash.c
** PROJECT      : amazon_se
** MODULES      : SPI Flash
**
** DATE         : 20 Nov 2007
** AUTHOR       : Lei Chuanhua
** DESCRIPTION  : SPI Flash MTD Driver
** COPYRIGHT    :       Copyright (c) 2007
**                      Infineon Technologies AG
**                      Am Campeon 1-12, 85579 Neubiberg, Germany
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
** HISTORY
** $Date        $Author         $Comment
**  20 Nov 2007  Lei Chuanhua    Initiate Version
**  28 Nov 2007  Lei Chuanhua    SPI DMA support
*******************************************************************************/

#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <asm/delay.h>
#include <asm/io.h>
#include <asm/semaphore.h>
#include <linux/ctype.h>
#include <linux/interrupt.h>

#if CONFIG_MODVERSIONS == 1
#include <linux/modversions.h>
#endif
#include <asm/amazon_se/amazon_se.h>
#include <asm/amazon_se/irq.h>
#include <asm/amazon_se/ifx_ssc_defines.h>
#include <asm/amazon_se/ifx_ssc.h>

#include <asm/ioctl.h>

#include "amazon_se_sflash.h"

#undef SPI_FLASH_DBG

#if defined(SPI_FLASH_DBG)
#define DBG_SPI(format, arg...)   \
    printk("%s: " format, __func__, ##arg)
#define INLINE 
#else
#define DBG_SPI(format, arg...)   \
    do {} while(0)
#define INLINE inline
#endif

/* SPI flash device driver parts */
#define MAX_ADDRESS_NUM			    5
#define MAX_DUMMY_CYCLES            10

#define ITEMS(X) (sizeof(X)/sizeof (X[0]))

/* Driver private data */
typedef struct {
	struct 	mtd_info       *mtd;	
	struct 	mtd_partition  *parsed_parts;     /* parsed partitions */
    u32 cs;
    u8 addr_cycles;
} spi_dev_t;

typedef struct ifx_flash_config {
    u32     size;
    u32     nsectors;
    u32     sector_size;
    u32     pgsize;
}ifx_flash_config_t; 

static spi_dev_t *spi_sflash;


/* Other vendor, ie, sst/stm can use different different table according to vendor and device id */
ifx_flash_config_t mx_flash_cfgtbl[] = {
    { 0, 0, 0, 0},
    { MX_256KB_SIZE_COUNT, MX_256KB_SECTOR_COUNT, MX_256KB_SECTOR_SIZE, SPI_FALSH_PAGE_SIZE},	
    { MX_512KB_SIZE_COUNT, MX_512KB_SECTOR_COUNT, MX_512KB_SECTOR_SIZE, SPI_FALSH_PAGE_SIZE},	
    { MX_1MB_SIZE_COUNT, MX_1MB_SECTOR_COUNT, MX_1MB_SECTOR_SIZE, SPI_FALSH_PAGE_SIZE},	
    { MX_2MB_SIZE_COUNT, MX_2MB_SECTOR_COUNT, MX_2MB_SECTOR_SIZE, SPI_FALSH_PAGE_SIZE}, 
    { MX_4MB_SIZE_COUNT, MX_4MB_SECTOR_COUNT, MX_4MB_SECTOR_SIZE, SPI_FALSH_PAGE_SIZE},
    { MX_8MB_SIZE_COUNT, MX_8MB_SECTOR_COUNT, MX_8MB_SECTOR_SIZE, SPI_FALSH_PAGE_SIZE},
};

#ifdef CONFIG_MTD_PARTITIONS
/*
 * Here are partition information for all known amazon_se series devices.
 * See include/linux/mtd/partitions.h for definition of the mtd_partition
 * structure.
 */
#if 0	// use single MTD table for simulate smaller flash size
#define AMAZON_SE_SYTEM_PARTITION_NUMBER       3
#define AMAZON_SE_SYSTEM_PARTITION_2MB_SIZE    0x001C0000
#define AMAZON_SE_SYSTEM_PARTITION_4MB_SIZE    0x003C0000
#define AMAZON_SE_SYSTEM_PARTITION_8MB_SIZE    0x007C0000

static struct mtd_partition ifx_spi_partitions[MX_MAX_FLASH][AMAZON_SE_SYTEM_PARTITION_NUMBER] = {
    {{0, 0, 0}},

/* 256K Byte */
    {{0, 0, 0}},

/* 512K Byte */ 
    {{0, 0, 0}},

/* 1M Byte */ 
    {{0, 0, 0}},

/* 2M Byte */
    {{
        name:         "spi-boot",		/* U-Boot firmware */
        offset:       0x00000000,
        size:         0x00010000,	       /* 128K */
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    },
    {
        name:         "spi-firmware",	/* firmware */
        offset:       0x00010000,
        size:         0x00030000,	       /* 192K */
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    },
    {
        name:         "spi-rootfs,kernel,Data,Environment",		/* default partition */
        offset:       0x00040000,
        size:         AMAZON_SE_SYSTEM_PARTITION_2MB_SIZE,
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    }},
/* 4M Byte */
    {{
        name:         "spi-boot",		/* U-Boot firmware */
        offset:       0x00000000,
        size:         0x00010000,	       /* 128K */
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    },
    {
        name:         "spi-firmware",	/* firmware */
        offset:       0x00010000,
        size:         0x00030000,	       /* 256K */
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    },
    {
        name:         "spi-rootfs,kernel,Data,Environment",		/* default partition */
        offset:       0x00040000,
        size:         AMAZON_SE_SYSTEM_PARTITION_4MB_SIZE,
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    }},
/* 8M Byte */
    {{
        name:         "spi-boot",		/* U-Boot firmware */
        offset:       0x00000000,
        size:         0x00010000,	       /* 128K */
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    },
    {
        name:         "spi-firmware",	/* firmware */
        offset:       0x00010000,
        size:         0x00030000,	       /* 192K */
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    },
    {
        name:         "spi-rootfs,kernel,Data,Environment",		/* default partition */
        offset:       0x00040000,
        size:         AMAZON_SE_SYSTEM_PARTITION_8MB_SIZE,
    /*	mask_flags:   MTD_WRITEABLE,    force read-only */
    }},
};
#else
#if (CONFIG_MTD_AMAZON_SE_SPI_FLASH_SIZE == 2)
	#ifdef CONFIG_IFX_ADSL_FIRMWARE_IN_ROOTFS
		#define AMAZON_SE_SYSTEM_PARTITION_SIZE    0x001F0000
	#else
		#define AMAZON_SE_SYSTEM_PARTITION_SIZE    0x001C0000
	#endif
#elif (CONFIG_MTD_AMAZON_SE_SPI_FLASH_SIZE == 4)
	#ifdef CONFIG_IFX_ADSL_FIRMWARE_IN_ROOTFS
		#define AMAZON_SE_SYSTEM_PARTITION_SIZE    0x003F0000
	#else
		#define AMAZON_SE_SYSTEM_PARTITION_SIZE    0x003C0000
	#endif
#else
	#error  "Configure Amazon_SE MTD flash size first!!"
#endif

#if 0
#ifdef CONFIG_IFX_ADSL_FIRMWARE_IN_ROOTFS
static struct mtd_partition ifx_spi_partitions[] = {
	{
		name:	"U-Boot",
		offset:	0x00000000,
		size:	0x00010000,
	},
	{
		name:	"RootFS,Kernel,Data,Environment",
		offset:	0x00010000,
		size:	AMAZON_SE_SYSTEM_PARTITION_SIZE,
	},
};
#else
static struct mtd_partition ifx_spi_partitions[] = {
	{
		name:	"U-Boot",
		offset:	0x00000000,
		size:	0x00010000,
	},
	{
		name:	"Firmware",
		offset:	0x00010000,
		size:	0x00030000,
	},
	{
		name:	"RootFS,Kernel,Data,Environment",
		offset:	0x00040000,
		size:	AMAZON_SE_SYSTEM_PARTITION_SIZE,
	},
};
#endif /* CONFIG_IFX_ADSL_FIRMWARE_IN_ROOTFS */
#endif
/* partition for DGN1000 */
#define _1K             1024
#define _1M             (_1K * _1K)
#define _64K            (64 * _1K)              //64K
#define _2M             (2 * _1M)               //2M
#define _4M             (4 * _1M)
#define BASEOFF         0x00000000
#define BOOT_S          (2*_64K)
#define ENV_MAC_S       _64K
#define DPF_S           _64K
#define NVRAM_S         _64K
#define KERNEL_S        (_1M)                   //512K
#define ROOTFS_S        (_4M - (BOOT_S + ENV_MAC_S + DPF_S + NVRAM_S + KERNEL_S))

static struct mtd_partition ifx_spi_partitions[] = {
    {
            name:   "U-Boot",
            offset: BASEOFF,
            size:   BOOT_S,
    },
    {
            name:   "ENV_MAC",
            offset: BASEOFF + BOOT_S,
            size:   ENV_MAC_S,
    },
    {
            name:   "DPF",
            offset: BASEOFF + BOOT_S + ENV_MAC_S,
            size:   DPF_S,
    },
    {
            name:   "NVRAM",
            offset: BASEOFF + BOOT_S + ENV_MAC_S + DPF_S,
            size:   NVRAM_S,
    },
    {
            name:   "RootFS_DPUMP",
            offset: BASEOFF + BOOT_S + ENV_MAC_S + DPF_S + NVRAM_S,
            size:   ROOTFS_S,
    },
    {
            name:   "Kernel",
            offset: BASEOFF + BOOT_S + ENV_MAC_S + DPF_S + NVRAM_S + ROOTFS_S,
            size:   KERNEL_S,
    },
    {       //for easy web upgrade
            name:   "ROOTFS_KERNEL",
            offset: BASEOFF + BOOT_S + ENV_MAC_S + DPF_S + NVRAM_S,
            size:   ROOTFS_S+KERNEL_S,
    },
    {       //for pot
            name:   "POT",
            offset: BASEOFF,
            size:   BOOT_S,
    },
};

#endif

#endif  /* CONFIG_MTD_PARTITIONS */

#ifdef CONFIG_AMAZON_SE_SPI_DMA
static  int spi_dma_enabled = 0;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,52))
MODULE_PARM(spi_dma_enabled, "i");
#else
#include <linux/moduleparam.h>
module_param(spi_dma_enabled, int, 0600);
#endif 
MODULE_PARM_DESC(spi_dma_enabled, "spi flash dma mode enabled or disabled(0 -disabled(default), 1 -enabled");
#endif

extern int ifx_ssc_init (void);
extern int ifx_ssc_open (struct inode *inode, struct file *filp);
extern int ifx_ssc_close (struct inode *inode, struct file *filp);
extern void ifx_ssc_cleanup_module (void);
extern int ifx_ssc_ioctl (struct inode *inode, struct file *filp,
			  unsigned int cmd, unsigned long data);
extern int ifx_ssc_cs_low (u32 pin);
extern int ifx_ssc_cs_high (u32 pin);
extern int ifx_ssc_txrx (char *tx_buf, u32 tx_len, char *rx_buf, u32 rx_len);
extern int ifx_ssc_tx (char *tx_buf, u32 tx_len);
extern int ifx_ssc_rx (char *rx_buf, u32 rx_len);

#ifdef CONFIG_AMAZON_SE_SPI_DMA
/* DMA operation */
extern int ifx_spi_dma_tx(char * tx_buf, u32 tx_len);
extern int ifx_spi_dma_rx(char * rx_buf, u32 rx_len); 
#endif

#ifdef SPI_FLASH_DBG
static void 
flash_dump(const char *title, const u8 *buf, size_t len)
{
    int i, llen;
    const u8 *pos = buf;
    const int line_len = 16;

    printk("%s - hex_ascii(len=%lu):\n", title, (unsigned long) len);
    while (len) {
        llen = len > line_len ? line_len : len;
        printk("    ");
        for (i = 0; i < llen; i++)
            printk(" %02x", pos[i]);
        for (i = llen; i < line_len; i++)
            printk("   ");
        printk("   ");
        for (i = 0; i < llen; i++) {
            if (isprint(pos[i]))
                printk("%c", pos[i]);
            else
                printk("_");
        }
        for (i = llen; i < line_len; i++)
            printk(" ");
        printk("\n");
        pos += llen;
        len -= llen;
    }
}
#endif /* SPI_FLASH_DBG */

/*	Brief:	convert address from u32 to u8 array 
 *	Parameter:
 *		address: the address, maximum 32 bit
 *		addr:	array that holds the results, maximum 4 elemets
 */
static void 
u32_to_u8_addr(u32 address, u8* addr)
{
    addr[0] = (u8) ((address>>16) & 0xff);
    addr[1] = (u8) ((address>>8) & 0xff);
    addr[2] = (u8) (address & 0xff);
}

/* Serial flash interface */
static int 
sflash_rdsr(spi_dev_t *dev, u8 *status)
{
    int ret = 0;
    unsigned long flag;
    u8 cmd = SFLASH_RDSR;

    local_irq_save(flag);
    if ((ret = ifx_ssc_cs_low (dev->cs))) {
        goto sflash_rdsr_out;
    }
    if ((ret = ifx_ssc_txrx((char *)&cmd, sizeof(u8), status, sizeof (u8))) < 0) {
        ifx_ssc_cs_high(dev->cs);
        goto sflash_rdsr_out;
    }

    if ((ret = ifx_ssc_cs_high(dev->cs))) {
        goto sflash_rdsr_out;
    }
sflash_rdsr_out:
    local_irq_restore(flag);
    return ret;
}

static void 
sflash_sync(spi_dev_t *dev)	
{
    int ret = 0;
    u8 status;
    
    while (1) {
        ret = sflash_rdsr(dev, &status);
        if (ret) {
            DBG_SPI("Read back status fails %d\n", ret);
            break;
        }
        else if ((status & SFLASH_SR_WIP) == 0) {
            break;
        }
    }
}

static int
amazon_se_spi_txonly(void)
{
    int ret = 0;
    u32 data;
	
    data = (u32) IFX_SSC_MODE_TX;
    if ((ret = ifx_ssc_ioctl ((struct inode *) 0, 
        NULL, IFX_SSC_RXTX_MODE_SET, (unsigned long) &data))) {
		DBG_SPI("Set RxTx mode fails\n");
		return ret;
	}
	return ret;
}

static int
amazon_se_spi_rxonly(void)
{
    int ret = 0;
    u32 data;
	
    data = (u32) IFX_SSC_MODE_RX;
    if ((ret = ifx_ssc_ioctl ((struct inode *) 0, 
            NULL, IFX_SSC_RXTX_MODE_SET, (unsigned long) &data))) {
        DBG_SPI("Set RxTx mode fails\n");
        return ret;
    }
    return ret;
}

static int
amazon_se_spi_txrx(void)
{
    int ret = 0;
    u32 data;
	
    data = (u32) IFX_SSC_MODE_RXTX;
    if ((ret = ifx_ssc_ioctl ((struct inode *) 0, 
            NULL, IFX_SSC_RXTX_MODE_SET, (unsigned long) &data))) {
        DBG_SPI("Set RxTx mode fails %d\n", ret);
        return ret;
    }
    return ret;
}

static INLINE int 
amazon_se_spi_tx(char * txbuf, u32 txlen)
{
#ifdef CONFIG_AMAZON_SE_SPI_DMA
    if (spi_dma_enabled) {
        return  ifx_spi_dma_tx(txbuf, txlen);
    }
    else
#endif
    return ifx_ssc_tx(txbuf, txlen);
}

static INLINE int 
amazon_se_spi_rx(char * rxbuf, u32 rxlen)
{
#ifdef CONFIG_AMAZON_SE_SPI_DMA
    if (spi_dma_enabled) {
        return  ifx_spi_dma_rx(rxbuf, rxlen);
    }
    else
#endif
    return ifx_ssc_rx(rxbuf, rxlen);
}

static int
sflash_session_cmd_addr(spi_dev_t *dev, u8 cmd, u8 *addr, u8 dummy_cycles)
{
    int i;
    int start = 0;
    u8 buf[16] = {0};
    int ret = 0;

    /* CMD */
    buf[0] = cmd;
    start = 1;

    /* Address */
    if (addr != NULL) {
        for (i = 0; i < dev->addr_cycles; i++) {
            buf[start + i] = addr[i];
        }
        start += dev->addr_cycles;
    }

    /* Dummy cycles */
    if (dummy_cycles > 0 && dummy_cycles < MAX_DUMMY_CYCLES) {
        for (i = 0; i < dummy_cycles; i ++) {
            buf[start + i] = 0;
        }
        start += dummy_cycles;
    }
    if ((ret = ifx_ssc_tx((char *)buf, start)) < 0) {
        goto cmd_out;
    }
cmd_out:
    return ret;		
}

static int 
sflash_session(spi_dev_t *dev, u8 cmd, u8 *addr, u8 dummy_cycles, 
                 u8 * wbuf, u32 wcnt, u8 * rbuf, u32 rcnt)
{
 	int ret = 0;
    unsigned long flag;
	
    sflash_sync(dev);
	local_irq_save(flag);
	if ((ret = ifx_ssc_cs_low(dev->cs)) < 0) {
        goto sflash_session_out;
    }
	
	if (sflash_session_cmd_addr(dev, cmd, addr, dummy_cycles) < 0){
        DBG_SPI("sflash_session_cmd_addr fails %d\n", ret);
        ifx_ssc_cs_high(dev->cs);
        goto sflash_session_out;
	}

	/* Write operation */
	if (wcnt > 0) {
        if ((ret = amazon_se_spi_tx(wbuf, wcnt)) != wcnt) {
            DBG_SPI("amazon_se_spi_tx fails %d\n", ret);
            ifx_ssc_cs_high(dev->cs);
            goto sflash_session_out;
        }
	}

	/* Read operation */
    if (rcnt > 0) {
        if ((ret = amazon_se_spi_rx(rbuf, rcnt)) != rcnt) {
            DBG_SPI("amazon_se_spi_rx fails %d\n", ret);
            ifx_ssc_cs_high(dev->cs);
            goto sflash_session_out;
        }
    }
    if ((ret = ifx_ssc_cs_high(dev->cs))) {
        goto sflash_session_out;
    }	
sflash_session_out:		
    local_irq_restore(flag);
    return ret;	
}

static INLINE int 
sflash_wren(void)
{
    return sflash_session(spi_sflash, SFLASH_WREN, NULL, 0, NULL, 0, NULL, 0);
}

static INLINE int 
sflash_rdid(u32 *data)
{
    return sflash_session(spi_sflash, SFLASH_RDID, NULL, 0, NULL, 0, (char *)data, sizeof (u32));
}

static INLINE int
sflash_se(u8 *addr)
{
    return sflash_session(spi_sflash, SFLASH_SE, addr, 0, NULL, 0, NULL, 0);
}

static INLINE int 
sflash_ce(void)
{
    return sflash_session(spi_sflash, SFLASH_CE, NULL, 0, NULL, 0, NULL, 0);
}

static INLINE int 
sflash_pp(u8 *addr, u8 *buf, u32 len)
{
    return sflash_session(spi_sflash, SFLASH_PP, addr, 0, buf, len, NULL, 0);
}

static INLINE int 
sflash_rd(u8 *addr, u8 *buf, u32 len)
{
    return sflash_session(spi_sflash, SFLASH_READ, addr, 0, NULL, 0, buf, len);
}

static INLINE int
amazon_se_spi_open(void)
{
    int ret = 0;	
    if ((ret = ifx_ssc_open ((struct inode *) 0, NULL))) {
        DBG_SPI("ifx_ssc_open fails\n");
        return ret;
    }
    amazon_se_spi_txrx();
    return ret;
}

static INLINE int 
amazon_se_spi_close(void)
{
    int ret = 0;
    if ((ret = ifx_ssc_close((struct inode *) 0, NULL))) {
        DBG_SPI("ifx_ssc_close fails\n");
    }
    return ret;
}

static INLINE void 
amazon_se_spi_flash_baudrate(u32 bdrate)
{
    u32 data = bdrate;
    ifx_ssc_ioctl ((struct inode *) 0, NULL, IFX_SSC_BAUD_SET,
        (unsigned long) &data);
}

static INLINE void 
amazon_se_spi_flash_mode(int mode)
{
}

static INLINE int 
amazon_se_spi_read(u32 saddr, u8 *buf, u32 len)
{
    int ret; int i, j;
    u8 addr[MAX_ADDRESS_NUM] = {0};

    u32_to_u8_addr(saddr, addr);	
    amazon_se_spi_open();
    ret = sflash_rd(addr, buf, len); 
#if 0
    for(j=0;j<len/16;j++){printk("%p|", saddr);
	for(i=0;i<16;i++) 
		printk("%2x ", *(buf++));
	printk("\n");
	saddr +=16;
    }
#endif
//    flash_dump("abc", buf, len);
    amazon_se_spi_close();
    return ret;
}

static void hex_dump(char *str, unsigned char *pSrcBufVA, unsigned int SrcBufLen)
{
    unsigned char *pt;
    int x;
    pt = pSrcBufVA;
    printk("%s: %p, len = %d\n", str, pSrcBufVA, SrcBufLen);
    for (x=0; x<SrcBufLen; x++)
    {
        if (x % 16 == 0)
            printk("0x%04x : ", x);
        printk("%02x ", ((unsigned char)pt[x]));
        if (x%16 == 15)
            printk("\n");
    }
    printk("\n");
}

/* define for Atheros eeprom */
#define SC_EEPROM_START_ADDR         0x0002f000
#define SC_EEPROM_MAX                0x1000
static char eeprom_buf[4096];

int sc_flash_read(u32 saddr, u8 *buf, u32 len)
{
    if (saddr !=  SC_EEPROM_START_ADDR || len != SC_EEPROM_MAX){
        printk("Error: Atheros eeprom address = 0x%08x, len = 0x%08x.", SC_EEPROM_START_ADDR, SC_EEPROM_MAX);
		return 0;
	}
    memcpy(buf, eeprom_buf, len);
    printk("#### saddr=%x,len=%d\n",saddr,len);
    hex_dump("Read out:", buf, len);
    return 0;
}
EXPORT_SYMBOL(sc_flash_read);

static INLINE int 
amazon_se_spi_write(u32 saddr, u8 *buf,u32 len)
{
    int ret;
    u8 addr[MAX_ADDRESS_NUM] = {0};

    u32_to_u8_addr(saddr, addr);
    amazon_se_spi_open();
    sflash_wren();
    ret = sflash_pp(addr, buf, len);
    amazon_se_spi_close();
    return ret;
}

static INLINE void 
amazon_se_spi_chip_erase(void)
{
    amazon_se_spi_open();
    sflash_wren();
    sflash_ce(); 
    amazon_se_spi_close();
}

static INLINE void 
amazon_se_spi_sector_erase(u32 saddr)
{
    u8 addr[MAX_ADDRESS_NUM] = {0};

    u32_to_u8_addr(saddr, addr);
    amazon_se_spi_open();	
    sflash_wren();
    sflash_se(addr);
    amazon_se_spi_close();
}

static INLINE void
amazon_se_spi_rdid(u32 *status)
{
    amazon_se_spi_open();
    sflash_rdid(status);
    amazon_se_spi_close();
}

/* OS glue interface according to different platforms */
#ifdef CONFIG_MIPS_AMAZON_SE
#define IFX_SFLASH_MODE                0
#define IFX_SFLASH_DRV_VERSION         "0.0.1"
#define IFX_SFLASH_CS                  IFX_SSC_WHBGPOSTAT_OUT0_POS
#define IFX_SFLASH_NAME                "amazon_se-spi"
#if 0
#define IFX_SFLASH_BAUDRATE	       2000000                       /* 10 MHz */
#else	// suggested by Xiaogang
#define IFX_SFLASH_BAUDRATE	       5000000                       /* 25 MHz */
#endif
#define IFX_SFLASH_ADDR_CYCLES         3                             /* 24 bit address */
#define ifx_spi_rdid(status)           amazon_se_spi_rdid((status))
#define ifx_spi_sector_erase(addr)     amazon_se_spi_sector_erase((addr))
#define ifx_spi_read(addr, buf, len)   amazon_se_spi_read((addr), (buf), (len))
#define ifx_spi_write(addr, buf, len)  amazon_se_spi_write((addr), (buf), (len))
#define ifx_spi_chip_erase(dummy)      amazon_se_spi_chip_erase((dummy))  
#define ifx_spi_flash_baudrate(bdrate) amazon_se_spi_flash_baudrate((bdrate))
#define ifx_spi_flash_mode(mode)       amazon_se_spi_flash_mode((mode))
#else 
#error "SPI flash not supported yet for this platform"
#endif

static void 
ifx_spi_flash_version(void)
{
    printk("Infineon Technologies Synchronous SPI flash driver version %s \n", 
        IFX_SFLASH_DRV_VERSION);
} 

static int
ifx_spi_flash_probe(void)
{
    u32 tmp;
    u8  vid;
    u8  mid;
    u8  sig;
    u8  idx = 0;
	
    ifx_spi_rdid(&tmp);
	
    vid = tmp & 0xff;
    sig = (tmp >> 8) & 0xff;
    mid = (tmp >> 16) & 0xff;
	DBG_SPI("Vendor %02x Type %02x sig %02x\n", vid, mid, sig);
    switch (vid) {
        case MX_VENDOR_ID:
            if (mid == MX_MEM_TYPE) {
                switch (sig) {
                    case MX_2MBIT_SIGNATURE:
                        idx = FLASH_256KB;
                        break;
                    case MX_4MBIT_SIGNATURE:
                        idx = FLASH_512KB;
                        break;
                    case MX_8MBIT_SIGNATURE:
                        idx = FLASH_1MB;
                        break;						
                    case MX_16MBIT_SIGNATURE:
                        idx = FLASH_2MB;
                        break;						
                    case MX_32MBIT_SIGNATURE:
                        idx = FLASH_4MB;
                        break;
                    case MX_64MBIT_SIGNATURE:
                        idx = FLASH_8MB;
                        break;
                    default:
                        idx = 0;
                        break;						
                }
            }
            break;
        case STM_VENDOR_ID:
            printk("%s STM SPI flash Not tested yet\n", __func__);
            break;
        default:
            break;
    }
    return idx;
}

static int
ifx_spi_flash_read(struct mtd_info *mtd, loff_t from, size_t len,
                  size_t *retlen ,u_char *buf)
{
    DBG_SPI ("(from = 0x%.8x, len = %d)\n",(u32) from, (int)len); 	
    if (!len)
        return 0;
    if ((from + len) > mtd->size)
        return (-EINVAL);	
	ifx_spi_read(from, buf, len);
	*retlen = len;
    return 0;
}

static int
ifx_spi_flash_write (struct mtd_info *mtd, loff_t to, size_t len, 
                    size_t *retlen, const u_char *buf)
{
    int total = 0, len_this_lp, bytes_this_page;
    u32 addr = 0;
    u_char *mem;
	
    DBG_SPI ("(to = 0x%.8x, len = %d)\n",(u32) to,len);
    *retlen = 0;
	
	/* sanity check */
    if (!len)
        return 0;
	
    if ((to + len) > mtd->size) 
        return (-1);
    while(total < len) {
        mem              = buf + total;
        addr             = to  + total;
        bytes_this_page  = SPI_FALSH_PAGE_SIZE - (addr % SPI_FALSH_PAGE_SIZE);
        len_this_lp      = min((len - total), bytes_this_page);
        ifx_spi_write(addr, mem, len_this_lp);
        total += len_this_lp;
    }

    *retlen = len;
    return 0;
}

static int
ifx_spi_flash_erase(struct mtd_info *mtd,struct erase_info *instr)
{
    int nsect, s_curr, s_last;

    DBG_SPI("(addr = 0x%.8x, len = %d)\n", instr->addr, instr->len);

    if ((instr->addr + instr->len) > mtd->size) 
        return (-EINVAL);

    nsect = instr->len/mtd->erasesize;
    if ((instr->len % mtd->erasesize) > 0)
        nsect ++;

    s_curr = instr->addr/mtd->erasesize;
    s_last  = s_curr + nsect;

    do {	
		ifx_spi_sector_erase(s_curr * mtd->erasesize);
    } while (++s_curr < s_last);

	instr->state = MTD_ERASE_DONE;	
    if (instr->callback) {
        instr->callback (instr);
    }
    DBG_SPI ("return\n");
    return 0;
}

static int __init 
ifx_spi_flash_init (void)
{
    int result;
    ifx_flash_config_t *flash_cfg;
    struct mtd_info *mtd;
    u8 index;
#ifdef CONFIG_MTD_CMDLINE_PARTS
    int np;
#endif /* CONFIG_MTD_CMDLINE_PARTS */	

#if 1
        /* P0.7 SPI_CS1, P0.8 SPI_DIN, P0.9 SPI_DOUT, P0.10 SPI_CLK */
        *(AMAZON_SE_GPIO_P0_DIR) = (((*AMAZON_SE_GPIO_P0_DIR)|(0x0780)) & (~(0x0100)));
        *(AMAZON_SE_GPIO_P0_ALTSEL0) = (((*AMAZON_SE_GPIO_P0_ALTSEL0)|(0x0780)) & (~(0x0000)));
        *(AMAZON_SE_GPIO_P0_ALTSEL1) = (((*AMAZON_SE_GPIO_P0_ALTSEL1)|(0x0780)) & (~(0x0780)));
        *(AMAZON_SE_GPIO_P0_OD) = (*AMAZON_SE_GPIO_P0_OD)|(0x0680);
#endif

    ifx_spi_flash_version();

    spi_sflash = (spi_dev_t*)kmalloc(sizeof(spi_dev_t), GFP_KERNEL);
    if (!spi_sflash) 
        return (-ENOMEM);
	
    spi_sflash->cs = IFX_SFLASH_CS;  
    spi_sflash->addr_cycles = IFX_SFLASH_ADDR_CYCLES;
	
    ifx_spi_flash_mode(IFX_SFLASH_MODE);
    ifx_spi_flash_baudrate(IFX_SFLASH_BAUDRATE);
	
    mtd = kmalloc(sizeof(struct mtd_info), GFP_KERNEL);
    if ( mtd == NULL ) {
        kfree(spi_sflash);
        printk(KERN_WARNING "%s Cant allocate mtd stuff\n", __func__);
        return -1;
    }
    memset(mtd, 0, sizeof(struct mtd_info));
	
    printk("MTD driver for SPI flash.\n");
    printk("Probing for Serial flash ...\n");

    index = ifx_spi_flash_probe();
    if (index == 0) {
        printk (KERN_WARNING "%s: Found no serial flash device\n", __func__);
        kfree(spi_sflash);
        kfree(mtd);
       	return (-ENOMEM);
    }
    flash_cfg               =   &mx_flash_cfgtbl[index];
    mtd->name               =   IFX_SFLASH_NAME;
    mtd->type               =   MTD_NORFLASH;
    mtd->flags              =   (MTD_CAP_NORFLASH|MTD_WRITEABLE); 
    mtd->size               =   flash_cfg->size;
    mtd->erasesize          =   flash_cfg->sector_size;
    mtd->numeraseregions    =   0;
    mtd->eraseregions       =   NULL;
//#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,5,41)
//    mtd->module 			=   THIS_MODULE;
//#else
    mtd->owner              =   THIS_MODULE;
//#endif 
    mtd->erase              =   ifx_spi_flash_erase;
    mtd->read               =   ifx_spi_flash_read;
    mtd->write              =   ifx_spi_flash_write;
#ifdef SPI_FLASH_DBG
    printk (KERN_DEBUG
            "mtd->name = %s\n"
            "mtd->size = 0x%.8x (%uM)\n"
            "mtd->erasesize = 0x%.8x (%uK)\n"
            "mtd->numeraseregions = %d\n",
            mtd->name,
            mtd->size, mtd->size / (1024*1024),
            mtd->erasesize, mtd->erasesize / 1024,
            mtd->numeraseregions);

    if (mtd->numeraseregions) {
        for (result = 0; result < mtd->numeraseregions; result++) {
            printk (KERN_DEBUG
                "\n\n"
                "mtd->eraseregions[%d].offset = 0x%.8x\n"
                "mtd->eraseregions[%d].erasesize = 0x%.8x (%uK)\n"
                "mtd->eraseregions[%d].numblocks = %d\n",
                result, mtd->eraseregions[result].offset,
                result, mtd->eraseregions[result].erasesize,
                mtd->eraseregions[result].erasesize / 1024,
                result, mtd->eraseregions[result].numblocks);
         }
    }
#endif	/* SPI_FLASH_DBG */

#ifdef CONFIG_MTD_PARTITIONS

#ifdef CONFIG_MTD_CMDLINE_PARTS
    np = parse_cmdline_partitions(mtd, &spi_sflash->parsed_parts, "ifx spi controller");
    if ( np > 0 ) 
    	add_mtd_partitions(mtd, spi_sflash->parsed_parts, np);
    else
        add_mtd_device(mtd);
#else
#if 0	// use single MTD table for simulate smaller flash size
    add_mtd_partitions(mtd, ifx_spi_partitions[index], ITEMS(ifx_spi_partitions[index]));
#else
    add_mtd_partitions(mtd, ifx_spi_partitions, ITEMS(ifx_spi_partitions));
#endif
#endif /* CONFIG_MTD_CMDLINE_PARTS */

#endif /* CONFIG_MTD_PARTITIONS */
    spi_sflash->mtd = mtd;

    //Read atheros's eeprom to buffer.
    amazon_se_spi_read(SC_EEPROM_START_ADDR, eeprom_buf, SC_EEPROM_MAX);
    return 0;
}

static void __exit 
ifx_spi_flash_exit(void)
{
    if (spi_sflash != NULL) {  
        if (spi_sflash->parsed_parts != NULL) {
            del_mtd_partitions (spi_sflash->mtd);
        }
        kfree(spi_sflash->mtd);
		kfree(spi_sflash);
    }
}

module_init(ifx_spi_flash_init);
module_exit(ifx_spi_flash_exit);
MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Chuanhua.Lei@infineon.com");
MODULE_DESCRIPTION ("IFAP SPI flash device driver");

