
/*
**  FILE               ssd.h
**  AUTHOR:            Yanir Lubetkin
**  ABSTRACT:
      This file contains the interface between the shared sdio framework and its
      clients (device drivers). The code within this file implement the interface
      as defined in the following documents:
      TI Shared SDIO Driver FS.doc
      DaVinci sdio driver design.doc


**  RECORD OF MODIFICATIONS:
**    DATE            VERSION      AUTHOR              DESCRIPTION
**    Mar 02 2006     1.0          Yanir Lubetkin      file creation
**    Mar 13 2006     1.1          Yanir Lubetkin      changes to naming conventions, added constants
**    Apr 05 2006     1.2          Yanir Lubetkin      added legacy sd module external interface
*/





#ifndef __DAVINCI_SSD_H
#define __DAVINCI_SSD_H

/*
 *   DATA DEFINITIONS
 */

/*
 * NAME:          struct client_params
 * ABSTRACT:      This structure defines a device driver's request to use the SDIO bus.
 *                It is passed by the client as a parameter to the "open()" call.
 * MEMBERS:

DevId: The Device ID is used to designate the specific device to which the command
is sent (as given in the remapping command: CMD0).

OrigCmdCode: The device default proprietary CMD code. This value is used by the
driver when issuing the remapping command. If OrigCmdCode is 0 then standard
SDIO operation is used for the device.

NewCmdCode: The driver uses this parameter to change the CMD code used for
the device. If the parameter is equal to the OrigCmdCode, the remapping command
is not required, otherwise, the driver will issue CMD0 (where VS Command Index =
OrigCmdCode, Device VSCI = NewCmdCode).

BusWidth: 1 or 4 bit.
FunctionSelectMode: the parameter indicates the method that selects the device
function field. The value 0 will be used if the function index corresponds to
proprietary command code (i.e. CmdCode = NewCmdCode + FunctionIndex). The
value 1 will be used if the function index is selected using the remapping command
(CMD0).

BlockSize: For multi-block mode, this is the basic block size (0 means no multi-block
support).

Timeout: This is the timeout [in mSec] for the bus transaction. Passing the timeout in
synchronous mode immediately returns the context of the call with an error indication.
Passing timeout in Asynchronous mode triggers the callback with an error status.

BusTxnCB: The callback routine that is triggered upon the end of a bus transaction
(after asynchronous read/write). The status parameter is OK (0) if the transaction was
completed successfully, or an error code (negative number) upon any fault. Possible
errors: TIMEOUT (-1).
Note:
BusTxnCB is not used to trigger bus transactions or to manipulate the bus guard, but
to schedule a DPC of the specific device driver that issues a bus transaction.

BusTxnHandle:
The SDIO driver and BusTxnCB callback returns this handler. Used as SSD
client context.

MaxClockRate: The maximum allow SDIO clock rate for the required device in MHz.
              the actual max timeout is 0xFFFF clock cycles. timeout will be calculated
							to a number of clock cycles closest but not exceeding the requested timeout.
BusGuardCallback: Bus Guard takes client callback called by the Bus Guard
module whenever a pending bus request of the client is granted.

BusBuardHandle: Client specific Bus-Guard parameters, for example, priority (schedule
dependent).
*/
/* bus width allowed values*/
#define SSD_BUSWIDTH_1 1
#define SSD_BUSWIDTH_4 4
/* function select mode values */
#define FUNCTION_SELECT_MODE_LEGACY 0
#define FUNCTION_SELECT_MODE_REMAP 1
/* orig/new cmd code values */
#define LEGACY_ORG_CMD_CODE 0
#define LEGACY_NEW_CMD_CODE 0

typedef struct{
	int DevId;/* WLAN dev-id is 2 */
	int OrigCmdCode;
	int NewCmdCode;
	int BusWidth;
	int  FunctionSelectMode;
	int blockSize;
	void (*BusTxnCB)(void* BusTxnHandle, int status);
	void* BusTxnHandle;
	void (*BusGuardCB)(void* BusGuardHandle, int status);
	void* BusGuardHandle;
	void* BusGuardParams;
	int Timeout;
	int MaxClockRate;
} ssd_client_params_t;


static unsigned long rcaValue;


/*
 * NAME:          ssd_dma_params_t
 * ABSTRACT:      will be used for client configured dma IO. (see legacy SD)
 * MEMBERS:       TBD
*/

typedef struct{
	int TBD;
}ssd_dma_params_t;




/*
 *   FUNCTION DECLARATIONS
 */

/*
 * NAME:          ssd_SDIO_open
 * ABSTRACT:      a device driver's request to use the SDIO framework to access
 *                it's device.
 * PARAMETERS:    struct client_params* param: a set of parameters needed to
 *                initialize sdio (see above)
 * RETURN VALUE:  int: a client identifier to be passed as clientID to any other
 *                call to the framework. negative value on failure, >=0 on success.
 */
int ssd_SDIO_open(ssd_client_params_t* param);


/*
 * NAME:          ssd_SDIO_enumerate
 * ABSTRACT:      perform bus enumeration (cmd0, cmd5, cmd7, ....) on the device
 * PARAMETERS:    int ClientID: a valid client identifier that was returned by
 *                              a previous call to open.
 * RETURN VALUE:  int: indicating the status of the operation. 0 on success,
 *                negative value otherwise.
 */
int ssd_SDIO_enumerate(int ClientID);


/*
 * NAME:          ssd_SDIO_close
 * ABSTRACT:      notify that a device driver is no longer going to use the
 *                framework.
 * PARAMETERS:    int ClientID: a valid client identifier that was returned by
 *                              a previous call to open.
 * RETURN VALUE:  int: indicating the status of the operation. 0 on success,
 *                negative value otherwise.
 */
int ssd_SDIO_close(int ClientID);


/*
 * NAME:          ssd_SDIO_writeSync
 * ABSTRACT:      write to the device. the call returns after all the data has
 *                been written. the implementation uses the CPU and is designed
 *                for small transfers.
 * PARAMETERS:    int ClientID: as returned from a previous call to open.
 *                int Address: bus address to which data will be written
 *                void *DataPointer: pointer to the data to be written
 *                int Length: length of the data buffer. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SDIO_writeSync(int ClientID, int Address, void *DataPointer, int Length);


/*
 * NAME:          ssd_SDIO_writeAsync
 * ABSTRACT:      write to the device. the call returns after the request was
 *                submitted. the implementation uses DMA to transfer data
 *                NOTE: buffers passed to this function must be allocated from
 *                a memory region accessible by DMA.
 * PARAMETERS:    int ClientID: as returned from a previous call to open.
 *                int Address: bus address to which data will be written
 *                void *DataPointer: pointer to the data to be written (DMA accessible)
 *                int Length: length of the data buffer. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SDIO_writeAsync(int ClientID, int Address, void *DataPointer, int Length);


/*
 * NAME:          ssd_SDIO_writeByte
 * ABSTRACT:      write 1 byte to the device using command 52. the call returns
 *                after data has been written.
 * PARAMETERS:    int ClientID: as returned from a previous call to open.
 *                int Address: bus address to which data will be written
 *                char data: the value to be written to the device
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SDIO_writeByte(int ClientID, int Address, unsigned char data);


/*
 * NAME:          ssd_SDIO_readSync
 * ABSTRACT:      read from the device. the call returns after all the data has
 *                been read. the implementation uses the CPU and is designed
 *                for small transfers.
 * PARAMETERS:    int ClientID: as returned from a previous call to open.
 *                int Address: bus address from which data will be read
 *                void *DataPointer: pointer to write the data to
 *                int Length: how much data to read. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SDIO_readSync(int ClientID, int Address, void *DataPointer, int Length);


/*
 * NAME:          ssd_SDIO_readAsync
 * ABSTRACT:      read from the device. the call returns after the request was
 *                submitted. the implementation uses DMA to transfer data
 *                NOTE: buffers passed to this function must be allocated from
 *                a memory region accessible by DMA.
 * PARAMETERS:    int ClientID: as returned from a previous call to open.
 *                int Address: bus address from which data will be read
 *                void *DataPointer: pointer to write the data to
 *                int Length: how much data to read. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SDIO_readAsync(int ClientID, int Address, void *DataPointer, int Length);


/*
 * NAME:          ssd_SDIO_readByte
 * ABSTRACT:      read 1 byte from the device using command 52. the call returns
 *                after data has been read.
 * PARAMETERS:    int ClientID: as returned from a previous call to open.
 *                int Address: bus address to which data will be written
 *                char* data: pointer to a memory area to receive the data
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SDIO_readByte(int ClientID, int Address, unsigned char* data);


/*
 * NAME:          ssd_functionSelect
 * ABSTRACT:      change the device function to which the driver addresses
 * PARAMETERS:    int clientID: as returned from a previous call to open.
 *                int functionID: the function number to which commands will be
 *                sent
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_functionSelect(int clientID, int functionID);


/*
 * NAME:          ssd_busSeize
 * ABSTRACT:      acquire exclusive access to the SDIO bus opened by the driver.
 * PARAMETERS:    int clientID: as returned from a previous call to open.
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_busSeize(int ClientID);


/*
 * NAME:          ssd_busReSeize
 * ABSTRACT:      release and acquire exclusive access to the SDIO bus opened by
 *                the driver. allows a driver to yield and release the bus for
 *                other devices.
 * PARAMETERS:    int clientID: as returned from a previous call to open.
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_busReSeize(int ClientID);


/*
 * NAME:          ssd_busRelease
 * ABSTRACT:      release the SDIO bus held by the driver.
 * PARAMETERS:    int clientID: as returned from a previous call to open.
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_busRelease(int ClientID);


/*
 * NAME:          ssd_SD_Open
 * ABSTRACT:      opens a legacy SD device
 * PARAMETERS:    int BusWidth: device requested bus width
                  int blockSize: device block size
									void (*BusTxnCB)(int status, void* handler): callback for
									async completion notification
									int Timeout: how long should the hardware wait before 
									abandoning an operation
									int MaxClockRate: requested clock rate for the bus
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SD_Open(int BusWidth, int blockSize, 
                void (*BusTxnCB)(int status, void* handler), int Timeout,
								int MaxClockRate);

								
/*
 * NAME:          ssd_SD_close
 * ABSTRACT:      notify that a device driver is no longer going to use the
                  framework.
 * PARAMETERS:    none
 * RETURN VALUE:  int: indicating the status of the operation. 0 on success,
                  negative value otherwise.
 */
void ssd_SD_Close(void);


/*
 * NAME:          ssd_SD_writeAsync
 * ABSTRACT:      write to the device. the call returns after the request was
 *                submitted. the implementation uses DMA to transfer data
 *                NOTE: buffers passed to this function must be allocated from
 *                a memory region accessible by DMA.
 * PARAMETERS:    int Address: bus address to which data will be written
 *                void *DataPointer: pointer to the data to be written (DMA accessible)
 *                int Length: length of the data buffer. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SD_WriteAsync(int Address, void* DataPointer, int Length);


/*
 * NAME:          ssd_SD_readAsync
 * ABSTRACT:      read from the device. the call returns after the request was
 *                submitted. the implementation uses DMA to transfer data
 *                NOTE: buffers passed to this function must be allocated from
 *                a memory region accessible by DMA.
 * PARAMETERS:    int Address: bus address from which data will be read
 *                void *DataPointer: pointer to write the data to
 *                int Length: how much data to read. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SD_ReadAsync(int Address, void* DataPointer, int Length);


/*
 * NAME:          ssd_SD_WriteDMA_Async
 * ABSTRACT:      write to the device. the call returns after the request was
 *                submitted. the implementation uses client configured DMA 
 *                to transfer data
 * PARAMETERS:    int Address: bus address to which data will be written
 *                void *DmaParams: pointer to the above struct dma_params
 *                int Length: length of the data buffer. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SD_WriteDMA_Async(int Address, void* DmaParams, int Length);


/*
 * NAME:          ssd_SD_readDMA_Async
 * ABSTRACT:      read from the device. the call returns after the request was
 *                submitted. the implementation uses client configured DMA 
 *                to transfer data
 * PARAMETERS:    int Address: bus address from which data will be read
 *                void *DmaParams: pointer to the above struct dma_params
 *                int Length: length of the data buffer. must be a multiple of 4
 * RETURN VALUE:  int: operation status. 0 on success, negative value on error
 */
int ssd_SD_readDMA_Async(int Address, void* DmaParams, int Length);

#endif
