/* =============================================================================
** FILE NAME     : hmcp.c
** PROJECT       : HMCP on AMAZON_S
** MODULES       : Kernel specific HMCP
** DATE          : 26-09-2008
** AUTHOR        : Kishore Kankipati
** DESCRIPTION   : Kernel specific HMCP functions' implementation
**                 
** REFERENCES    :
** COPYRIGHT     : Copyright (c) 2008
**                 Infineon Technologies AG,
**                 Am Campeon 1-12, 85579 Neubiberg, Germany
**
** Any use of this software is subject to the conclusion of a respective
** License agreement. Without such a License agreement no rights to the
** software are granted
**
** HISTORY       :
** $Date   $Author    $Comment
**
** ============================================================================
*/

#include <asm/hmcp.h>
#include <linux/netdevice.h>
volatile unsigned long vpe_device_schedule[NUM_VPE] = {0};
struct net_device *vpe_devices[NUM_VPE][MAX_NET_DEV_VPE];

#ifdef CONFIG_MIPS_MT_HMCP_IPI
spinlock_t vpe_lock[NUM_VPE];
void (*ipi_vpe)(unsigned int, unsigned int);

static int inter_vpe_msg_ipi(struct net_device* dev, u32 dest_vpe, u32 src_vpe)
{
	if (dev && test_and_set_bit(dev->vpe_sched_idx, 
				&vpe_device_schedule[dest_vpe])) {
		return -1;
	}
	if (!spin_trylock(&vpe_lock[dest_vpe]))
		return -1;
	(*ipi_vpe)(dest_vpe, src_vpe);
	spin_unlock(&vpe_lock[dest_vpe]);
	return 0;
}

irqreturn_t vpe_ipi_intr_handler(int irq, void *dev_id)
{
	int vpe;
	unsigned int temp, ffsbit;

#ifdef CONFIG_MIPS_MT_SMTC
	vpe = cpu_data[smp_processor_id()].vpe_id;
#else
	vpe = smp_processor_id();
#endif
	temp = vpe_device_schedule[vpe];
	ffsbit = 0;

	do {
		if(temp & 0x01) {
			netif_schedule(vpe_devices[vpe][ffsbit]);
			clear_bit(ffsbit, &vpe_device_schedule[vpe]);
		}

		ffsbit++;
		temp >>= 1;

	}while (temp);

	return IRQ_HANDLED;
}

struct irqaction vpe0_ipi;
struct irqaction vpe1_ipi;

static void init_vpe_ipi_framework(void)
{
	unsigned int vpe;
	hmcp_ipi ipi;

	plat_init_vpe_ipi_framework(&ipi);
	ipi_vpe = ipi.hmcp_ipi_vpe;

        vpe0_ipi.handler = vpe_ipi_intr_handler;
        vpe0_ipi.flags = IRQF_DISABLED;
        vpe0_ipi.name = "VPE0_IPI";
        setup_irq(ipi.vpe0_irq, &vpe0_ipi);
	
        vpe1_ipi.handler = vpe_ipi_intr_handler;
        vpe1_ipi.flags = IRQF_DISABLED;
        vpe1_ipi.name = "VPE1_IPI";
        setup_irq(ipi.vpe1_irq, &vpe1_ipi);

	for (vpe = 0; vpe < NUM_VPE; vpe++) {
		spin_lock_init(&vpe_lock[vpe]);
	}
}
#endif

#ifdef CONFIG_MIPS_MT_HMCP_KTHREAD
#include <linux/wait.h>
#include <linux/kthread.h>

struct task_struct *vpe_kthread[NUM_VPE];
wait_queue_head_t vpe_wq[NUM_VPE];

static int vpe_kthread_func(void *bindcpus)
{
	unsigned int temp, ffsbit;
        unsigned int vpe;

#ifdef CONFIG_MIPS_MT_SMTC
	vpe = cpu_data[smp_processor_id()].vpe_id;
#else
	vpe = smp_processor_id();
#endif

	while(1)
	{
		wait_event_interruptible(vpe_wq[vpe], (vpe_device_schedule[vpe] != 0));
		temp = vpe_device_schedule[vpe];
		ffsbit = 0;
		do {
			if(temp & 0x01) {
				netif_schedule(vpe_devices[vpe][ffsbit]);
				clear_bit(ffsbit, &vpe_device_schedule[vpe]);
			}
			ffsbit++;
			temp >>= 1;
		}while (temp);
	}
	return 0;
}

static int inter_vpe_msg_wq(struct net_device* dev, u32 dest_vpe, u32 src_vpe)
{
        if(!test_and_set_bit(dev->vpe_sched_idx, 
				&vpe_device_schedule[dest_vpe])) {
                wake_up_interruptible(&vpe_wq[dest_vpe]);
        }
	spin_unlock(&dev->queue_lock);
	return 0;
}

static void init_vpe_kthread_framework(void)
{
	struct task_struct *p;
	struct sched_param sched;
	int retval;
	int vpe = 0;

	for(vpe = 0; vpe < NUM_VPE; vpe++)
		init_waitqueue_head(&vpe_wq[vpe]);

	for(vpe = 0; vpe < NUM_VPE; vpe++) {

		p = kthread_create(vpe_kthread_func, NULL, "vpe_kthread/%d", vpe); 
		if (IS_ERR(p))
			printk("Creation of Kthread for VPE %d has failed\n", vpe);

		sched.sched_priority = 8;
		retval = sched_setscheduler(p, SCHED_RR, &sched);

		kthread_bind(p, vpe);
		wake_up_process(p);

		vpe_kthread[vpe] = p;
	}
}
#endif

int inter_vpe_msg(struct net_device* dev, u32 dest_vpe, u32 src_vpe)
{
	if (!dev)
		return -1;

#ifdef CONFIG_MIPS_MT_HMCP_IPI
	spin_unlock(&dev->queue_lock);
	if (!test_bit(__LINK_STATE_SCHED, &dev->state))
		inter_vpe_msg_ipi(dev, dest_vpe, src_vpe);
#endif

#ifdef CONFIG_MIPS_MT_HMCP_KTHREAD
	if (!test_bit(__LINK_STATE_SCHED, &dev->state))
		inter_vpe_msg_wq(dev, dest_vpe, src_vpe);
	spin_unlock(&dev->queue_lock);
#endif
	return 0;
}

void hmcp_init(void)
{
#ifdef CONFIG_MIPS_MT_HMCP_IPI
	init_vpe_ipi_framework();
#endif
#ifdef CONFIG_MIPS_MT_HMCP_KTHREAD
	init_vpe_kthread_framework();
#endif
}

