/*
 * Original code based on Host AP (software wireless LAN access point) driver
 * for Intersil Prism2/2.5/3.
 *
 * Copyright (c) 2001-2002, SSH Communications Security Corp and Jouni Malinen
 * <jkmaline@cc.hut.fi>
 * Copyright (c) 2002-2003, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * Adaption to a generic IEEE 802.11 stack by James Ketrenos
 * <jketreno@linux.intel.com>
 *
 * Copyright (c) 2004, Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

/*
 * This file defines the interface to the ieee80211 crypto module.
 */
#ifndef IEEE80211_CRYPT_H
#define IEEE80211_CRYPT_H

#include <linux/types.h>
#include <linux/list.h>
#include <net/ieee80211.h>
#include <asm/atomic.h>

/* Import from Atheros driver src */
/*
 * 802.11 protocol crypto-related definitions.
 */
#define IEEE80211_KEYBUF_SIZE   16
#define IEEE80211_MICBUF_SIZE   (8+8)   /* space for both tx+rx keys */
#define IEEE80211_TID_SIZE  17 /* total number of TIDs */

/*
 * NB: these values are ordered carefully; there are lots of
 * of implications in any reordering.  In particular beware
 * that 4 is not used to avoid conflicting with IEEE80211_F_PRIVACY.
 */
#define IEEE80211_CIPHER_WEP            0
#define IEEE80211_CIPHER_TKIP           1
#define IEEE80211_CIPHER_AES_OCB        2
#define IEEE80211_CIPHER_AES_CCM        3
#define IEEE80211_CIPHER_CKIP           5
#define IEEE80211_CIPHER_NONE           6       /* pseudo value */

#define IEEE80211_CIPHER_MAX            (IEEE80211_CIPHER_NONE+1)

#define IEEE80211_KEYIX_NONE    ((u_int16_t) -1)

/* End import from Atheros driver src */

enum {
	IEEE80211_CRYPTO_TKIP_COUNTERMEASURES = (1 << 0),
};

struct sk_buff;
struct module;

struct ieee80211_crypto_ops {
	const char *name;
	struct list_head list;

	/* init new crypto context (e.g., allocate private data space,
	 * select IV, etc.); returns NULL on failure or pointer to allocated
	 * private data on success */
	void *(*init) (int keyidx);

	/* deinitialize crypto context and free allocated private data */
	void (*deinit) (void *priv);

	int (*build_iv) (struct sk_buff * skb, int hdr_len,
			 u8 *key, int keylen, void *priv);

	/* encrypt/decrypt return < 0 on error or >= 0 on success. The return
	 * value from decrypt_mpdu is passed as the keyidx value for
	 * decrypt_msdu. skb must have enough head and tail room for the
	 * encryption; if not, error will be returned; these functions are
	 * called for all MPDUs (i.e., fragments).
	 */
	int (*encrypt_mpdu) (struct sk_buff * skb, int hdr_len, void *priv);
	int (*decrypt_mpdu) (struct sk_buff * skb, int hdr_len, void *priv);

	/* These functions are called for full MSDUs, i.e. full frames.
	 * These can be NULL if full MSDU operations are not needed. */
	int (*encrypt_msdu) (struct sk_buff * skb, int hdr_len, void *priv);
	int (*decrypt_msdu) (struct sk_buff * skb, int keyidx, int hdr_len,
			     void *priv);

	int (*set_key) (void *key, int len, u8 * seq, void *priv);
	int (*get_key) (void *key, int len, u8 * seq, void *priv);

	/* procfs handler for printing out key information and possible
	 * statistics */
	char *(*print_stats) (char *p, void *priv);

	/* Crypto specific flag get/set for configuration settings */
	unsigned long (*get_flags) (void *priv);
	unsigned long (*set_flags) (unsigned long flags, void *priv);

	/* maximum number of bytes added by encryption; encrypt buf is
	 * allocated with extra_prefix_len bytes, copy of in_buf, and
	 * extra_postfix_len; encrypt need not use all this space, but
	 * the result must start at the beginning of the buffer and correct
	 * length must be returned */
	int extra_mpdu_prefix_len, extra_mpdu_postfix_len;
	int extra_msdu_prefix_len, extra_msdu_postfix_len;

	struct module *owner;
};

struct ieee80211_crypt_data {
	struct list_head list;	/* delayed deletion list */
	struct ieee80211_crypto_ops *ops;
	void *priv;
	atomic_t refcnt;
};

struct ieee80211_device;

int ieee80211_register_crypto_ops(struct ieee80211_crypto_ops *ops);
int ieee80211_unregister_crypto_ops(struct ieee80211_crypto_ops *ops);
struct ieee80211_crypto_ops *ieee80211_get_crypto_ops(const char *name);
void ieee80211_crypt_deinit_entries(struct ieee80211_device *, int);
void ieee80211_crypt_deinit_handler(unsigned long);
void ieee80211_crypt_delayed_deinit(struct ieee80211_device *ieee,
				    struct ieee80211_crypt_data **crypt);
void ieee80211_crypt_quiescing(struct ieee80211_device *ieee);

/* Import from Atheros driver src */
struct ieee80211vap;
struct ieee80211_key;
/*
 * Template for a supported cipher.  Ciphers register with the
 * crypto code and are typically loaded as separate modules
 * (the null cipher is always present).
 * XXX may need refcnts
 */
struct ieee80211_cipher {
        const char *ic_name;            /* printable name */
        u_int   ic_cipher;              /* IEEE80211_CIPHER_* */
        u_int   ic_header;              /* size of privacy header (bytes) */
        u_int   ic_trailer;             /* size of privacy trailer (bytes) */
        u_int   ic_miclen;              /* size of mic trailer (bytes) */
        void*   (*ic_attach)(struct ieee80211vap *, struct ieee80211_key *);
        void    (*ic_detach)(struct ieee80211_key *);
        int     (*ic_setkey)(struct ieee80211_key *);
        int     (*ic_encap)(struct ieee80211_key *, struct sk_buff *,
                        u_int8_t keyid);
        int     (*ic_decap)(struct ieee80211_key *, struct sk_buff *, int);
        int     (*ic_enmic)(struct ieee80211_key *, struct sk_buff *, int);
        int     (*ic_demic)(struct ieee80211_key *, struct sk_buff *, int, int);
};

/*
 * Crypto key state.  There is sufficient room for all supported
 * ciphers (see below).  The underlying ciphers are handled
 * separately through loadable cipher modules that register with
 * the generic crypto support.  A key has a reference to an instance
 * of the cipher; any per-key state is hung off wk_private by the
 * cipher when it is attached.  Ciphers are automatically called
 * to detach and cleanup any such state when the key is deleted.
 *
 * The generic crypto support handles encap/decap of cipher-related
 * frame contents for both hardware- and software-based implementations.
 * A key requiring software crypto support is automatically flagged and
 * the cipher is expected to honor this and do the necessary work.
 * Ciphers such as TKIP may also support mixed hardware/software
 * encrypt/decrypt and MIC processing.
 */
/* XXX need key index typedef */
/* XXX pack better? */
/* XXX 48-bit rsc/tsc */
struct ieee80211_key {
        u_int8_t        wk_keylen;      /* key length in bytes */
        u_int8_t        wk_flags;
#define IEEE80211_KEY_XMIT      0x01    /* key used for xmit */
#define IEEE80211_KEY_RECV      0x02    /* key used for recv */
#define IEEE80211_KEY_GROUP     0x04    /* key used for WPA group operation */
#define IEEE80211_KEY_SWCRYPT   0x10    /* host-based encrypt/decrypt */
#define IEEE80211_KEY_SWMIC     0x20    /* host-based enmic/demic */
        u_int16_t       wk_keyix;       /* key index */
        u_int8_t        wk_key[IEEE80211_KEYBUF_SIZE+IEEE80211_MICBUF_SIZE];
#define wk_txmic        wk_key+IEEE80211_KEYBUF_SIZE+0  /* XXX can't () right */
#define wk_rxmic        wk_key+IEEE80211_KEYBUF_SIZE+8  /* XXX can't () right */
        u_int64_t       wk_keyrsc[IEEE80211_TID_SIZE];  /* key receive sequence counter */
        u_int64_t       wk_keytsc;      /* key transmit sequence counter */
        const struct ieee80211_cipher *wk_cipher;
        void            *wk_private;    /* private cipher state */
};
#define IEEE80211_KEY_COMMON            /* common flags passed in by apps */\
        (IEEE80211_KEY_XMIT | IEEE80211_KEY_RECV | IEEE80211_KEY_GROUP)

/*
 * Reset key state to an unused state.  The crypto
 * key allocation mechanism insures other state (e.g.
 * key data) is properly setup before a key is used.
 */
extern const struct ieee80211_cipher ieee80211_cipher_none;

static __inline void
ieee80211_crypto_resetkey(struct ieee80211vap *vap,
        struct ieee80211_key *k, u_int16_t ix)
{
        k->wk_cipher = &ieee80211_cipher_none;;
        k->wk_private = k->wk_cipher->ic_attach(vap, k);
        k->wk_keyix = ix;
        k->wk_flags = IEEE80211_KEY_XMIT | IEEE80211_KEY_RECV;
}

/*
 * Add any MIC.
 */
static __inline int
ieee80211_crypto_enmic(struct ieee80211vap *vap,
                       struct ieee80211_key *k, struct sk_buff *skb, int force)
{
        const struct ieee80211_cipher *cip = k->wk_cipher;
        return (cip->ic_miclen > 0 ? cip->ic_enmic(k, skb, force) : 1);
}

/*
 * Check and remove any MIC.
 */
static __inline int
ieee80211_crypto_demic(struct ieee80211vap *vap, struct ieee80211_key *k,
        struct sk_buff *skb, int hdrlen, int force)
{
        const struct ieee80211_cipher *cip = k->wk_cipher;
        return (cip->ic_miclen > 0 ? cip->ic_demic(k, skb, hdrlen, force) : 1);
}

/* End import from Atheros driver src  */
#endif
