#ifndef _LINUX_MARKER_H
#define _LINUX_MARKER_H

/*
 * marker.h
 *
 * Code markup for dynamic and static tracing.
 *
 * See Documentation/marker.txt.
 *
 * (C) Copyright 2006 Mathieu Desnoyers <mathieu.desnoyers@polymtl.ca>
 *
 * This file is released under the GPLv2.
 * See the file COPYING for more details.
 */

#ifdef __KERNEL__

typedef void marker_probe_func(const char *fmt, ...);

struct __mark_marker_c {
	const char *name;
	marker_probe_func **call;
	const char *format;
	int flags;
} __attribute__((packed));

struct __mark_marker {
	const struct __mark_marker_c *cmark;
	void *enable;
} __attribute__((packed));

/* Generic marker flavor always available */
#ifdef CONFIG_MARKERS

#define MF_OPTIMIZED 1	/* Use optimized markers */
#define MF_LOCKDEP 2	/* Can call lockdep */
#define MF_PRINTK 3	/* vprintk can be called in the probe */

#define _MF_OPTIMIZED (1 << MF_OPTIMIZED)
#define _MF_LOCKDEP (1 << MF_LOCKDEP)
#define _MF_PRINTK (1 << MF_PRINTK)

#define MARK_GENERIC(flags, name, format, args...) \
	do { \
		static marker_probe_func *__mark_call_##name = \
					__mark_empty_function; \
		static char __marker_enable_##name = 0; \
		static const struct __mark_marker_c __mark_c_##name \
			__attribute__((section(".markers.c"))) = \
			{ #name, &__mark_call_##name, format, \
			(flags) | ~_MF_OPTIMIZED } ; \
		static const struct __mark_marker __mark_##name \
			__attribute__((section(".markers"))) = \
			{ &__mark_c_##name, &__marker_enable_##name } ; \
		asm volatile ( "" : : "i" (&__mark_##name)); \
		__mark_check_format(format, ## args); \
		if (unlikely(__marker_enable_##name)) { \
			preempt_disable(); \
			(*__mark_call_##name)(format, ## args); \
			preempt_enable(); \
		} \
	} while (0)

#define MARK_GENERIC_ENABLE_IMMEDIATE_OFFSET 0
#define MARK_GENERIC_ENABLE_TYPE char
/* Dereference enable as lvalue from a pointer to its instruction */
#define MARK_GENERIC_ENABLE(a) \
	*(MARK_GENERIC_ENABLE_TYPE*) \
		((char*)a+MARK_GENERIC_ENABLE_IMMEDIATE_OFFSET)

static inline int marker_generic_set_enable(void *address, char enable)
{
	MARK_GENERIC_ENABLE(address) = enable;
	return 0;
}

#else /* !CONFIG_MARKERS */
#define MARK_GENERIC(flags, name, format, args...) \
		__mark_check_format(format, ## args)
#endif /* CONFIG_MARKERS */

#ifdef CONFIG_MARKERS_ENABLE_OPTIMIZATION
#include <asm/marker.h>			/* optimized marker flavor */
#else
#include <asm-generic/marker.h>		/* fallback on generic markers */
#endif

#define MARK_MAX_FORMAT_LEN	1024
#define MARK_NOARGS " "

static inline __attribute__ ((format (printf, 1, 2)))
void __mark_check_format(const char *fmt, ...)
{ }

extern marker_probe_func __mark_empty_function;

extern int _marker_set_probe(int flags, const char *name, const char *format,
				marker_probe_func *probe);

#define marker_set_probe(name, format, probe) \
	_marker_set_probe(_MF_DEFAULT, name, format, probe)

extern int marker_remove_probe(marker_probe_func *probe);
extern int marker_list_probe(marker_probe_func *probe);

#endif /* __KERNEL__ */
#endif
