/******************************************************************************
**
** FILE NAME    : amazon_se_sdio_card.h
** PROJECT      : IFX 
** MODULES      : MMC
**
** DATE         : 07 Jan 2008
** AUTHOR       : Reddy Mallikarjuna
** DESCRIPTION  : IFX SD CARD Driver
** COPYRIGHT    :       Copyright (c) 2008
**                      Infineon Technologies AG
**                      Am Campeon 1-12, 85579 Neubiberg, Germany
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
** HISTORY
** $Version $Date      $Author     $Comment
*******************************************************************************/
#ifndef IFX_SD_CARD_H
#define IFX_SD_CARD_H

#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/device.h>

struct mmc_cid {
        unsigned int            manfid;
        char                    prod_name[8];
        unsigned int            serial;
        unsigned short          oemid;
        unsigned short          year;
        unsigned char           hwrev;
        unsigned char           fwrev;
        unsigned char           month;
};

struct mmc_csd {
        unsigned char           mmca_vsn;
        unsigned short          cmdclass;
        unsigned short          tacc_clks;
        unsigned int            tacc_ns;
        unsigned int            r2w_factor;
        unsigned int            max_dtr;
        unsigned int            read_blkbits;
        unsigned int            write_blkbits;
        unsigned int            capacity;
        unsigned int            read_partial:1,
                                read_misalign:1,
                                write_partial:1,
                                write_misalign:1;
};

struct mmc_ext_csd {
        unsigned int            hs_max_dtr;
};

struct sd_scr {
        unsigned char           sda_vsn;
        unsigned char           bus_widths;
#define SD_SCR_BUS_WIDTH_1      (1<<0)
#define SD_SCR_BUS_WIDTH_4      (1<<2)
};

struct sd_switch_caps {
        unsigned int            hs_max_dtr;
};

/*
 * MMC device
 */
struct mmc_card {
        struct list_head        node;           /* node in hosts devices list */
        struct mmc_host         *host;          /* the host this device belongs to */
        struct device           dev;            /* the device */
        unsigned int            rca;            /* relative card address of device */
        unsigned int            state;          /* (our) card state */
#define MMC_STATE_PRESENT       (1<<0)          /* present in sysfs */
#define MMC_STATE_DEAD          (1<<1)          /* device no longer in stack */
#define MMC_STATE_BAD           (1<<2)          /* unrecognised device */
#define MMC_STATE_SDCARD        (1<<3)          /* is an SD card */
#define MMC_STATE_READONLY      (1<<4)          /* card is read-only */
#define MMC_STATE_HIGHSPEED     (1<<5)          /* card is in high speed mode */
        u32                     raw_cid[4];     /* raw card CID */
        u32                     raw_csd[4];     /* raw card CSD */
        u32                     raw_scr[2];     /* raw card SCR */
        struct mmc_cid          cid;            /* card identification */
        struct mmc_csd          csd;            /* card specific */
        struct mmc_ext_csd      ext_csd;        /* mmc v4 extended card specific */
        struct sd_scr           scr;            /* extra SD information */
        struct sd_switch_caps   sw_caps;        /* switch (CMD6) caps */
};

#define mmc_card_present(c)     ((c)->state & MMC_STATE_PRESENT)
#define mmc_card_dead(c)        ((c)->state & MMC_STATE_DEAD)
#define mmc_card_bad(c)         ((c)->state & MMC_STATE_BAD)
#define mmc_card_sd(c)          ((c)->state & MMC_STATE_SDCARD)
#define mmc_card_readonly(c)    ((c)->state & MMC_STATE_READONLY)
#define mmc_card_highspeed(c)   ((c)->state & MMC_STATE_HIGHSPEED)

#define mmc_card_set_present(c) ((c)->state |= MMC_STATE_PRESENT)
#define mmc_card_set_dead(c)    ((c)->state |= MMC_STATE_DEAD)
#define mmc_card_set_bad(c)     ((c)->state |= MMC_STATE_BAD)
#define mmc_card_set_sd(c)      ((c)->state |= MMC_STATE_SDCARD)
#define mmc_card_set_readonly(c) ((c)->state |= MMC_STATE_READONLY)
#define mmc_card_set_highspeed(c) ((c)->state |= MMC_STATE_HIGHSPEED)

#define mmc_card_name(c)        ((c)->cid.prod_name)
#define mmc_card_id(c)          ((c)->dev.bus_id)

#define mmc_list_to_card(l)     container_of(l, struct mmc_card, node)
#define mmc_get_drvdata(c)      dev_get_drvdata(&(c)->dev)
#define mmc_set_drvdata(c,d)    dev_set_drvdata(&(c)->dev, d)
/*
 * MMC device driver (e.g., Flash card, I/O card...)
 */
struct mmc_driver {
        struct device_driver drv;
        int (*probe)(struct mmc_card *);
        void (*remove)(struct mmc_card *);
        int (*suspend)(struct mmc_card *, pm_message_t);
        int (*resume)(struct mmc_card *);
};

extern int ifx_mmc_register_driver(struct mmc_driver *);
extern void ifx_mmc_unregister_driver(struct mmc_driver *);
extern int ifx_register_sdio_driver(struct mmc_driver *);
extern void ifx_unregister_sdio_driver(struct mmc_driver *);

extern int __ifx_mmc_claim_host(struct mmc_host *host, struct mmc_card *card);
static inline int ifx_mmc_card_claim_host(struct mmc_card *card)
{
        return __ifx_mmc_claim_host(card->host, card);
}
#define ifx_mmc_card_release_host(c)        ifx_mmc_release_host((c)->host)

struct request;
struct mmc_data;
struct mmc_request;

struct mmc_command {
        u32                     opcode;
        u32                     arg;
        u32                     resp[4];
        unsigned int            flags;          /* expected response type */
#define MMC_RSP_PRESENT (1 << 0)
#define MMC_RSP_136     (1 << 1)                /* 136 bit response */
#define MMC_RSP_CRC     (1 << 2)                /* expect valid crc */
#define MMC_RSP_BUSY    (1 << 3)                /* card may send busy */
#define MMC_RSP_OPCODE  (1 << 4)                /* response contains opcode */
#define MMC_CMD_MASK    (3 << 5)                /* command type */
#define MMC_CMD_AC      (0 << 5)
#define MMC_CMD_ADTC    (1 << 5)
#define MMC_CMD_BC      (2 << 5)
#define MMC_CMD_BCR     (3 << 5)

/*
 * These are the response types, and correspond to valid bit
 * patterns of the above flags.  One additional valid pattern
 * is all zeros, which means we don't expect a response.
 */
#define MMC_RSP_NONE    (0)
#define MMC_RSP_R1      (MMC_RSP_PRESENT|MMC_RSP_CRC|MMC_RSP_OPCODE)
#define MMC_RSP_R1B     (MMC_RSP_PRESENT|MMC_RSP_CRC|MMC_RSP_OPCODE|MMC_RSP_BUSY)
#define MMC_RSP_R2      (MMC_RSP_PRESENT|MMC_RSP_136|MMC_RSP_CRC)
#define MMC_RSP_R3      (MMC_RSP_PRESENT)
#define MMC_RSP_R6      (MMC_RSP_PRESENT|MMC_RSP_CRC|MMC_RSP_OPCODE)

#define mmc_resp_type(cmd)      ((cmd)->flags & (MMC_RSP_PRESENT|MMC_RSP_136|MMC_RSP_CRC|MMC_RSP_BUSY|MMC_RSP_OPCODE))

/*
 * These are the command types.
 */
#define mmc_cmd_type(cmd)       ((cmd)->flags & MMC_CMD_MASK)

        unsigned int            retries;        /* max number of retries */
        unsigned int            error;          /* command error */

#define MMC_ERR_NONE    0
#define MMC_ERR_TIMEOUT 1
#define MMC_ERR_BADCRC  2
#define MMC_ERR_FIFO    3
#define MMC_ERR_FAILED  4
#define MMC_ERR_INVALID 5
#define MMC_ERR_STARTBIT 6

        struct mmc_data         *data;          /* data segment associated with cmd */
        struct mmc_request      *mrq;           /* associated request */
};

struct mmc_data {
        unsigned int            timeout_ns;     /* data timeout (in ns, max 80ms) */
        unsigned int            timeout_clks;   /* data timeout (in clocks) */
        unsigned int            blksz;          /* data block size */
        unsigned int            blocks;         /* number of blocks */
        unsigned int            error;          /* data error */
        unsigned int            flags;

#define MMC_DATA_WRITE  (1 << 8)
#define MMC_DATA_READ   (1 << 9)
#define MMC_DATA_STREAM (1 << 10)
#define MMC_DATA_MULTI  (1 << 11)

        unsigned int            bytes_xfered;

        struct mmc_command      *stop;          /* stop command */
        struct mmc_request      *mrq;           /* associated request */

        unsigned int            sg_len;         /* size of scatter list */
        struct scatterlist      *sg;            /* I/O scatter list */
};

struct mmc_request {
        struct mmc_command      *cmd;
        struct mmc_data         *data;
        struct mmc_command      *stop;

        void                    *done_data;     /* completion data */
        void                    (*done)(struct mmc_request *);/* completion function */
};


struct mmc_host;
struct mmc_card;

extern int ifx_mmc_wait_for_req(struct mmc_host *, struct mmc_request *);
extern int ifx_mmc_wait_for_cmd(struct mmc_host *, struct mmc_command *, int);
extern int ifx_mmc_wait_for_app_cmd(struct mmc_host *, unsigned int,
        struct mmc_command *, int);

extern void ifx_mmc_set_data_timeout(struct mmc_data *, const struct mmc_card *, int);

static inline void ifx_mmc_claim_host(struct mmc_host *host)
{
        __ifx_mmc_claim_host(host, (struct mmc_card *)-1);
}
extern void ifx_mmc_release_host(struct mmc_host *host);

#endif

