/*****************************************************************************
 **   FILE NAME       : dwc_otg_driver.c
 **   PROJECT         : USB Host and Device driver
 **   MODULES         : USB Host and Device driver
 **   SRC VERSION     : 2.0
 **   DATE            : 1/March/2008
 **   AUTHOR          : Chen, Howard based on Synopsys Original
 **   DESCRIPTION     : The dwc_otg_driver module provides the initialization and cleanup entry
 **                     points for the DWC_otg driver. This module will be dynamically installed
 **                     after Linux is booted using the insmod command. When the module is
 **                     installed, the dwc_otg_driver_init function is called. When the module is
 **                     removed (using rmmod), the dwc_otg_driver_cleanup function is called.
 *
 **                     This module also defines a data structure for the dwc_otg_driver, which is
 **                     used in conjunction with the standard ARM lm_device structure. These
 **                     structures allow the OTG driver to comply with the standard Linux driver
 **                     model in which devices and drivers are registered with a bus driver. This
 **                     has the benefit that Linux can expose attributes of the driver and device
 **                     in its special sysfs file system. Users can then read or write files in
 **                     this file system to perform diagnostics on the driver components or the
 **                     device.
 **   FUNCTIONS       :
 **   COMPILER        : gcc
 **   REFERENCE       :
 **   COPYRIGHT       :
 **  Version Control Section  **
 **   $Author$
 **   $Date$
 **   $Revisions$
 **   $Log$       Revision history
*****************************************************************************/

/*! \file dwc_otg_driver.c
    \brief This file contains the interface to the Linux driver.
*/

#include <linux/version.h>
#include "ifxusb_version.h"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	#include <linux/config.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	#include <linux/device.h>
	#include <linux/platform_device.h>
#endif

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/stat.h>  /* permission constants */

#include <asm/io.h>



#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	#include "dwc_otg_plat.h"
#else
	#include <asm/uaccess.h>
	#include <linux/proc_fs.h>
	#include "dwc_otg_plat.h"
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	#include "dwc_otg_attr.h"
#endif

#include "dwc_otg_driver.h"
#include "dwc_otg_cil.h"
#include "dwc_otg_cil_ifx.h"

#ifdef DWC_IS_DEVICE
#include "dwc_otg_pcd.h" // device
#endif

#ifdef DWC_IS_HOST
#include "dwc_otg_hcd.h"   // host
#endif

#include "dwc_otg_ifx.h" // for Infineon platform specific.

#define	DWC_DRIVER_VERSION	"2.40a 10-APR-2006"
#define	DWC_DRIVER_DESC		"HS OTG USB Controller driver"


#if    defined(__IS_FIRST__)
const char dwc_driver_name[] = "dwc_otg1";
#elif  defined(__IS_SECOND__)
const char dwc_driver_name[] = "dwc_otg2";
#else
const char dwc_driver_name[] = "dwc_otg";
#endif
dwc_otg_device_t *g_dwc_otg_device = NULL;

static unsigned long dwc_iomem_base = IFX_USB_IOMEM_BASE;
int dwc_irq = IFX_USB_IRQ;
int dwc_core_num = IFX_USB_NUM;
int dwc_fifo_base = IFX_USB_FIFOMEM_BASE;
int dwc_fifo_dbg  = IFX_USB_FIFOMEM_DBG;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	extern int ifx_usb_hc_init(unsigned long base_addr, int irq);
	extern void ifx_usb_hc_remove(void);
#endif

/*-------------------------------------------------------------------------*/
/* Encapsulate the module parameter settings */

static dwc_otg_core_params_t dwc_otg_module_params = {
	.opt = -1,
	.dma_burst_size = -1,
	.speed = -1,
	.data_fifo_size = -1,
#ifdef DWC_IS_DEVICE
	.dev_rx_fifo_size = -1,
	.dev_nperio_tx_fifo_size = -1,
#ifdef USE_PERIODIC_EP
	.dev_perio_tx_fifo_size = {-1,   /* dev_perio_tx_fifo_size_1 */
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1,
	                           -1},  /* 15 */
#endif
	.dev_endpoints = -1,
#endif
#ifdef DWC_IS_HOST
	.host_rx_fifo_size = -1,
	.host_nperio_tx_fifo_size = -1,
	.host_perio_tx_fifo_size = -1,
	.host_channels = -1,
#endif
	.max_transfer_size = -1,
	.max_packet_count = -1,
	.phy_utmi_width = -1,
	.turn_around_time_hs = -1,
	.turn_around_time_fs = -1,
	.timeout_cal_hs=-1,
	.timeout_cal_fs=-1,
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	static struct proc_dir_entry * proc_ifx_root = NULL;
	/*
	 * initialize the proc file system and make a dir named /proc/[name]
	 */
	static void ifx_proc_init(void)
	{
		proc_ifx_root = proc_mkdir(dwc_driver_name, (void *)0);
		if (!proc_ifx_root){
			printk("%s proc initialization failed! \n", dwc_driver_name);
			return;
		}
		printk("%s proc initialization okay! \n", dwc_driver_name);
	}
	/*
	 * proc file system add function for debugging.
	 */
	int ifx_proc_addproc(char *funcname, read_proc_t *hookfuncr, write_proc_t *hookfuncw)
	{
		struct proc_dir_entry *pe;

		if (!proc_ifx_root)
			ifx_proc_init();

		if (hookfuncw == NULL){
			pe = create_proc_read_entry(funcname, 0, proc_ifx_root, hookfuncr, NULL);
			if (!pe){
				printk("ERROR in creating read proc entry (%s)! \n", funcname);
				return -1;
			}
		}else{
			pe = create_proc_entry(funcname, S_IRUGO | S_IWUGO, proc_ifx_root);
			if (pe){
				pe->read_proc = hookfuncr;
				pe->write_proc = hookfuncw;
			}else{
				printk("ERROR in creating proc entry (%s)! \n", funcname);
				return -1;
			}
		}
		return 0;
	}
	/*
	 * proc file system del function for removing module.
	 */
	void ifx_proc_delproc(char *funcname)
	{
		char pname[30];
		sprintf(pname, "%s/%s", dwc_driver_name, funcname);
		remove_proc_entry(pname, proc_ifx_root);
	}

#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	/*
	 * This function shows the Driver Version.
	 */
		static ssize_t version_show(struct device_driver *dev, char *buf)
		{
			return snprintf(buf, sizeof(IFXUSB_VERSION)+2,"%s\n",
			           IFXUSB_VERSION);
		}
		static DRIVER_ATTR(version, S_IRUGO, version_show, NULL);
#else // For proc filesystem in linux-2.4.
		static int version_show(char *buf, char **start, off_t offset, int count, int *eof, void *data)
		{
			int len = 0;
			len += snprintf(buf+len, sizeof(IFXUSB_VERSION)+2,"%s\n",
			           IFXUSB_VERSION);
			return len;
		}
#endif

/*
 * Global Debug Level Mask.
 */
uint32_t g_dbg_lvl = 0x00; /* OFF */


#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	/*
	 * This function shows the driver Debug Level.
	 */
	static ssize_t dbg_level_show(struct device_driver *_drv, char *_buf)
	{
		return sprintf(_buf, "0x%0x\n", g_dbg_lvl);
	}
	/*
	 * This function stores the driver Debug Level.
	 */
	static ssize_t dbg_level_store(struct device_driver *_drv, const char *_buf,
	                               size_t _count)
	{
		g_dbg_lvl = simple_strtoul(_buf, NULL, 16);
		return _count;
	}
	static DRIVER_ATTR(debuglevel, S_IRUGO|S_IWUSR, dbg_level_show, dbg_level_store);
#else // For proc filesystem in linux-2.4.
	static int dbg_level_show(char *buf, char **start, off_t offset, int count, int *eof, void *data)
	{
		int len = 0;
		len += sprintf(buf+len, "0x%0x\n", g_dbg_lvl);
		return len;
	}

	static int dbg_level_store(struct file *file, const char *buffer, unsigned long count, void *data)
	{
		char buf[10];
		int i = 0;
		if (copy_from_user(buf, &buffer[i], sizeof(buf)))
			return -EFAULT;
		g_dbg_lvl = simple_strtoul(buf, NULL, 16);
		return count;
	}
#endif

/*
 * This function is called during module intialization to verify that
 * the module parameters are in a valid state.
 */
static int check_parameters(dwc_otg_core_if_t *core_if)
{
	int i;
	int retval = 0;

/* Checks if the parameter is outside of its valid range of values */
#define DWC_OTG_PARAM_TEST(_param_,_low_,_high_) \
	((dwc_otg_module_params._param_ < (_low_)) || \
	 (dwc_otg_module_params._param_ > (_high_)))

/* If the parameter has been set by the user, check that the parameter value is
 * within the value range of values.  If not, report a module error. */
#define DWC_OTG_PARAM_ERR(_param_,_low_,_high_,_string_) \
	do { \
		if (dwc_otg_module_params._param_ != -1) { \
			if (DWC_OTG_PARAM_TEST(_param_,(_low_),(_high_))) { \
				DWC_ERROR("`%d' invalid for parameter `%s'\n", \
					  dwc_otg_module_params._param_, _string_); \
				dwc_otg_module_params._param_ = dwc_param_##_param_##_default; \
				retval ++; \
			} \
		} \
	} while (0)

	DWC_OTG_PARAM_ERR(opt,0,1,"opt");
	DWC_OTG_PARAM_ERR(speed,0,1,"speed");
	DWC_OTG_PARAM_ERR(data_fifo_size,32,32768,"data_fifo_size");

#ifdef DWC_IS_DEVICE
	DWC_OTG_PARAM_ERR(dev_rx_fifo_size,16,32768,"dev_rx_fifo_size");
	DWC_OTG_PARAM_ERR(dev_nperio_tx_fifo_size,16,32768,"dev_nperio_tx_fifo_size");
	DWC_OTG_PARAM_ERR(dev_endpoints,1,15,"dev_endpoints");
#endif
#ifdef DWC_IS_HOST
	DWC_OTG_PARAM_ERR(host_rx_fifo_size,16,32768,"host_rx_fifo_size");
	DWC_OTG_PARAM_ERR(host_nperio_tx_fifo_size,16,32768,"host_nperio_tx_fifo_size");
	DWC_OTG_PARAM_ERR(host_perio_tx_fifo_size,16,32768,"host_perio_tx_fifo_size");
	DWC_OTG_PARAM_ERR(host_channels,1,16,"host_channels");
#endif
	DWC_OTG_PARAM_ERR(max_transfer_size,2047,524288,"max_transfer_size");
	DWC_OTG_PARAM_ERR(max_packet_count,15,511,"max_packet_count");
	DWC_OTG_PARAM_ERR(turn_around_time_hs,1,15,"turn_around_time_hs");
	DWC_OTG_PARAM_ERR(turn_around_time_fs,1,15,"turn_around_time_fs");
	DWC_OTG_PARAM_ERR(timeout_cal_hs,0,7,"timeout_cal_hs");
	DWC_OTG_PARAM_ERR(timeout_cal_fs,0,7,"timeout_cal_fs");

	if (dwc_otg_module_params.dma_burst_size != -1) {
		if (DWC_OTG_PARAM_TEST(dma_burst_size,0,0) &&
		    DWC_OTG_PARAM_TEST(dma_burst_size,1,1) &&
		    DWC_OTG_PARAM_TEST(dma_burst_size,4,4) &&
		    DWC_OTG_PARAM_TEST(dma_burst_size,8,8) &&
		    DWC_OTG_PARAM_TEST(dma_burst_size,16,16))
		{
			DWC_ERROR("`%d' invalid for parameter `dma_burst_size'\n",
				  dwc_otg_module_params.dma_burst_size);
			dwc_otg_module_params.dma_burst_size = 0;
			retval ++;
		}
	}

	if (dwc_otg_module_params.phy_utmi_width != -1) {
		if (DWC_OTG_PARAM_TEST(phy_utmi_width,8,8) &&
		    DWC_OTG_PARAM_TEST(phy_utmi_width,16,16))
		{
			DWC_ERROR("`%d' invalid for parameter `phy_utmi_width'\n",
				  dwc_otg_module_params.phy_utmi_width);
			dwc_otg_module_params.phy_utmi_width = 16;
			retval ++;
		}
	}

#if defined(DWC_IS_DEVICE) && defined(USE_PERIODIC_EP)
	for (i=0; i<15; i++) {
		//DWC_OTG_PARAM_ERR(dev_perio_tx_fifo_size[i],4,768,"dev_perio_tx_fifo_size");
		if (dwc_otg_module_params.dev_perio_tx_fifo_size[i] != -1) {
			if (DWC_OTG_PARAM_TEST(dev_perio_tx_fifo_size[i],4,768)) {
				DWC_ERROR("`%d' invalid for parameter `%s_%d'\n",
					  dwc_otg_module_params.dev_perio_tx_fifo_size[i], "dev_perio_tx_fifo_size", i);
				dwc_otg_module_params.dev_perio_tx_fifo_size[i] = dwc_param_dev_perio_tx_fifo_size_default;
				retval ++;
			}
		}
	}
#endif


	/* At this point, all module parameters that have been set by the user
	 * are valid, and those that have not are left unset.  Now set their
	 * default values and/or check the parameters against the hardware
	 * configurations of the OTG core. */



/* This sets the parameter to the default value if it has not been set by the
 * user */
#define DWC_OTG_PARAM_SET_DEFAULT(_param_) \
	({ \
		int changed = 1; \
		if (dwc_otg_module_params._param_ == -1) { \
			changed = 0; \
			dwc_otg_module_params._param_ = dwc_param_##_param_##_default; \
		} \
		changed; \
	})

/* This checks the macro agains the hardware configuration to see if it is
 * valid.  It is possible that the default value could be invalid.  In this
 * case, it will report a module error if the user touched the parameter.
 * Otherwise it will adjust the value without any error. */
#define DWC_OTG_PARAM_CHECK_VALID(_param_,_str_,_is_valid_,_set_valid_) \
	({ \
		int changed = DWC_OTG_PARAM_SET_DEFAULT(_param_); \
		int error = 0; \
		if (!(_is_valid_)) { \
			if (changed) { \
				DWC_ERROR("`%d' invalid for parameter `%s'.  Check HW configuration.\n", dwc_otg_module_params._param_,_str_); \
				error = 1; \
			} \
			dwc_otg_module_params._param_ = (_set_valid_); \
		} \
		error; \
	})

	retval += DWC_OTG_PARAM_CHECK_VALID(opt,"opt",
		1,
		0);

	DWC_OTG_PARAM_SET_DEFAULT(dma_burst_size);

	retval += DWC_OTG_PARAM_CHECK_VALID(data_fifo_size,
		"data_fifo_size",
		(dwc_otg_module_params.data_fifo_size <= core_if->hwcfg3.b.dfifo_depth),
		core_if->hwcfg3.b.dfifo_depth);

#ifdef DWC_IS_DEVICE
	retval += DWC_OTG_PARAM_CHECK_VALID(dev_rx_fifo_size,
		"dev_rx_fifo_size",
		(dwc_otg_module_params.dev_rx_fifo_size <= dwc_read_reg32(&core_if->core_global_regs->grxfsiz)),
		dwc_read_reg32(&core_if->core_global_regs->grxfsiz));
	retval += DWC_OTG_PARAM_CHECK_VALID(dev_nperio_tx_fifo_size,
		"dev_nperio_tx_fifo_size",
		(dwc_otg_module_params.dev_nperio_tx_fifo_size <= (dwc_read_reg32(&core_if->core_global_regs->gnptxfsiz) >> 16)),
		(dwc_read_reg32(&core_if->core_global_regs->gnptxfsiz) >> 16));
	retval += DWC_OTG_PARAM_CHECK_VALID(dev_endpoints,
		"dev_endpoints",
		(dwc_otg_module_params.dev_endpoints <= (core_if->hwcfg2.b.num_dev_ep)),
		core_if->hwcfg2.b.num_dev_ep);
#endif

#ifdef DWC_IS_HOST
	retval += DWC_OTG_PARAM_CHECK_VALID(host_rx_fifo_size,
		"host_rx_fifo_size",
		(dwc_otg_module_params.host_rx_fifo_size <= dwc_read_reg32(&core_if->core_global_regs->grxfsiz)),
		dwc_read_reg32(&core_if->core_global_regs->grxfsiz));
	retval += DWC_OTG_PARAM_CHECK_VALID(host_nperio_tx_fifo_size,
		"host_nperio_tx_fifo_size",
		(dwc_otg_module_params.host_nperio_tx_fifo_size <= (dwc_read_reg32(&core_if->core_global_regs->gnptxfsiz) >> 16)),
		(dwc_read_reg32(&core_if->core_global_regs->gnptxfsiz) >> 16));
	retval += DWC_OTG_PARAM_CHECK_VALID(host_perio_tx_fifo_size,
		"host_perio_tx_fifo_size",
		(dwc_otg_module_params.host_perio_tx_fifo_size <= ((dwc_read_reg32(&core_if->core_global_regs->hptxfsiz) >> 16))),
		((dwc_read_reg32(&core_if->core_global_regs->hptxfsiz) >> 16)));
	retval += DWC_OTG_PARAM_CHECK_VALID(host_channels,
		"host_channels",
		(dwc_otg_module_params.host_channels <= (core_if->hwcfg2.b.num_host_chan + 1)),
		(core_if->hwcfg2.b.num_host_chan + 1));
#endif

	retval += DWC_OTG_PARAM_CHECK_VALID(max_transfer_size,
		"max_transfer_size",
		(dwc_otg_module_params.max_transfer_size < (1 << (core_if->hwcfg3.b.xfer_size_cntr_width + 11))),
		((1 << (core_if->hwcfg3.b.xfer_size_cntr_width + 11)) - 1));

	retval += DWC_OTG_PARAM_CHECK_VALID(max_packet_count,
		"max_packet_count",
		(dwc_otg_module_params.max_packet_count < (1 << (core_if->hwcfg3.b.packet_size_cntr_width + 4))),
		((1 << (core_if->hwcfg3.b.packet_size_cntr_width + 4)) - 1));

	DWC_OTG_PARAM_SET_DEFAULT(turn_around_time_hs);
	DWC_OTG_PARAM_SET_DEFAULT(turn_around_time_fs);
	DWC_OTG_PARAM_SET_DEFAULT(timeout_cal_hs);
	DWC_OTG_PARAM_SET_DEFAULT(timeout_cal_fs);


/*
 * Define the following to disable the FS PHY Hardware checking.  This is for
 * internal testing only.
 *
 * #define NO_FS_PHY_HW_CHECKS
 */
	retval += DWC_OTG_PARAM_CHECK_VALID(speed,"speed",1,0);

	DWC_OTG_PARAM_SET_DEFAULT(phy_utmi_width);

#if defined(DWC_IS_DEVICE) && defined(USE_PERIODIC_EP)
	for (i=0; i<16; i++) {
		int changed = 1;
		int error = 0;
		if (dwc_otg_module_params.dev_perio_tx_fifo_size[i] == -1) {
			changed = 0;
			dwc_otg_module_params.dev_perio_tx_fifo_size[i] = dwc_param_dev_perio_tx_fifo_size_default;
		}
		if (!(dwc_otg_module_params.dev_perio_tx_fifo_size[i] <= (dwc_read_reg32(&core_if->core_global_regs->dptxfsiz[i])))) {
			if (changed) {
				DWC_ERROR("`%d' invalid for parameter `dev_perio_fifo_size_%d'.  Check HW configuration.\n", dwc_otg_module_params.dev_perio_tx_fifo_size[i],i);
				error = 1;
			}
			dwc_otg_module_params.dev_perio_tx_fifo_size[i] = dwc_read_reg32(&core_if->core_global_regs->dptxfsiz[i]);
		}
		retval += error;
	}
#endif
	return retval;
} // check_parameters


/*
 * This function is the top level interrupt handler for the Common
 * (Device and host modes) interrupts.
 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	static irqreturn_t dwc_otg_common_irq(int _irq, void *_dev)
#else
	static irqreturn_t dwc_otg_common_irq(int _irq, void *_dev, struct pt_regs *_r)
#endif
{
	dwc_otg_device_t *otg_dev = _dev;
	int32_t retval=0;
	retval = dwc_otg_handle_common_intr( otg_dev->core_if );
	mask_and_ack_ifx_irq (_irq);
	return IRQ_RETVAL(retval);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	static unsigned long pri_drvdata = 0;

	/*
	 * This function is called when a DWC_OTG device is registerred and need to get private
	 * device data.
	 *
	 * @return (dwc_otg_device_t *) pri_drvdata.
	 */
	dwc_otg_device_t *dwc_get_drvdata(void) // winder, get device data.
	{
		return (dwc_otg_device_t *)pri_drvdata;
	}

	/*
	 * This function is called when a DWC_OTG device is registerred and need to set private
	 * device data.
	 *
	 * @return
	 */
	void dwc_set_drvdata(dwc_otg_device_t *drvdata) // winder, set device data.
	{
		pri_drvdata = (unsigned long)drvdata;
	}

	/*
	 * This function is called when a DWC_OTG device is registerred and need to get module
	 * parameter data.
	 *
	 * @return (dwc_otg_core_params_t *) dwc_otg_module_params.
	 */
	dwc_otg_core_params_t *dwc_otg_get_module_params(void)
	{
		return (dwc_otg_core_params_t *) &dwc_otg_module_params;
	}
#endif

/*
 * This function is called when a DWC_OTG device is unregistered with the
 * dwc_otg_driver. This happens, for example, when the rmmod command is
 * executed. The device may or may not be electrically present. If it is
 * present, the driver stops device processing. Any resources used on behalf
 * of this device are freed.
 *
 * @return
 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	static int dwc_otg_driver_remove(struct device *_dev)
#else
	static int dwc_otg_driver_remove(void)
#endif
{
	dwc_otg_device_t *otg_dev = dev_get_drvdata(_dev);

	//DWC_DEBUGPL(DBG_ANY, "%s(%p)\n", __func__, _dev);

	if (otg_dev == NULL) {
		/* Memory allocation for the dwc_otg_device failed. */
		return 0;
	}

	/*
	 * Free the IRQ
	 */
	if (otg_dev->common_irq_installed) {
		free_irq( otg_dev->irq, otg_dev );
	}

#ifdef DWC_IS_HOST
	if (otg_dev->hcd != NULL) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
		dwc_otg_hcd_remove(_dev);
#else
		dwc_otg_hcd_remove();
#endif
		free_irq((unsigned int)IFX_USB_OC_IRQ, otg_dev );
	}
#endif

#ifdef DWC_IS_DEVICE
	if (otg_dev->pcd != NULL) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
		dwc_otg_pcd_remove(_dev);
#else
		dwc_otg_pcd_remove();
#endif
	}
#endif
	if (otg_dev->core_if != NULL) {
		dwc_otg_cil_remove( otg_dev->core_if );
	}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	/*
	 * Remove the device attributes
	 */
	dwc_otg_attr_remove(_dev);
#endif
	/*
	 * Return the memory.
	 */
	if (otg_dev->base != NULL) {
		iounmap(otg_dev->base);
	}
	kfree(otg_dev);

	/*
	 * Clear the drvdata pointer.
	 */
		dev_set_drvdata(_dev, 0);
		g_dwc_otg_device = 0;

	return 0;
}

/*
 * This function is called when an DWC_OTG device is bound to a
 * dwc_otg_driver. It creates the driver components required to
 * control the device (CIL, HCD, and PCD) and it initializes the
 * device. The driver components are stored in a dwc_otg_device
 * structure. A reference to the dwc_otg_device is saved in the
 * lm_device. This allows the driver to access the dwc_otg_device
 * structure on subsequent calls to driver methods for this device.
 *
 * @return
 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#if defined(CONFIG_TWINPASS)
#include <asm/ifx/danube_mei_linux.h>
#endif

#if defined(CONFIG_TWINPASS)
#include <asm/danube/irq.h>
#endif

#if defined(CONFIG_DANUBE)
#include <asm/danube/irq.h>
#endif

#if defined(CONFIG_AMAZON_SE)
#include <asm/amazon_se/irq.h>
#endif
#if defined(CONFIG_AMAZON_S)
#include <asm/ifx/irq.h>
#endif
#else
#include <asm/ifx/irq.h>
#include <asm/time.h>
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	static int dwc_otg_driver_probe(struct device *_dev)
#else
	static int dwc_otg_driver_probe(void)
#endif
{
	int retval = 0;
	dwc_otg_device_t *dwc_otg_device;
	int32_t	snpsid;

#ifdef IFX_USB2_IOMEM_BASE
	if     (dwc_core_num==0){
		dwc_irq = IFX_USB1_IRQ;
		dwc_iomem_base = IFX_USB1_IOMEM_BASE;
	}
	else if(dwc_core_num==1){
		dwc_irq = IFX_USB2_IRQ;
		dwc_iomem_base = IFX_USB2_IOMEM_BASE;
	}
	else                    {
		dwc_core_num = IFX_USB_NUM;
		dwc_irq = IFX_USB_IRQ;
		dwc_iomem_base = IFX_USB_IOMEM_BASE;
	}
#endif

#ifndef IFX_USB2_IOMEM_BASE
//	dev_dbg(_dev, "dwc_otg_driver_probe\n");
//	dev_dbg(_dev, "start=0x%08x irq=%d\n", (unsigned int)dwc_iomem_base,dwc_irq);
#else
//	dev_dbg(_dev, "dwc_otg_driver_probe\n");
//	dev_dbg(_dev, "core=%d start=0x%08x irq=%d\n",dwc_core_num, (unsigned int)dwc_iomem_base,dwc_irq);
#endif

	dwc_otg_device = kmalloc(sizeof(dwc_otg_device_t), GFP_KERNEL);
	if (dwc_otg_device == 0) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
		dev_err(_dev, "kmalloc of dwc_otg_device failed\n");
#endif
		retval = -ENOMEM;
		goto fail;
	}
	memset(dwc_otg_device, 0, sizeof(*dwc_otg_device));
	dwc_otg_device->reg_offset = 0xFFFFFFFF;
	dwc_otg_device->fifo = ioremap_nocache(dwc_fifo_base, IFX_USB_FIFOMEM_SIZE);
	dwc_otg_device->fifodbg = ioremap_nocache(dwc_fifo_dbg, IFX_USB_FIFODBG_SIZE);

	/*
	 * Map the DWC_otg Core memory into virtual address space.
	 */
	dwc_otg_device->base = ioremap_nocache(dwc_iomem_base, REGSIZE);
	dwc_otg_device->reg_size = REGSIZE;

	if (dwc_otg_device->base == NULL){
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
		dev_err(_dev, "ioremap() failed\n");
#endif
		retval = -ENOMEM;
		goto fail;
	}
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	dev_dbg(_dev, "base=0x%08x\n", (unsigned)dwc_otg_device->base);
#endif

	/*
	 * Attempt to ensure this device is really a DWC_otg Controller.
	 * Read and verify the SNPSID register contents. The value should be
	 * 0x45F42XXX, which corresponds to "OT2", as in "OTG version 2.XX".
	 */
	snpsid = dwc_read_reg32((uint32_t *)((uint8_t *)dwc_otg_device->base + 0x40));
	if ((snpsid & 0xFFFFF000) != 0x4F542000) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
		dev_err(_dev, "Bad value for SNPSID: 0x%08x\n", snpsid);
#endif
		retval = -EINVAL;
		goto fail;
	}

	/*
	 * Initialize driver data to point to the global DWC_otg
	 * Device structure.
	 */
	dev_set_drvdata(_dev, dwc_otg_device);
	g_dwc_otg_device = dwc_otg_device;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	dev_dbg(_dev, "dwc_otg_device=0x%p\n", dwc_otg_device);
#endif
	dwc_otg_device->core_if =
	   dwc_otg_cil_init( dwc_otg_device->base,dwc_otg_device->fifo,dwc_otg_device->fifodbg,
	                     &dwc_otg_module_params);
	if (dwc_otg_device->core_if == 0) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
		dev_err(_dev, "CIL initialization failed!\n");
#endif
		retval = -ENOMEM;
		goto fail;
	}

	/*
	 * Validate parameter values.
	 */
	if (check_parameters(dwc_otg_device->core_if) != 0) {
		retval = -EINVAL;
		goto fail;
	}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	/*
	 * Create Device Attributes in sysfs
	 */
	dwc_otg_attr_create (_dev);
#endif

	/*
	 * Disable the global interrupt until all the interrupt
	 * handlers are installed.
	 */
	dwc_otg_disable_global_interrupts( dwc_otg_device->core_if );
	/*
	 * Install the interrupt handler for the common interrupts before
	 * enabling common interrupts in core_init below.
	 */
	DWC_DEBUGPL( DBG_CIL, "registering (common) handler for irq%d\n", dwc_irq);
	retval = request_irq((unsigned int)dwc_irq, &dwc_otg_common_irq,
	    SA_INTERRUPT|SA_SHIRQ, "dwc_otg", (void *)dwc_otg_device );
	if (retval != 0) {
		DWC_ERROR("request of irq%d failed\n", dwc_irq);
		retval = -EBUSY;
		goto fail;
	} else {
		dwc_otg_device->common_irq_installed = 1;
		dwc_otg_device->irq = dwc_irq;
	}

	/*
	 * Initialize the DWC_otg core.
	 */
	dwc_otg_core_init( dwc_otg_device->core_if );

#ifdef DWC_IS_DEVICE  // otg device mode. (gadget.)
    /*
     * Initialize the PCD
     */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
    retval = dwc_otg_pcd_init(_dev);
#else // for linux-2.4
    retval = dwc_otg_pcd_init();
#endif
    if (retval != 0) {
        DWC_ERROR("dwc_otg_pcd_init failed\n");
        dwc_otg_device->pcd = NULL;
        goto fail;
    }
#endif // DWC_IS_DEVICE

#ifdef DWC_IS_HOST // otg host mode. (HCD)
	/*
	 * Initialize the HCD
	 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	retval = dwc_otg_hcd_init(_dev);
#else // for linux-2.4
	retval = dwc_otg_hcd_init();
#endif
	if (retval != 0) {
		DWC_ERROR("dwc_otg_hcd_init failed\n");
		dwc_otg_device->hcd = NULL;
		goto fail;
	}

	DWC_DEBUGPL( DBG_CIL, "registering (overcurrent) handler for irq%d\n", dwc_irq);
//    request_irq((unsigned int)IFX_USB_OC_IRQ, &dwc_otg_hcd_oc_irq,
//        SA_INTERRUPT|SA_SHIRQ, "dwc_otg_oc", (void *)dwc_otg_device );

#if defined(CONFIG_TWINPASS)
MEI_ENABLE_IRQ(DANUBE_USB_OC_INT);
#endif

#endif // DWC_IS_HOST

	/*
	 * Enable the global interrupt after all the interrupt
	 * handlers are installed.
	 */
	dwc_otg_enable_global_interrupts( dwc_otg_device->core_if );

	return 0;

fail:
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	dwc_otg_driver_remove(_dev);
#else
	dwc_otg_driver_remove();
#endif
	return retval;
}

/*
 * This structure defines the methods to be called by a bus driver
 * during the lifecycle of a device on that bus. Both drivers and
 * devices are registered with a bus driver. The bus driver matches
 * devices to drivers based on information in the device and driver
 * structures.
 *
 * The probe function is called when the bus driver matches a device
 * to this driver. The remove function is called when a device is
 * unregistered with the bus driver.
 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	static struct device_driver dwc_otg_driver = {
		.name		= dwc_driver_name,
		.bus		= &platform_bus_type,
		.probe		= dwc_otg_driver_probe,
		.remove		= dwc_otg_driver_remove,
	};
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0) && defined(USE_PROC_FS)
	extern void dwc_otg_proc_init(void);
	extern void dwc_otg_proc_clean(void);
#endif
/*
 * This function is called when the dwc_otg_driver is installed with the
 * insmod command. It registers the dwc_otg_driver structure with the
 * appropriate bus driver. This will cause the dwc_otg_driver_probe function
 * to be called. In addition, the bus driver will automatically expose
 * attributes defined for the device and driver in the special sysfs file
 * system.
 *
 * @return
 */

/*!
   \fn int __init dwc_otg_driver_init(void)
   \brief This function is called when the dwc_otg_driver is installed with the
 	insmod command. It registers the dwc_otg_driver structure with the
 	appropriate bus driver. This will cause the dwc_otg_driver_probe function
 	to be called. In addition, the bus driver will automatically expose
 	attributes defined for the device and driver in the special sysfs file
 	system.
   \param void
   \return     IFX_SUCCESS or IFX_FAILURE
*/
int __init dwc_otg_driver_init(void)
{
	int retval = 0;

	dwc_otg_power_on(); // ifx only!!
	dwc_otg_hard_reset();

	printk(KERN_INFO "%s: version %s\n", dwc_driver_name, IFXUSB_VERSION);

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	retval = driver_register(&dwc_otg_driver);
#else
	retval = dwc_otg_driver_probe();
	#if defined(USE_PROC_FS)
		// add proc filesystem.
		ifx_proc_addproc("version", version_show, NULL);
		ifx_proc_addproc("dbg_level", dbg_level_show, dbg_level_store);
		// Winder
		// except version_show and dbg_level, I put proc init to dwc_otg_proc.c.
		dwc_otg_proc_init();
	#endif // USE_PROC_FS
#endif
	if (retval < 0) {
		printk(KERN_ERR "%s retval=%d\n", __func__, retval);
		return retval;
	}
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	retval = ifx_usb_hc_init(dwc_iomem_base, dwc_irq);
	if (retval < 0){
		printk(KERN_ERR "%s retval=%d\n", __func__, retval);
		goto error1;
	}
	retval = driver_create_file(&dwc_otg_driver, &driver_attr_version);
	if (retval < 0){
		printk(KERN_ERR "%s retval=%d\n", __func__, retval);
		goto error2;
	}
	retval = driver_create_file(&dwc_otg_driver, &driver_attr_debuglevel);
	if (retval < 0){
		printk(KERN_ERR "%s retval=%d\n", __func__, retval);
		goto error3;
	}

	return retval;

error3:
	driver_remove_file(&dwc_otg_driver, &driver_attr_version);
error2:
	ifx_usb_hc_remove();
error1:
	driver_unregister(&dwc_otg_driver);
#endif
	return retval;
}
module_init(dwc_otg_driver_init);

/*
 * This function is called when the driver is removed from the kernel
 * with the rmmod command. The driver unregisters itself with its bus
 * driver.
 *
 */
void __exit dwc_otg_driver_cleanup(void)
{
	printk(KERN_DEBUG "dwc_otg_driver_cleanup()\n");

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	driver_remove_file(&dwc_otg_driver, &driver_attr_debuglevel);
	driver_remove_file(&dwc_otg_driver, &driver_attr_version);

    driver_unregister(&dwc_otg_driver);
#ifdef DWC_IS_HOST // otg host mode. (HCD)
    ifx_usb_hc_remove();
#endif
#else // for linux-2.4
	// remove proc filesystem.
#if defined(USE_PROC_FS)
	ifx_proc_delproc("version");
	ifx_proc_delproc("dbg_level");
	dwc_otg_proc_clean();
	remove_proc_entry(dwc_driver_name, 0);
#endif // USE_PROC_FS
	dwc_otg_driver_remove();
#endif

	printk(KERN_INFO "%s module removed\n", dwc_driver_name);
}
module_exit(dwc_otg_driver_cleanup);

MODULE_DESCRIPTION(DWC_DRIVER_DESC);
MODULE_AUTHOR("Synopsys Inc.");
MODULE_LICENSE("GPL");

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
MODULE_PARM (dwc_core_num, "i");
MODULE_PARM_DESC (dwc_core_num, "The USB Core number");
MODULE_PARM(g_dbg_lvl, "i");
MODULE_PARM_DESC(g_dbg_lvl, "Debug level.");
MODULE_PARM (dwc_param_opt, "l");
MODULE_PARM_DESC (dwc_param_opt, "OPT Mode");
MODULE_PARM(dwc_param_dma_burst_size, "l");
MODULE_PARM_DESC(dwc_param_dma_burst_size, "DMA Burst Size 0, 1, 4, 8, 16");
MODULE_PARM(dwc_param_speed, "l");
MODULE_PARM_DESC(dwc_param_speed, "Speed 0=High Speed 1=Full Speed");
MODULE_PARM(dwc_param_data_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_data_fifo_size, "Total number of words in the data FIFO memory 32-32768");
#ifdef DWC_IS_DEVICE
MODULE_PARM(dwc_param_dev_rx_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_dev_rx_fifo_size, "Number of words in the Rx FIFO 16-32768");
MODULE_PARM(dwc_param_dev_nperio_tx_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_dev_nperio_tx_fifo_size, "Number of words in the non-periodic Tx FIFO 16-32768");
#ifdef USE_PERIODIC_EP
MODULE_PARM(dwc_param_dev_perio_tx_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_dev_perio_tx_fifo_size, "Number of words in the periodic Tx FIFO 4-768");
#endif
MODULE_PARM(dwc_param_dev_endpoints, "l");
MODULE_PARM_DESC(dwc_param_dev_endpoints, "The number of endpoints in addition to EP0 available for device mode 1-15");
#endif
#ifdef DWC_IS_HOST
MODULE_PARM(dwc_param_host_rx_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_host_rx_fifo_size, "Number of words in the Rx FIFO 16-32768");
MODULE_PARM(dwc_param_host_nperio_tx_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_host_nperio_tx_fifo_size, "Number of words in the non-periodic Tx FIFO 16-32768");
MODULE_PARM(dwc_param_host_perio_tx_fifo_size, "l");
MODULE_PARM_DESC(dwc_param_host_perio_tx_fifo_size, "Number of words in the host periodic Tx FIFO 16-32768");
MODULE_PARM(dwc_param_host_channels, "l");
MODULE_PARM_DESC(dwc_param_host_channels, "The number of host channel registers to use 1-16");
#endif
MODULE_PARM(dwc_param_max_transfer_size, "l");
MODULE_PARM_DESC(dwc_param_max_transfer_size, "The maximum transfer size supported in bytes 2047-65535");
MODULE_PARM(dwc_param_max_packet_count, "l");
MODULE_PARM_DESC(dwc_param_max_packet_count, "The maximum number of packets in a transfer 15-511");
MODULE_PARM(dwc_param_phy_utmi_width, "l");
MODULE_PARM_DESC(dwc_param_phy_utmi_width, "Specifies the UTMI+ Data Width 8 or 16 bits");
#else // for linux-2.6, winder modified.
module_param_named(opt, dwc_otg_module_params.opt, int, 0444);
MODULE_PARM_DESC(opt, "OPT Mode");
module_param_named(dma_burst_size, dwc_otg_module_params.dma_burst_size, int, 0444);
MODULE_PARM_DESC(dma_burst_size, "DMA Burst Size 0, 1, 4, 8, 16");
module_param_named(speed, dwc_otg_module_params.speed, int, 0444);
MODULE_PARM_DESC(speed, "Speed 0=High Speed 1=Full Speed");
module_param_named(data_fifo_size, dwc_otg_module_params.data_fifo_size, int, 0444);
MODULE_PARM_DESC(data_fifo_size, "Total number of words in the data FIFO memory 32-32768");
#ifdef DWC_IS_DEVICE
module_param_named(dev_rx_fifo_size, dwc_otg_module_params.dev_rx_fifo_size, int, 0444);
MODULE_PARM_DESC(dev_rx_fifo_size, "Number of words in the Rx FIFO 16-32768");
module_param_named(dev_nperio_tx_fifo_size, dwc_otg_module_params.dev_nperio_tx_fifo_size, int, 0444);
MODULE_PARM_DESC(dev_nperio_tx_fifo_size, "Number of words in the non-periodic Tx FIFO 16-32768");
#ifdef USE_PERIODIC_EP
module_param_named(dev_perio_tx_fifo_size_1, dwc_otg_module_params.dev_perio_tx_fifo_size[0], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_1, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_2, dwc_otg_module_params.dev_perio_tx_fifo_size[1], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_2, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_3, dwc_otg_module_params.dev_perio_tx_fifo_size[2], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_3, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_4, dwc_otg_module_params.dev_perio_tx_fifo_size[3], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_4, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_5, dwc_otg_module_params.dev_perio_tx_fifo_size[4], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_5, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_6, dwc_otg_module_params.dev_perio_tx_fifo_size[5], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_6, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_7, dwc_otg_module_params.dev_perio_tx_fifo_size[6], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_7, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_8, dwc_otg_module_params.dev_perio_tx_fifo_size[7], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_8, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_9, dwc_otg_module_params.dev_perio_tx_fifo_size[8], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_9, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_10, dwc_otg_module_params.dev_perio_tx_fifo_size[9], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_10, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_11, dwc_otg_module_params.dev_perio_tx_fifo_size[10], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_11, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_12, dwc_otg_module_params.dev_perio_tx_fifo_size[11], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_12, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_13, dwc_otg_module_params.dev_perio_tx_fifo_size[12], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_13, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_14, dwc_otg_module_params.dev_perio_tx_fifo_size[13], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_14, "Number of words in the periodic Tx FIFO 4-768");
module_param_named(dev_perio_tx_fifo_size_15, dwc_otg_module_params.dev_perio_tx_fifo_size[14], int, 0444);
MODULE_PARM_DESC(dev_perio_tx_fifo_size_15, "Number of words in the periodic Tx FIFO 4-768");
#endif
module_param_named(dev_endpoints, dwc_otg_module_params.dev_endpoints, int, 0444);
MODULE_PARM_DESC(dev_endpoints, "The number of endpoints in addition to EP0 available for device mode 1-15");
#endif
#ifdef DWC_IS_HOST
module_param_named(host_rx_fifo_size, dwc_otg_module_params.host_rx_fifo_size, int, 0444);
MODULE_PARM_DESC(host_rx_fifo_size, "Number of words in the Rx FIFO 16-32768");
module_param_named(host_nperio_tx_fifo_size, dwc_otg_module_params.host_nperio_tx_fifo_size, int, 0444);
MODULE_PARM_DESC(host_nperio_tx_fifo_size, "Number of words in the non-periodic Tx FIFO 16-32768");
module_param_named(host_perio_tx_fifo_size, dwc_otg_module_params.host_perio_tx_fifo_size, int, 0444);
MODULE_PARM_DESC(host_perio_tx_fifo_size, "Number of words in the host periodic Tx FIFO 16-32768");
module_param_named(host_channels, dwc_otg_module_params.host_channels, int, 0444);
MODULE_PARM_DESC(host_channels, "The number of host channel registers to use 1-16");
#endif
module_param_named(max_transfer_size, dwc_otg_module_params.max_transfer_size, int, 0444);
MODULE_PARM_DESC(max_transfer_size, "The maximum transfer size supported in bytes 2047-65535");
module_param_named(max_packet_count, dwc_otg_module_params.max_packet_count, int, 0444);
MODULE_PARM_DESC(max_packet_count, "The maximum number of packets in a transfer 15-511");
module_param_named(phy_utmi_width, dwc_otg_module_params.phy_utmi_width, int, 0444);
MODULE_PARM_DESC(phy_utmi_width, "Specifies the UTMI+ Data Width 8 or 16 bits");
module_param_named(debug, g_dbg_lvl, int, 0444);
MODULE_PARM_DESC(debug, "");
module_param_named (core_num, dwc_core_num, int, 0444);
MODULE_PARM_DESC (dwc_core_num, "The USB Core number");
#endif

/** @page "Module Parameters"
 *
 * The following parameters may be specified when starting the module.
 * These parameters define how the DWC_otg controller should be
 * configured.  Parameter values are passed to the CIL initialization
 * function dwc_otg_cil_init
 *
 * Example: <code>modprobe dwc_otg speed=1 otg_cap=1</code>
 *

 <table>
 <tr><td>Parameter Name</td><td>Meaning</td></tr>

 <tr>
 <td>otg_cap</td>
 <td>Specifies the OTG capabilities. The driver will automatically detect the
 value for this parameter if none is specified.
 - 0: HNP and SRP capable (default, if available)
 - 1: SRP Only capable
 - 2: No HNP/SRP capable
 </td></tr>

 <tr>
 <td>dma_enable</td>
 <td>Specifies whether to use slave or DMA mode for accessing the data FIFOs.
 The driver will automatically detect the value for this parameter if none is
 specified.
 - 0: Slave
 - 1: DMA (default, if available)
 </td></tr>

 <tr>
 <td>dma_burst_size</td>
 <td>The DMA Burst size (applicable only for External DMA Mode).
 - Values: 1, 4, 8 16, 32, 64, 128, 256 (default 32)
 </td></tr>

 <tr>
 <td>speed</td>
 <td>Specifies the maximum speed of operation in host and device mode. The
 actual speed depends on the speed of the attached device and the value of
 phy_type.
 - 0: High Speed (default)
 - 1: Full Speed
 </td></tr>

 <tr>
 <td>host_support_fs_ls_low_power</td>
 <td>Specifies whether low power mode is supported when attached to a Full
 Speed or Low Speed device in host mode.
 - 0: Don't support low power mode (default)
 - 1: Support low power mode
 </td></tr>

 <tr>
 <td>host_ls_low_power_phy_clk</td>
 <td>Specifies the PHY clock rate in low power mode when connected to a Low
 Speed device in host mode. This parameter is applicable only if
 HOST_SUPPORT_FS_LS_LOW_POWER is enabled.
 - 0: 48 MHz (default)
 - 1: 6 MHz
 </td></tr>

 <tr>
 <td>enable_dynamic_fifo</td>
 <td> Specifies whether FIFOs may be resized by the driver software.
 - 0: Use cC FIFO size parameters
 - 1: Allow dynamic FIFO sizing (default)
 </td></tr>

 <tr>
 <td>data_fifo_size</td>
 <td>Total number of 4-byte words in the data FIFO memory. This memory
 includes the Rx FIFO, non-periodic Tx FIFO, and periodic Tx FIFOs.
 - Values: 32 to 32768 (default 8192)

 Note: The total FIFO memory depth in the FPGA configuration is 8192.
 </td></tr>

 <tr>
 <td>dev_rx_fifo_size</td>
 <td>Number of 4-byte words in the Rx FIFO in device mode when dynamic
 FIFO sizing is enabled.
 - Values: 16 to 32768 (default 1064)
 </td></tr>

 <tr>
 <td>dev_nperio_tx_fifo_size</td>
 <td>Number of 4-byte words in the non-periodic Tx FIFO in device mode when
 dynamic FIFO sizing is enabled.
 - Values: 16 to 32768 (default 1024)
 </td></tr>

 <tr>
 <td>dev_perio_tx_fifo_size_n (n = 1 to 15)</td>
 <td>Number of 4-byte words in each of the periodic Tx FIFOs in device mode
 when dynamic FIFO sizing is enabled.
 - Values: 4 to 768 (default 256)
 </td></tr>

 <tr>
 <td>host_rx_fifo_size</td>
 <td>Number of 4-byte words in the Rx FIFO in host mode when dynamic FIFO
 sizing is enabled.
 - Values: 16 to 32768 (default 1024)
 </td></tr>

 <tr>
 <td>host_nperio_tx_fifo_size</td>
 <td>Number of 4-byte words in the non-periodic Tx FIFO in host mode when
 dynamic FIFO sizing is enabled in the core.
 - Values: 16 to 32768 (default 1024)
 </td></tr>

 <tr>
 <td>host_perio_tx_fifo_size</td>
 <td>Number of 4-byte words in the host periodic Tx FIFO when dynamic FIFO
 sizing is enabled.
 - Values: 16 to 32768 (default 1024)
 </td></tr>

 <tr>
 <td>max_transfer_size</td>
 <td>The maximum transfer size supported in bytes.
 - Values: 2047 to 65,535 (default 65,535)
 </td></tr>

 <tr>
 <td>max_packet_count</td>
 <td>The maximum number of packets in a transfer.
 - Values: 15 to 511 (default 511)
 </td></tr>

 <tr>
 <td>host_channels</td>
 <td>The number of host channel registers to use.
 - Values: 1 to 16 (default 12)

 Note: The FPGA configuration supports a maximum of 12 host channels.
 </td></tr>

 <tr>
 <td>dev_endpoints</td>
 <td>The number of endpoints in addition to EP0 available for device mode
 operations.
 - Values: 1 to 15 (default 6 IN and OUT)

 Note: The FPGA configuration supports a maximum of 6 IN and OUT endpoints in
 addition to EP0.
 </td></tr>

 <tr>
 <td>phy_type</td>
 <td>Specifies the type of PHY interface to use. By default, the driver will
 automatically detect the phy_type.
 - 0: Full Speed
 - 1: UTMI+ (default, if available)
 - 2: ULPI
 </td></tr>

 <tr>
 <td>phy_utmi_width</td>
 <td>Specifies the UTMI+ Data Width. This parameter is applicable for a
 phy_type of UTMI+. Also, this parameter is applicable only if the
 OTG_HSPHY_WIDTH cC parameter was set to "8 and 16 bits", meaning that the
 core has been configured to work at either data path width.
 - Values: 8 or 16 bits (default 16)
 </td></tr>

 <tr>
 <td>phy_ulpi_ddr</td>
 <td>Specifies whether the ULPI operates at double or single data rate. This
 parameter is only applicable if phy_type is ULPI.
 - 0: single data rate ULPI interface with 8 bit wide data bus (default)
 - 1: double data rate ULPI interface with 4 bit wide data bus
 </td></tr>

 <tr>
 <td>i2c_enable</td>
 <td>Specifies whether to use the I2C interface for full speed PHY. This
 parameter is only applicable if PHY_TYPE is FS.
 - 0: Disabled (default)
 - 1: Enabled
 </td></tr>

*/


