/* ==========================================================================
 * $File: //dwh/usb_iip/dev/software/otg_ipmate/linux/drivers/dwc_otg_pcd.h $
 * $Revision: #3 $
 * $Date: 2005/09/15 $
 * $Change: 537387 $
 *
 * Synopsys HS OTG Linux Software Driver and documentation (hereinafter,
 * "Software") is an Unsupported proprietary work of Synopsys, Inc. unless
 * otherwise expressly agreed to in writing between Synopsys and you.
 *
 * The Software IS NOT an item of Licensed Software or Licensed Product under
 * any End User Software License Agreement or Agreement for Licensed Product
 * with Synopsys or any supplement thereto. You are permitted to use and
 * redistribute this Software in source and binary forms, with or without
 * modification, provided that redistributions of source code must retain this
 * notice. You may not view, use, disclose, copy or distribute this file or
 * any information contained herein except pursuant to this license grant from
 * Synopsys. If you do not agree with this notice, including the disclaimer
 * below, then you are not authorized to use the Software.
 *
 * THIS SOFTWARE IS BEING DISTRIBUTED BY SYNOPSYS SOLELY ON AN "AS IS" BASIS
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE HEREBY DISCLAIMED. IN NO EVENT SHALL SYNOPSYS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 * ========================================================================== */
#ifdef DWC_IS_DEVICE
#if !defined(__DWC_PCD_H__)
#define __DWC_PCD_H__

#include <linux/types.h>
#include <linux/list.h>
#include <linux/errno.h>

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	#include <linux/device.h>
#else
	#include <linux/init.h>
#endif
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#include <linux/interrupt.h>

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	#include <linux/dma-mapping.h>
#endif

struct lm_device;
struct dwc_otg_device;

#include "dwc_otg_cil.h"

#ifndef container_of
#define container_of list_entry
#endif

/**
 * @file
 *
 * This file contains the structures, constants, and interfaces for
 * the Perpherial Contoller Driver (PCD).
 *
 * The Peripheral Controller Driver (PCD) for Linux will implement the
 * Gadget API, so that the existing Gadget drivers can be used.  For
 * the Mass Storage Function driver the File-backed USB Storage Gadget
 * (FBS) driver will be used.  The FBS driver supports the
 * Control-Bulk (CB), Control-Bulk-Interrupt (CBI), and Bulk-Only
 * transports.
 *
 */

/** Invalid DMA Address */
#define	DMA_ADDR_INVALID	(~(dma_addr_t)0)
/** Maxpacket size for EP0 */
#define	MAX_EP0_SIZE	64
/** Maxpacket size for any EP */
#define MAX_PACKET_SIZE 1024

/**
 * Get the pointer to the core_if from the pcd pointer.
 */
#define GET_CORE_IF( _pcd ) (_pcd->otg_dev->core_if)

/**
 * States of EP0.
 */
typedef enum ep0_state
{
	EP0_DISCONNECT,		/* no host */
	EP0_IDLE,
	EP0_IN_DATA_PHASE,
	EP0_OUT_DATA_PHASE,
	EP0_IN_STATUS,
	EP0_OUT_STATUS,
	EP0_STALL,
} ep0state_e;

/** Fordward declaration.*/
struct dwc_otg_pcd;

/**   PCD EP structure.
 * This structure describes an EP, there is an array of EPs in the PCD
 * structure.
 */
typedef struct dwc_otg_pcd_ep
{
	/** USB EP data */
	struct usb_ep		ep;
	/** USB EP Descriptor */
	const struct usb_endpoint_descriptor	*desc;

	/** queue of dwc_otg_pcd_requests. */
	struct list_head	queue;
	unsigned stopped : 1;
	unsigned disabling : 1;
	unsigned queue_sof : 1;
	unsigned gnpdisabling : 1;

	/** DWC_otg ep data. */
	dwc_ep_t dwc_ep;
	/** Pointer to PCD */
	struct dwc_otg_pcd *pcd;

	//unaligned
#if   defined(GADGET_UNALIGNED_BUFFER_ADJUST)
	uint8_t using_aligned_tx_buf;
	uint8_t using_aligned_rx_buf;
	uint8_t *aligned_tx_buf;
	unsigned aligned_tx_buf_len : 19;
	uint8_t *aligned_rx_buf;
	unsigned aligned_rx_buf_len : 19;
#endif

	/** Count of pending Requests */
	unsigned request_pending;

}dwc_otg_pcd_ep_t;



/* DWC_otg PCD Structure.
 * This structure encapsulates the data for the dwc_otg PCD.
 */

 /*!
   \brief This structure encapsulates the data for the dwc_otg PCD.
 */

typedef struct dwc_otg_pcd
{
	/** USB gadget */
	struct usb_gadget gadget;
	/** USB gadget driver pointer*/
	struct usb_gadget_driver *driver;
	/** The DWC otg device pointer. */
	struct dwc_otg_device *otg_dev;

	/** State of EP0 */
	ep0state_e	ep0state;
	/** EP0 Request is pending */
	unsigned 	ep0_pending : 1;
	/** Indicates when SET CONFIGURATION Request is in process */
	unsigned	request_config : 1;
	/** Count of pending Requests */
	unsigned 	request_pending;

	/** SETUP packet for EP0
	 * This structure is allocated as a DMA buffer on PCD initialization
	 * with enough space for up to 3 setup packets.
	 */
	union
	{
		struct usb_ctrlrequest	req;
		uint32_t	d32[2];
	} *setup_pkt;

	uint32_t	*setup_pkt_buf;


	/** 2-byte dma buffer used to return status from GET_STATUS */
	uint16_t *status_buf;

	/** Array of EPs. */
	dwc_otg_pcd_ep_t ep[ MAX_EPS_CHANNELS ];
	/** number of valid EPs in the above array. */
	unsigned 	num_eps : 4;
	spinlock_t	lock;

	/** Tasklet to defer starting of TEST mode transmissions until
	 *  Status Phase has been completed.
	 */
	struct tasklet_struct  test_mode_tasklet;

	/** Tasklet to delay starting of xfer in DMA mode */
	struct tasklet_struct *start_xfer_tasklet;

	/** The test mode to enter when the tasklet is executed. */
	unsigned test_mode;

	unsigned gnaksource ; // 1:SwitchEP 2:Mismatch 3:Timeout
	unsigned gnaknext   ;

} dwc_otg_pcd_t;


/** DWC_otg request structure.
 * This structure is a list of requests.
 */
typedef struct dwc_otg_pcd_request
{
	struct usb_request	req;   /**< USB Request. */
	struct list_head	queue; /**< queue of these requests. */
	dwc_otg_pcd_ep_t *ep;
#ifdef TxQueueTimer
	int    req_timeout_on;
	struct timer_list	req_timeout;
#endif
	int txing;
	struct list_head	txing_queue;
} dwc_otg_pcd_request_t;


#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	/** @name PCD Create/Destroy Functions */
	/** @{ */
	extern int dwc_otg_pcd_init(struct device *_dev);
	extern void dwc_otg_pcd_remove(struct device *_dev);
	/** @} */
#else // for linux-2.4
	extern int dwc_otg_pcd_init(void);
	extern void dwc_otg_pcd_remove(void);
#endif

extern int32_t dwc_otg_pcd_handle_intr( dwc_otg_pcd_t *_pcd );
//extern void dwc_otg_pcd_start_srp_timer(dwc_otg_pcd_t *_pcd );

//extern void dwc_otg_pcd_initiate_srp(dwc_otg_pcd_t *_pcd);
//extern void dwc_otg_pcd_remote_wakeup(dwc_otg_pcd_t *_pcd, int set);

extern int dwc_otg_pcd_ep_stop(dwc_otg_pcd_ep_t *_ep);

extern void ep0_out_start( dwc_otg_core_if_t *_core_if, dwc_otg_pcd_t *_pcd );

#endif
#endif /* DWC_IS_DEVICE */
