/*****************************************************************************
 **   FILE NAME       : dwc_otg_driver.h
 **   PROJECT         : USB Host and Device driver
 **   MODULES         : USB Host and Device driver
 **   SRC VERSION     : 2.0
 **   DATE            : 1/March/2008
 **   AUTHOR          : Chen, Howard based on Synopsys Original
 **   DESCRIPTION     : The dwc_otg_driver module provides the initialization and cleanup entry
 **                     points for the DWC_otg driver. This module will be dynamically installed
 **                     after Linux is booted using the insmod command. When the module is
 **                     installed, the dwc_otg_driver_init function is called. When the module is
 **                     removed (using rmmod), the dwc_otg_driver_cleanup function is called.
 *
 **                     This module also defines a data structure for the dwc_otg_driver, which is
 **                     used in conjunction with the standard ARM lm_device structure. These
 **                     structures allow the OTG driver to comply with the standard Linux driver
 **                     model in which devices and drivers are registered with a bus driver. This
 **                     has the benefit that Linux can expose attributes of the driver and device
 **                     in its special sysfs file system. Users can then read or write files in
 **                     this file system to perform diagnostics on the driver components or the
 **                     device.
 **   FUNCTIONS       :
 **   COMPILER        : gcc
 **   REFERENCE       :
 **   COPYRIGHT       :
 **  Version Control Section  **
 **   $Author$
 **   $Date$
 **   $Revisions$
 **   $Log$       Revision history
*****************************************************************************/

/*! \file dwc_otg_driver.h
    \brief This file contains the interface to the Linux driver.
*/

/*---------- First Level Grouping ---------------------*/
/** \ingroup USB_DRIVER
	\defgroup USB_DRIVER_DRIVER Driver for Linux
	\brief The dwc_otg_driver module provides the initialization and cleanup entry
 points for the DWC_otg driver. This module will be dynamically installed
 after Linux is booted using the insmod command. When the module is
 installed, the dwc_otg_driver_init function is called. When the module is
 removed (using rmmod), the dwc_otg_driver_cleanup function is called.

 This module also defines a data structure for the dwc_otg_driver, which is
 used in conjunction with the standard ARM lm_device structure. These
 structures allow the OTG driver to comply with the standard Linux driver
 model in which devices and drivers are registered with a bus driver. This
 has the benefit that Linux can expose attributes of the driver and device
 in its special sysfs file system. Users can then read or write files in
 this file system to perform diagnostics on the driver components or the
 device.
 */

/* @{ */

#if !defined(__DWC_OTG_DRIVER_H__)
#define __DWC_OTG_DRIVER_H__

#include "dwc_otg_cil.h"

/* Type declarations */
struct dwc_otg_pcd;
struct dwc_otg_hcd;

/**
 * \brief This structure is a wrapper that encapsulates the driver components used to
 * manage a single DWC_otg controller.
 */
typedef struct dwc_otg_device
{
	/** Base address returned from ioremap() */
	void *base;
	void *fifo;
	void *fifodbg;

	/** irq number and register size. */
	int irq;
	int reg_size;

	/** Pointer to the core interface structure. */
	dwc_otg_core_if_t *core_if;

	/** Register offset for Diagnostic API.*/
	uint32_t reg_offset;

	/** Pointer to the PCD structure. */
	struct dwc_otg_pcd *pcd;

	/** Pointer to the HCD structure. */
	struct dwc_otg_hcd *hcd;

	/** Flag to indicate whether the common IRQ handler is installed. */
	uint8_t common_irq_installed;

} dwc_otg_device_t;

/*--------------------------------------------------------------------------*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	#if DEBUG
		#define dev_dbg(fake, format, arg...) printk(KERN_DEBUG __FILE__ ":%d: " format "\n" , __LINE__, ## arg)
	#else // DEBUG
		#define dev_dbg(fake, format, arg...)
	#endif // DEBUG
	#define dev_err(fake, format, arg...) printk(KERN_ERR __FILE__ ": " format "\n" , ## arg)
#endif
/*--------------------------------------------------------------------------*/

/** \ingroup USB_DRIVER
 * \brief This function is called when a DWC_OTG device is unregistered with the
 * dwc_otg_driver. This happens, for example, when the rmmod command is
 * executed. The device may or may not be electrically present. If it is
 * present, the driver stops device processing. Any resources used on behalf
 * of this device are freed.
 *
 * \return
 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	int dwc_otg_driver_remove(struct device *_dev);
#else
	int dwc_otg_driver_remove(void);
#endif

/** \ingroup USB_DRIVER
 * \brief This function is called when an DWC_OTG device is bound to a
 * dwc_otg_driver. It creates the driver components required to
 * control the device (CIL, HCD, and PCD) and it initializes the
 * device. The driver components are stored in a dwc_otg_device
 * structure. A reference to the dwc_otg_device is saved in the
 * lm_device. This allows the driver to access the dwc_otg_device
 * structure on subsequent calls to driver methods for this device.
 *
 * \return
 */

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	int dwc_otg_driver_probe(struct device *_dev);
#else
	int dwc_otg_driver_probe(void);
#endif

/** \ingroup USB_DRIVER
 * \brief This function is called when the dwc_otg_driver is installed with the
 * insmod command. It registers the dwc_otg_driver structure with the
 * appropriate bus driver. This will cause the dwc_otg_driver_probe function
 * to be called. In addition, the bus driver will automatically expose
 * attributes defined for the device and driver in the special sysfs file
 * system.
 *
 * \return
 */
//static int __init dwc_otg_driver_init(void);
static int dwc_otg_driver_init(void);

/**
 * \brief This function is called when the driver is removed from the kernel
 * with the rmmod command. The driver unregisters itself with its bus
 * driver.
 */
//static void __exit dwc_otg_driver_cleanup(void);
static void dwc_otg_driver_cleanup(void);

/* @} */ /* USB_DRIVER_DRIVER */




#endif
