/******************************************************************************
**
** FILE NAME    : amazon_se_sw.c
** PROJECT      : Amazon SE 
** MODULES      : ETH Interface (MII0)
**
** DATE         : 21 SEP 2006
** AUTHOR       : Reddy Mallikarjuna 
** DESCRIPTION  : ETH Interface (MII0) Driver
** COPYRIGHT    :       Copyright (c) 2006
**                      Infineon Technologies AG
**                      Am Campeon 1-12, 85579 Neubiberg, Germany
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
** HISTORY
** $Date        $Author         	$Comment
** 21 SEP 2006  Reddy Mallikarjuna 	Initiate Version
** 14 May 2007  Reddy Mallikarjuna      Jumbo packet length fixed, Set oversize frame length to 1500Bytes. 
** 30 May 2007  Reddy Mallikarjuna	CMV fail fixed, Reset PPE module.
** 05 June 2007	Reddy Mallikarjuna	Fixed, link Status information to the upper layer.
** 08 June 2007 Reddy Mallikarjuna	Set Class A value to EPHY 100M control register. 
** 14 June 2007 Reddy Mallikarjuna	Set GPIO settings for MDIO & MDC to fix Reduced MII mode support
** 01 April 2009 Huang Xiaogang         Add amazon_se_sw_config function for PTM module mode
*******************************************************************************/

#include <linux/module.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/interrupt.h>	/* mark_bh */
#include <linux/mii.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/netdevice.h>	/* struct device, and other headers */
#include <linux/etherdevice.h>	/* eth_type_trans */
#include <linux/ip.h>		/* struct iphdr */
#include <linux/tcp.h>		/* struct tcphdr */
#include <linux/skbuff.h>
#include <linux/in6.h>
#include <linux/proc_fs.h>
#include <linux/mm.h>
#include <linux/ethtool.h>
#include <asm/checksum.h>
#include <linux/init.h>
#include <asm/delay.h>

#include <asm/amazon_se/amazon_se.h>
#include <asm/amazon_se/amazon_se_sw.h>
#include <asm/amazon_se/amazon_se_dma.h>

//#define AMAZON_SE_SW_DUMP
#undef AMAZON_SE_SW_DUMP

#undef ENABLE_TRACE
//#define  ENABLE_TRACE

#ifdef ENABLE_TRACE
#define TRACE(fmt,args...) printk("%s: " fmt, __FUNCTION__ , ##args)
#else
#define TRACE(fmt,args...)
#endif

#define AMAZON_SE_SW_DMSG(fmt, args...) printk( KERN_ERR  "%s: " fmt,__FUNCTION__, ## args)

#define AMAZON_SE_SW_INT_NO 1
#define ETHERNET_PACKET_DMA_BUFFER_SIZE 1536
#define MDIO_DELAY 1		/*mdio negotiation time */


#define IFX_SUCCESS 1
#define IFX_ERROR   -1

#define LOOPBACK_TEST 0

#define SET_GPIO	1	//1, for gpio, 0 for dedicated

#define EPHY_CLOCK	1	//1 External clock, 0 internal clock

//808261:<IFTW-fchang>.modified Detect the IFX_CONFIG_FLASH_TYPE and enable the right setting on the new ASE WLAN board
//805061:<IFTW-fchang>.removed Change the GPIO settings to 21,22,23,24 for RMII to support the new Amazon WLAN board
#ifndef CONFIG_MTD_AMAZON_SE_SPI_FLASH
	#define  RED_MII_MUX_WITH_SPI  1
#endif

#define 	SET_CLASS_A_VALUE 1

#define MII_MODE 	1
#define REV_MII_MODE 	2
#define  RED_MII_MODE    3
#define  EPHY_MODE      4


#define DPLUS_RXDB          ((volatile u32 *)(AMAZON_SE_PPE32_DATA_MEM_MAP_REG_BASE + (0x0710 * 4)))
#define DPLUS_RXCB              ((volatile u32 *)(AMAZON_SE_PPE32_DATA_MEM_MAP_REG_BASE + (0x0711 * 4)))
#define DPLUS_RXCFG               ((volatile u32 *)(AMAZON_SE_PPE32_DATA_MEM_MAP_REG_BASE + (0x0712 * 4)))




int g_ethernet_mode = 0;

/*ETOP_MDIO_ACCESS MASKS*/
#define EPHY_RA_MASK    0x80000000
#define EPHY_RW_MASK    0x40000000	//Mask

#define EPHY_WRITE_MASK	0x00000000
#define READ_MASK	0x40000000
#define EPHY_ADDRESS	0x08		
#define EPHY_ADDRESS_SHIFT	21
#define EPHY_REG_ADDRESS_SHIFT	16

#define RA_READ_ENABLE	0x3	//MDIO Request/Ack and Read
#define RA_WRITE_ENABLE	0x2	//MDIO Req/Ack and Write

#define EPHY_ADDRESS_MASK	0x1F
#define EPHY_REG_MASK	0x1F
#define EPHY_DATA_MASK	0xFFFF

#define EPHY_RESET                        0x8000
#define EPHY_AUTO_NEGOTIATION_ENABLE          0x1000
#define AUTO_NEGOTIATION_COMPLETE          0x20
#define EPHY_RESTART_AUTO_NEGOTIATION          0x200

#define EPHY_MDIO_BASE_CONTROL_REG               0x0	//PHY Control Register
#define EPHY_LINK_SPEED_MASK	0x2000
#define EPHY_AUTO_NEG_ENABLE_MASK	0x1000
#define EPHY_DUPLEX_MASK	0x0100



#define EPHY_MDIO_BASE_STATUS_REG		0x01	//PHY status register
#define EPHY_LINK_STATUS_MASK	0x02


#define EPHY_MDIO_ADVERTISMENT_REG               0x4	//Auto Negotiation Advertisement Register
#define EPHY_MDIO_ADVERT_100_FD                0x100
#define EPHY_MDIO_ADVERT_100_HD                0x080
#define EPHY_MDIO_ADVERT_10_FD                 0x040
#define EPHY_MDIO_ADVERT_10_HD                 0x020
#define EPHY_MDIO_BC_NEGOTIATE                0x0200


#define EPHY_SPECIFIC_STATUS_REG               0x17	//PHY specific status register
#define EPHY_SPECIFIC_STATUS_SPEED_MASK	0x020
#define EPHY_SPECIFIC_STATUS_DUPLEX_MASK	0x040
#define EPHY_SPECIFIC_STATUS_LINK_UP	0x010
#define	LINK_UP		0x01;
#define LINK_DOWN	0x00

#define EPHY_SET_CLASS_VALUE_REG	0x12	

/*ENET_MAC_CFG MASKS*/
#define BP_MASK     1<<12
#define CGEN_MASK   1<<11
#define IFG_MASK    0x3F<<5
#define IPAUS_MASK  1<<4
#define EPAUS_MASK  1<<3
#define DUPLEX_MASK 1<<2
#define SPEED_MASK  0x2
#define LINK_MASK   1



/***************************************** Functions  Declaration *********************************/
static int switch_init (struct net_device *dev);
void switch_tx_timeout (struct net_device *dev);
int switch_tx (struct sk_buff *skb, struct net_device *dev);
int switch_ioctl (struct net_device *dev, struct ifreq *ifr, int cmd);
struct net_device_stats *switch_stats (struct net_device *dev);
int switch_change_mtu (struct net_device *dev, int new_mtu);
int switch_set_mac_address (struct net_device *dev, void *p);
void switch_tx_timeout (struct net_device *dev);
static int probe_transceiver (struct net_device *dev);
void open_rx_dma (struct net_device *dev);
void close_rx_dma (struct net_device *dev);
void amazon_xon (struct net_device *dev);
int init_dma_device (_dma_device_info * dma_dev);


int ephy_auto_negotiate (int phy_addr);
u32 ephy_read_mdio_reg (int phy_addr, int phy_reg_num);
int ephy_write_mdio_reg (int phy_addr, int phy_reg_num, u32 value);
void negotiate (struct net_device *dev);
/***************************************** Global Data *******************************************/
static struct net_device switch_devs[2] = {
      {init:switch_init,},
      {init:switch_init,}
};


//060620:henryhsu modify for vlan
#if defined(CONFIG_IFX_NFEXT_AMAZON_SE_SWITCH_PHYPORT) || defined(CONFIG_IFX_NFEXT_AMAZON_SE_SWITCH_PHYPORT_MODULE)
#define AMAZON_SW_VLAN_BRIDGING_SUPPORT
int (*amazon_sw_phyport_rx) (struct net_device * dev, struct sk_buff * skb) =
	NULL;
EXPORT_SYMBOL (amazon_sw_phyport_rx);
struct net_device *amazon_sw_for_phyport = &(switch_devs[0]);
EXPORT_SYMBOL (amazon_sw_for_phyport);
#endif


struct proc_dir_entry *g_amazon_se_etop_dir;
static int timeout = 10 * HZ;
static int rx_stopped = 0;
static int g_transmit = 0;
/*************************************************************************************************/


void
ephy_set_duplex (struct net_device *dev, enum duplex new_duplex)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;
	if (new_duplex != priv->current_duplex) {
		priv->current_duplex = new_duplex;
		negotiate (dev);

	}

}

void
ephy_set_speed (struct net_device *dev, unsigned long speed)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;
	priv->current_speed_selection = speed;
	negotiate (dev);
}

void
negotiate (struct net_device *dev)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;
	unsigned short data =
		ephy_read_mdio_reg (priv->mdio_phy_addr,
				    EPHY_MDIO_ADVERTISMENT_REG);

	/* Discard old speed and duplex settings */
	data &= ~(EPHY_MDIO_ADVERT_100_HD | EPHY_MDIO_ADVERT_100_FD |
		  EPHY_MDIO_ADVERT_10_FD | EPHY_MDIO_ADVERT_10_HD);

	switch (priv->current_speed_selection) {
	case 10:
		if (priv->current_duplex == full)
			data |= EPHY_MDIO_ADVERT_10_FD;
		else if (priv->current_duplex == half)
			data |= EPHY_MDIO_ADVERT_10_HD;
		else
			data |= EPHY_MDIO_ADVERT_10_HD |
				EPHY_MDIO_ADVERT_10_FD;
		break;

	case 100:
		if (priv->current_duplex == full)
			data |= EPHY_MDIO_ADVERT_100_FD;
		else if (priv->current_duplex == half)
			data |= EPHY_MDIO_ADVERT_100_HD;
		else
			data |= EPHY_MDIO_ADVERT_100_HD |
				EPHY_MDIO_ADVERT_100_FD;
		break;

	case 0:		/* Auto */
		if (priv->current_duplex == full)
			data |= EPHY_MDIO_ADVERT_100_FD |
				EPHY_MDIO_ADVERT_10_FD;
		else if (priv->current_duplex == half)
			data |= EPHY_MDIO_ADVERT_100_HD |
				EPHY_MDIO_ADVERT_10_HD;
		else
			data |= EPHY_MDIO_ADVERT_100_HD |
				EPHY_MDIO_ADVERT_100_FD |
				EPHY_MDIO_ADVERT_10_FD |
				EPHY_MDIO_ADVERT_10_HD;
		break;

	default:		/* assume autoneg speed and duplex */
		data |= EPHY_MDIO_ADVERT_100_HD | EPHY_MDIO_ADVERT_100_FD |
			EPHY_MDIO_ADVERT_10_FD | EPHY_MDIO_ADVERT_10_HD;
	}

	ephy_write_mdio_reg (priv->mdio_phy_addr, EPHY_MDIO_ADVERTISMENT_REG,
			     data);

	/* Renegotiate with link partner */

	data = ephy_read_mdio_reg (priv->mdio_phy_addr,
				   EPHY_MDIO_BASE_CONTROL_REG);
	data |= EPHY_MDIO_BC_NEGOTIATE;

	ephy_write_mdio_reg (priv->mdio_phy_addr, EPHY_MDIO_BASE_CONTROL_REG,
			     data);

}


u32
ephy_read_mdio_reg (int phy_addr, int phy_reg_num)
{
	u32 phyRegVal;

	AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) =
		(RA_READ_ENABLE << 30) | ((phy_addr & EPHY_ADDRESS_MASK) <<
					  EPHY_ADDRESS_SHIFT) | ((phy_reg_num
								  &
								  EPHY_REG_MASK)
								 <<
								 EPHY_REG_ADDRESS_SHIFT);

	while (AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) & 0x80000000);	//fix me ...

	phyRegVal = AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) & EPHY_DATA_MASK;
	return phyRegVal;
}

int
ephy_write_mdio_reg (int phy_addr, int phy_reg_num, u32 phyData)
{

	AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) =
		(RA_WRITE_ENABLE << 30) | ((phy_addr & EPHY_ADDRESS_MASK) <<
					   EPHY_ADDRESS_SHIFT) | ((phy_reg_num
								   &
								   EPHY_REG_MASK)
								  <<
								  EPHY_REG_ADDRESS_SHIFT)
		| (phyData & EPHY_DATA_MASK);

	while (AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) & 0x80000000);	//fix me .. 

	return OK;
}


int
ephy_auto_negotiate (int phy_addr)
{
	u32 phyRegVal = 0;
	phyRegVal = ephy_read_mdio_reg (phy_addr, EPHY_MDIO_BASE_CONTROL_REG);
	ephy_write_mdio_reg (phy_addr, EPHY_MDIO_BASE_CONTROL_REG,
			     (phyRegVal | EPHY_RESTART_AUTO_NEGOTIATION |
			      EPHY_AUTO_NEGOTIATION_ENABLE | EPHY_RESET));

	return OK;
}



#ifdef AMAZON_SE_SW_DUMP
/*
 * Brief:	dump skb data
 */
static inline void
dump_skb (u32 len, char *data)
{
	int i;
	for (i = 0; i < len; i++) {
		printk ("%2.2x ", (u8) (data[i]));
		if (i % 16 == 15)
			printk ("\n");
	}
	printk ("\n");
}
#endif


static int
probe_transceiver (struct net_device *dev)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;



	if (g_ethernet_mode == EPHY_MODE)
		priv->mdio_phy_addr = EPHY_ADDRESS;	//0x08
	else
		priv->mdio_phy_addr = PHY0_ADDR;

	if (!(g_ethernet_mode == EPHY_MODE)) {
		*(ENET_MAC_DA0) =
			(dev->dev_addr[0] << 24) | (dev->dev_addr[1] << 16)
			| (dev->dev_addr[2] << 8) | (dev->dev_addr[3]);
		*(ENET_MAC_DA1) =
			(dev->dev_addr[4] << 24) | (dev->dev_addr[5] << 16);
		if (!(*(ENET_MAC_CFG) & LINK_MASK)) {

			/*enable the mdio state machine and reset */
			*ETOP_MDIO_CFG |=
				(*ETOP_MDIO_CFG &
				 ~(PHYA0_MASK | UMM0_MASK | SMRST_MASK))
				| PHY0_ADDR << 3 | 1 << 1 | 1 << 13;

			udelay (MDIO_DELAY);	/*wait for some time?, fix me */
			if (!(*(ENET_MAC_CFG) & LINK_MASK))
				return IFX_ERROR;
		}
	}
	return IFX_SUCCESS;
}



unsigned char g_my_ethaddr[MAX_ADDR_LEN];

/* need to get the ether addr from u-boot */
static int __init
ethaddr_setup (char *line)
{
	char *ep;
	int i;

	memset (g_my_ethaddr, 0, MAX_ADDR_LEN);
	/* there should really be routines to do this stuff */
	for (i = 0; i < 6; i++) {
		g_my_ethaddr[i] = line ? simple_strtoul (line, &ep, 16) : 0;
		if (line)
			line = (*ep) ? ep + 1 : ep;
	}
	AMAZON_SE_SW_DMSG ("mac address %2x-%2x-%2x-%2x-%2x-%2x \n",
			   g_my_ethaddr[0]
			   , g_my_ethaddr[1]
			   , g_my_ethaddr[2]
			   , g_my_ethaddr[3]
			   , g_my_ethaddr[4]
			   , g_my_ethaddr[5]);
	return 0;
}

__setup ("ethaddr=", ethaddr_setup);

/* Brief: open RX DMA  channels
 * Parameter: net_device
 */
void
open_rx_dma (struct net_device *dev)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;
	struct dma_device_info *dma_dev = priv->dma_device;
	int i;

	for (i = 0; i < dma_dev->max_rx_chan_num; i++) {
		if ((dma_dev->rx_chan[i])->control == AMAZON_SE_DMA_CH_ON)
			(dma_dev->rx_chan[i])->open (dma_dev->rx_chan[i]);
	}

}

/* Brief: close RX DMA  channels
 * Parameter: net_device
 */
void
close_rx_dma (struct net_device *dev)
{

	struct switch_priv *priv = (struct switch_priv *) dev->priv;
	struct dma_device_info *dma_dev = priv->dma_device;
	int i;

	for (i = 0; i < dma_dev->max_rx_chan_num; i++)
		dma_dev->rx_chan[i]->close (dma_dev->rx_chan[i]);


}

#ifdef CONFIG_NET_HW_FLOWCONTROL
/* Brief: Enable reciving DMA channel */
void
amazon_se_xon (struct net_device *dev)
{
	unsigned long flag;
	local_irq_save (flag);
	TRACE ("wakeup\n");
	open_rx_dma (dev);
	local_irq_restore (flag);
}
#endif //CONFIG_NET_HW_FLOWCONTROL

int
amazon_se_switch_open (struct net_device *dev)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;

#if 0
	if (probe_transceiver (dev) == IFX_ERROR) {
		TRACE ("%s cannot work because of hardware problem\n",
		       dev->name);
		MOD_DEC_USE_COUNT;
		return -1;
	}
	TRACE ("%s\n", dev->name);
#endif
	open_rx_dma (dev);	//TX is opened once there is an outgoing packet

#if !(LOOPBACK_TEST)		//VENUS

#ifdef CONFIG_NET_HW_FLOWCONTROL
	if ((priv->fc_bit = netdev_register_fc (dev, amazon_se_xon)) == 0) {
		TRACE ("Hardware Flow Control register fails\n");
	}
#endif //CONFIG_NET_HW_FLOWCONTROL
	netif_start_queue (dev);
#endif

	return OK;
}


int
switch_release (struct net_device *dev)
{

	struct switch_priv *priv = (struct switch_priv *) dev->priv;
	struct dma_device_info *dma_dev = priv->dma_device;

	//dma_device_release(dma_dev);  
	close_rx_dma (dev);

#if !(LOOPBACK_TEST)		//VENUS

#ifdef CONFIG_NET_HW_FLOWCONTROL
	if (priv->fc_bit) {
		netdev_unregister_fc (priv->fc_bit);
	}
#endif //CONFIG_NET_HW_FLOWCONTROL
	netif_stop_queue (dev);
#endif


	return OK;
}


int
switch_proc_read (char *buf, char **start, off_t offset,
		  int count, int *eof, void *data)
{
	*eof = 1;
	return (int) offset + 4;
}

int
switch_rx (struct net_device *dev, int len, struct sk_buff *skb)
{
	int ret = 0;
	struct switch_priv *priv = (struct switch_priv *) dev->priv;
#ifdef CONFIG_NET_HW_FLOWCONTROL
	int mit_sel = 0;
#endif

//060620:henryhsu modify for vlan
#if defined(CONFIG_IFX_NFEXT_AMAZON_SE_SWITCH_PHYPORT) || defined(CONFIG_IFX_NFEXT_AMAZON_SE_SWITCH_PHYPORT_MODULE)


	if (((dev - switch_devs) == 0) && amazon_sw_phyport_rx) {

		if (amazon_sw_phyport_rx (dev, skb) < 0) {
			priv->stats.rx_errors++;
			return;
		}
	}
	else
#endif
	{
		skb->dev = dev;
		/*  Protocol ID */
		skb->protocol = eth_type_trans (skb, dev);
	}

#ifdef CONFIG_NET_HW_FLOWCONTROL
	mit_sel = netif_rx (skb);
	switch (mit_sel) {
	case NET_RX_SUCCESS:
	case NET_RX_CN_LOW:
	case NET_RX_CN_MOD:
		ret = len;
		break;
	case NET_RX_CN_HIGH:
		break;
	case NET_RX_DROP:
		if ((priv->fc_bit)
		    && (!test_and_set_bit (priv->fc_bit, &netdev_fc_xoff))) {
			close_rx_dma (dev);
			TRACE ("RX STOPPED!\n");
			rx_stopped = 1;

		}
		ret = 0;
		break;
	}
#else
	netif_rx (skb);
#endif

	priv->stats.rx_packets++;
	priv->stats.rx_bytes += len;
	//printk("packet %d\n",priv->stats.rx_packets);
	return ret;
}



int
switch_hw_receive (struct net_device *dev, struct dma_device_info *dma_dev)
{
	u8 *buf = NULL;
	int len = 0;

#if LOOPBACK_TEST		//VENUS
	struct ethhdr *eth;
	unsigned char temp[6];
#endif


	//void* opt=NULL;
	struct sk_buff *skb = NULL;
	struct switch_priv *priv = (struct switch_priv *) dev->priv;

	len = dma_device_read (dma_dev, &buf, (void **) &skb);

	if (skb == NULL || len == 0) {
		TRACE ("cannot restore pointer\n");
		goto switch_hw_receive_err_exit;
	}

    // 805271:<IFTW-jelly>
    if ( len < 64 )	// Length 0 means error, buffer_alloc has failed.
    {
        printk("packet is too small\n");
        //printk("dev name=%s\n", skb->dev->name);
        goto switch_hw_receive_err_exit1;
    }
    // end

	TRACE ("DMA rx len:%d\n", len);

	if (len >= 0x600) {
		TRACE ("packet too large %d\n", len);
		goto switch_hw_receive_err_exit;
	}

	/* remove CRC */
	len -= 4;
	if (len > (skb->end - skb->tail)) {
		TRACE ("BUG, len:%d end:%p tail:%p\n", (len + 4), skb->end,
		       skb->tail);
		goto switch_hw_receive_err_exit;
	}
	skb_put (skb, len);
	skb->dev = dev;
	if (buf) {
#ifdef AMAZON_SE_SW_DUMP
		printk ("\nrx:\n");
		dump_skb (len, (char *) buf);
#endif
	}


#if LOOPBACK_TEST		//VENUS

	//swap src and des mac address
	eth = (struct ethhdr *) (skb->data);
	memcpy (temp, eth->h_source, 6);
	memcpy (eth->h_source, eth->h_dest, 6);
	memcpy (eth->h_dest, temp, 6);

	switch_tx (skb, dev);

#else
	len = switch_rx (dev, len, skb);
#endif

	return OK;
      switch_hw_receive_err_exit:
	if (buf) {
#ifdef AMAZON_SE_SW_DUMP
		dump_skb (len, (char *) buf);
#endif
	}
	if (len == 0) {
		if (skb)
			dev_kfree_skb_any (skb);
		priv->stats.rx_errors++;
		priv->stats.rx_dropped++;
		return -EIO;
	}
	else {
		return len;
	}
    // 805271:<IFTW-jelly>
    switch_hw_receive_err_exit1:
      if (skb)
          dev_kfree_skb_any (skb);
      priv->stats.rx_errors++;
      priv->stats.rx_dropped++;
      return 0;
    // end

}


int
switch_hw_tx (char *buf, int len, struct net_device *dev)
{
	int ret = 0;

	
	struct switch_priv *priv = dev->priv;
	struct dma_device_info *dma_dev = priv->dma_device;


	ret = dma_device_write (dma_dev, buf, len, priv->skb);
//        printk ("\nLength: %d  ret : %d  off:%d\n", len,ret, off);
#if 0
	if (rx_stopped) {
		amazon_se_xon (dma_dev);
		rx_stopped = 0;
	}
#endif
	return ret;
}

int
select_tx_chan (struct sk_buff *skb, struct net_device *dev)
{
	int chan = 0;
/*TODO: select the channel based on some criteria*/

	return chan;
}


int
switch_tx (struct sk_buff *skb, struct net_device *dev)
{

	int len;
	char *data;
	struct switch_priv *priv = dev->priv;
	struct dma_device_info *dma_dev = priv->dma_device;
	len = skb->len < ETH_ZLEN ? ETH_ZLEN : skb->len;
	data = skb->data;
	priv->skb = skb;
	dev->trans_start = jiffies;
	dma_dev->current_tx_chan = select_tx_chan (skb, dev);	/*select the tx channel */
//	printk("switch tx\n");
	g_transmit++;
	if (switch_hw_tx (data, len, dev) != len) {

#if LOOPBACK_TEST		//VENUS
		dev_kfree_skb (skb);
#else
		dev_kfree_skb_any (skb);
#endif
		priv->stats.tx_errors++;
		priv->stats.tx_dropped++;

	}
	else {
		priv->stats.tx_packets++;
		priv->stats.tx_bytes += len;
	}
	return OK;
}


void
switch_tx_timeout (struct net_device *dev)
{
	int i;
	struct switch_priv *priv = (struct switch_priv *) dev->priv;

	TRACE ("%s tx_status==%d\n", dev->name, priv->tx_status);
	//TODO:must restart the TX channels
	priv->stats.tx_errors++;
	for (i = 0; i < priv->dma_device->max_tx_chan_num; i++) {
		priv->dma_device->tx_chan[i]->disable_irq (priv->dma_device->
							   tx_chan[i]);
	}
	netif_wake_queue (dev);
	return;
}

int
dma_intr_handler (struct dma_device_info *dma_dev, int status)
{
	struct net_device *dev;
	int i;
//  struct switch_priv* priv;

	dev = switch_devs + (u32) dma_dev->priv;
	TRACE ("status:%d \n", status);
	switch (status) {
	case RCV_INT:
		TRACE ("switch receive chan=%d\n", dma_dev->current_rx_chan);
		switch_hw_receive (dev, dma_dev);
		break;
	case TX_BUF_FULL_INT:
		TRACE ("tx buffer full\n");
		netif_stop_queue (dev);
		for (i = 0; i < dma_dev->max_tx_chan_num; i++) {
			if ((dma_dev->tx_chan[i])->control ==
			    AMAZON_SE_DMA_CH_ON)
				dma_dev->tx_chan[i]->enable_irq (dma_dev->
								 tx_chan[i]);
		}
		break;
	case TRANSMIT_CPT_INT:
		TRACE ("tx buffer released\n");
		for (i = 0; i < dma_dev->max_tx_chan_num; i++) {
			dma_dev->tx_chan[i]->disable_irq (dma_dev->
							  tx_chan[i]);
		}
		netif_wake_queue (dev);
		break;
	}
	return OK;
}

/* reserve 2 bytes in front of data pointer*/
u8 *
etop_dma_buffer_alloc (int len, int *byte_offset, void **opt)
{
	u8 *buffer = NULL;
	struct sk_buff *skb = NULL;
	skb = dev_alloc_skb (ETHERNET_PACKET_DMA_BUFFER_SIZE);
	if (skb == NULL) {
		//printk ("\n%s: return NULL", __FUNCTION__);
		return NULL;
	}
	buffer = (u8 *) (skb->data);
	skb_reserve (skb, 2);
	//TRACE("%8p\n",skb->data);
	*(int *) opt = (int) skb;
	*byte_offset = 2;
	return buffer;
}

int
etop_dma_buffer_free (u8 * dataptr, void *opt)
{
	struct sk_buff *skb = NULL;
	if (opt == NULL) {
		kfree (dataptr);
	}
	else {
		skb = (struct sk_buff *) opt;
#if LOOPBACK_TEST		//VENUS
		dev_kfree_skb (skb);
#else
		dev_kfree_skb_any (skb);
#endif

	}
	return OK;
}


static int
set_mac (struct net_device *dev, u16 speed, u8 duplex, u8 autoneg)
{
	int ret;
	int dev_num = dev - switch_devs;

	if (autoneg == AUTONEG_ENABLE) {
		/*set property and start autonegotiation */
		/*have to set mdio advertisement register and restart autonegotiation */
		/*which is a very rare case, put it to future development if necessary. */
		AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) = EPHY_RA_MASK | EPHY_WRITE_MASK | (EPHY_ADDRESS << EPHY_ADDRESS_SHIFT) | (EPHY_MDIO_BASE_CONTROL_REG << EPHY_REG_ADDRESS_SHIFT) | EPHY_RESET;	//0x83e08000;
		ephy_auto_negotiate (EPHY_ADDRESS);	//0x08


		ret = IFX_SUCCESS;
	}
	else {			/*autoneg==AUTONEG_DISABLE or -1 */


		/*set property without autonegotiation */

		if (g_ethernet_mode == EPHY_MODE) {


			u32 data = 0;
			AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) =
				EPHY_RA_MASK | EPHY_WRITE_MASK | (EPHY_ADDRESS
								  <<
								  EPHY_ADDRESS_SHIFT)
				| (EPHY_MDIO_BASE_CONTROL_REG <<
				   EPHY_REG_ADDRESS_SHIFT | EPHY_RESET);

//     data = ephy_read_mdio_reg (phy_addr,EPHY_MDIO_BASE_CONTROL_REG);
			data = ephy_read_mdio_reg (0xff,
						   EPHY_MDIO_BASE_CONTROL_REG);


			if (speed == SPEED_10)
				data &= ~EPHY_LINK_SPEED_MASK;
			else
				data |= EPHY_LINK_SPEED_MASK;

			if (duplex == DUPLEX_HALF)
				data &= ~EPHY_DUPLEX_MASK;
			else
				data |= EPHY_DUPLEX_MASK;

			ephy_write_mdio_reg (EPHY_ADDRESS,
					     EPHY_MDIO_BASE_CONTROL_REG,
					     (data | EPHY_RESET));

			//  while(ephy_read_mdio_reg (phy_addr,EPHY_MDIO_BASE_STATUS_REG) & EPHY_LINK_STATUS_MASK) ;

		}
		else {
			*ETOP_MDIO_CFG &= ~UMM0_MASK;

			/*set speed */
			if (speed == SPEED_10)
				*(ENET_MAC_CFG) &= ~SPEED_MASK;
			else if (speed == SPEED_100)
				*(ENET_MAC_CFG) |= SPEED_MASK;

			/*set duplex */
			if (duplex == DUPLEX_HALF)
				*(ENET_MAC_CFG) &= ~DUPLEX_MASK;
			else if (duplex)
				*(ENET_MAC_CFG) |= DUPLEX_MASK;

			*(ENET_MAC_CFG) |= LINK_MASK;
			ret = IFX_SUCCESS;
		}

	}
	return ret;
}

static int
switch_ethtool_ioctl (struct net_device *dev, struct ifreq *ifr)
{
	int ret = IFX_SUCCESS;
	struct switch_priv *priv = (struct switch_priv *) dev->priv;

	struct ethtool_cmd ecmd;
	/*calculate the enet offset */


	if (copy_from_user (&ecmd, ifr->ifr_data, sizeof (ecmd)))
		return -EFAULT;

	unsigned short data=0;
	if (g_ethernet_mode == EPHY_MODE) {
		//unsigned short data;          
		data = ephy_read_mdio_reg (EPHY_ADDRESS,   EPHY_SPECIFIC_STATUS_REG);
	}

	switch (ecmd.cmd) {
	case ETHTOOL_GSET:	/*get hardware information */
		{
			memset ((void *) &ecmd, 0, sizeof (ecmd));
			ecmd.supported =
				SUPPORTED_Autoneg | SUPPORTED_TP |
				SUPPORTED_MII | SUPPORTED_10baseT_Half |
				SUPPORTED_10baseT_Full |
				SUPPORTED_100baseT_Half |
				SUPPORTED_100baseT_Full;
			ecmd.port = PORT_MII;	//PORT_TP  //PORT_AUI

			if (g_ethernet_mode == EPHY_MODE)
			{
				ecmd.phy_address = EPHY_ADDRESS;
				ecmd.transceiver = XCVR_INTERNAL;	//external transceiver
			}
			else
			{
				ecmd.phy_address = priv->mdio_phy_addr;
				ecmd.transceiver = XCVR_EXTERNAL;	//external transceiver
			}

#if 1
			if (g_ethernet_mode == EPHY_MODE) {
				if (data & EPHY_SPECIFIC_STATUS_LINK_UP) {
					ecmd.reserved[0]= LINK_UP;
				}
				else {
					ecmd.reserved[0]= LINK_DOWN;
				}

				if (data & EPHY_SPECIFIC_STATUS_SPEED_MASK) {
					priv->current_speed = SPEED_100;
				}
				else {
					priv->current_speed = SPEED_10;
				}
			}
			else {
				if ((*(ENET_MAC_CFG)) & SPEED_MASK)

				{
					priv->current_speed = SPEED_100;
				}
				else {
					priv->current_speed = SPEED_10;

				}
			}
			ecmd.speed = priv->current_speed;
			if (g_ethernet_mode == EPHY_MODE) {
				if (data & EPHY_SPECIFIC_STATUS_DUPLEX_MASK) {
					priv->full_duplex = DUPLEX_FULL;
				}
				else {
					priv->full_duplex = DUPLEX_HALF;
				}
			}
			else {
				if (*(ENET_MAC_CFG) & DUPLEX_MASK)
				{
					priv->full_duplex = DUPLEX_FULL;
				}
				else {
					priv->full_duplex = DUPLEX_HALF;
				}
			}
			ecmd.duplex = priv->full_duplex;


			if (g_ethernet_mode == EPHY_MODE) {
				data = ephy_read_mdio_reg (EPHY_ADDRESS,  EPHY_MDIO_BASE_CONTROL_REG);
				if (data & EPHY_AUTO_NEG_ENABLE_MASK) 
					ecmd.autoneg = AUTONEG_ENABLE;
				else
					ecmd.autoneg = AUTONEG_DISABLE;

				data = ephy_read_mdio_reg (EPHY_ADDRESS,  EPHY_MDIO_ADVERTISMENT_REG);

				if (data & EPHY_MDIO_ADVERT_100_FD) 
					ecmd.advertising |=  ADVERTISED_100baseT_Full;
				else
					ecmd.advertising &=  ~ADVERTISED_100baseT_Full;

				if (data & EPHY_MDIO_ADVERT_100_HD) 
					ecmd.advertising |=  ADVERTISED_100baseT_Half;
				else
					ecmd.advertising &=  ~ADVERTISED_100baseT_Half;

				if (data & EPHY_MDIO_ADVERT_10_FD) 
					ecmd.advertising |=  ADVERTISED_10baseT_Full;
				else
					ecmd.advertising &=  ~ADVERTISED_10baseT_Full;
				if (data & EPHY_MDIO_ADVERT_10_HD) 
					ecmd.advertising |=  ADVERTISED_10baseT_Half;
				else
					ecmd.advertising &=  ~ADVERTISED_10baseT_Half;

				
			}
			else {
				if (*ETOP_MDIO_CFG & UMM0_MASK) {
					ecmd.autoneg = AUTONEG_ENABLE;
					ecmd.advertising |=
						ADVERTISED_10baseT_Half |
						ADVERTISED_10baseT_Full |
						ADVERTISED_100baseT_Half |
						ADVERTISED_100baseT_Full;
				}
				else {
					ecmd.autoneg = AUTONEG_DISABLE;
					ecmd.advertising &=
						~(ADVERTISED_10baseT_Half |
						  ADVERTISED_10baseT_Full |
						  ADVERTISED_100baseT_Half |
						  ADVERTISED_100baseT_Full);
				}

			}

#endif
			if (copy_to_user
			    (ifr->ifr_data, &ecmd, sizeof (ecmd)))
				return -EFAULT;
			ret = IFX_SUCCESS;

		}

		break;

	case ETHTOOL_SSET:	/*force the speed and duplex mode */
		{
			if (!capable (CAP_NET_ADMIN)) {
				return -EPERM;
			}
			ret = set_mac (dev, ecmd.speed, ecmd.duplex,
				       ecmd.autoneg);
			/*
			   if (ecmd.autoneg == AUTONEG_ENABLE) {
			   ephy_set_duplex(dev,autoneg);
			   ephy_set_speed(dev,0);
			   } else {
			   ephy_set_duplex(dev,ecmd.duplex == DUPLEX_HALF ? half : full);
			   ephy_set_speed(dev,ecmd.speed == SPEED_10 ? 10: 100);
			   } */
		}
		break;

	case ETHTOOL_GDRVINFO:	/*get driver information */
		{
			struct ethtool_drvinfo info;
			memset ((void *) &info, 0, sizeof (info));
			strncpy (info.driver, "AMAZON_SE ETOP DRIVER",
				 sizeof (info.driver) - 1);
			strncpy (info.fw_version, "0.0.1",
				 sizeof (info.fw_version) - 1);
			strncpy (info.bus_info, "N/A",
				 sizeof (info.bus_info) - 1);
			info.regdump_len = 0;
			info.eedump_len = 0;
			info.testinfo_len = 0;
			if (copy_to_user
			    (ifr->ifr_data, &info, sizeof (info)))
				return -EFAULT;
			ret = IFX_SUCCESS;
		}
		break;
	case ETHTOOL_NWAY_RST:	/*restart auto negotiation */
		if (g_ethernet_mode == REV_MII_MODE) {
			*ETOP_MDIO_CFG |= UMM0_MASK | SMRST_MASK;
		}
		else if (g_ethernet_mode == EPHY_MODE) {
			// 
			AMAZON_SE_SW_REG32 (ETOP_MDIO_ACC) = EPHY_RA_MASK | EPHY_WRITE_MASK | (EPHY_ADDRESS << EPHY_ADDRESS_SHIFT) | (EPHY_MDIO_BASE_CONTROL_REG << EPHY_REG_ADDRESS_SHIFT) | EPHY_RESET;	//0x83e08000;
			ephy_auto_negotiate (EPHY_ADDRESS);	//0x08

		}
		ret = IFX_SUCCESS;
		break;
	default:
		return -EOPNOTSUPP;
		break;

	}
	return ret;
}

int
set_vlan_cos (_vlan_cos_req * vlan_cos_req)
{
	u32 pri = vlan_cos_req->pri;
	u32 cos_value = vlan_cos_req->cos_value;
	u32 value;
	value = *ETOP_IG_VLAN_COS & ~(3 << (pri * 2));
	*ETOP_IG_VLAN_COS = value | (cos_value << (pri * 2));
	return IFX_SUCCESS;	/*cannot fail */
}


int
set_dscp_cos (_dscp_cos_req * dscp_cos_req)
{
	u32 dscp = dscp_cos_req->dscp;
	u32 cos_value = dscp_cos_req->cos_value;
	u32 value;
	value = *(ETOP_IG_DSCP_COS0 -
		  (dscp / 16 * 4)) & ~(3 << dscp % 16 * 2);
	*(ETOP_IG_DSCP_COS0 - (dscp / 16 * 4)) |= cos_value << dscp % 16 * 2;
	return IFX_SUCCESS;
}



int
switch_ioctl (struct net_device *dev, struct ifreq *ifr, int cmd)
{

	int result = IFX_SUCCESS;

	_vlan_cos_req etop_vlan_cos_req;
	_dscp_cos_req etop_dscp_cos_req;
	switch (cmd) {
	case SIOCETHTOOL:
		switch_ethtool_ioctl (dev, ifr);
		break;
	case SET_VLAN_COS:
		copy_from_user (&etop_vlan_cos_req,
				(_vlan_cos_req *) ifr->ifr_data,
				sizeof (_vlan_cos_req));
		set_vlan_cos (&etop_vlan_cos_req);
		break;

	case SET_DSCP_COS:
		copy_from_user (&etop_dscp_cos_req,
				(_vlan_cos_req *) ifr->ifr_data,
				sizeof (_dscp_cos_req));
		set_dscp_cos (&etop_dscp_cos_req);
		break;
	case ENABLE_VLAN_CLASSIFICATION:
		*(ENETS_COS_CFG) |= VLAN_MASK;
		break;
	case DISABLE_VLAN_CLASSIFICATION:
		*(ENETS_COS_CFG) &= ~VLAN_MASK;
		break;
	case ENABLE_DSCP_CLASSIFICATION:
		*(ENETS_COS_CFG) |= DSCP_MASK;
		break;
	case DISABLE_DSCP_CLASSIFICATION:
		*(ENETS_COS_CFG) &= ~DSCP_MASK;
		break;
	case VLAN_CLASS_FIRST:
		*(ENETS_CFG) &= ~FTUC_MASK;
		break;
	case VLAN_CLASS_SECOND:
		*(ENETS_CFG) |= VL2_MASK;
		break;
	case PASS_UNICAST_PACKETS:
		*(ENETS_CFG) &= ~FTUC_MASK;
		break;
	case FILTER_UNICAST_PACKETS:
		*(ENETS_CFG) |= FTUC_MASK;
		break;
	case KEEP_BROADCAST_PACKETS:
		*(ENETS_CFG) &= ~DPBC_MASK;
		break;
	case DROP_BROADCAST_PACKETS:
		*(ENETS_CFG) |= DPBC_MASK;
		break;
	case KEEP_MULTICAST_PACKETS:
		*(ENETS_CFG) &= ~DPMC_MASK;
		break;
	case DROP_MULTICAST_PACKETS:
		*(ENETS_CFG) |= DPMC_MASK;
		break;
	default:
		break;
	}

	return result;
}



static struct net_device_stats *
amazon_se_get_stats (struct net_device *dev)
{
	return (struct net_device_stats *) dev->priv;
}

int
etop_register_proc_read (char *buf, char **start, off_t offset,
			 int count, int *eof, void *data)
{
	int len = 0;;

	len += sprintf (buf + len, "etop register\n");
	len += sprintf (buf + len,
			"%d packets from protocol stack received!\n",
			g_transmit);
	/*TODO: add the register value here, if the user does not want to use the mem application */

	return len;
}


int
etop_dscp_cos_proc_read (char *buf, char **start, off_t offset,
			 int count, int *eof, void *data)
{
	int len = 0;;
	int i;
	int j;
	len += sprintf (buf + len, "DSCP COS MAP:\n");
	for (j = 0; j < 4; j++) {
		for (i = 0; i < 16; i++) {
			len += sprintf (buf + len, "DSCP%d %d\n", i + j * 16,
					(u32) ((*(ETOP_IG_DSCP_COS0 - j * 4)
						>> (i * 2)) & 0x3));
		}
	}
	return len;
}

int
etop_vlan_cos_proc_read (char *buf, char **start, off_t offset,
			 int count, int *eof, void *data)
{
	int len = 0;;
	int i;
	len += sprintf (buf + len, "VLAN COS MAP\n");
	for (i = 0; i < 8; i++) {
		len += sprintf (buf + len, "Pri %d %d\n", i,
				(u32) ((*ETOP_IG_VLAN_COS >> (i * 2)) & 0x3));
	}
	return len;
}


struct net_device_stats *
switch_stats (struct net_device *dev)
{
	struct switch_priv *priv = (struct switch_priv *) dev->priv;

	return &priv->stats;
}



static int
switch_init (struct net_device *dev)
{
	int result;
	u64 retval = 0;
	int i,k;

	struct switch_priv *priv;
	ether_setup (dev);	/* assign some of the fields */

	TRACE ("%s up\n", dev->name);

	dev->open = amazon_se_switch_open;
	dev->stop = switch_release;
	dev->hard_start_xmit = switch_tx;
	dev->do_ioctl = switch_ioctl;
	dev->get_stats = amazon_se_get_stats;
	//dev->change_mtu      = switch_change_mtu;
	//dev->set_mac_address =switch_set_mac_address;
	dev->tx_timeout = switch_tx_timeout;
	dev->watchdog_timeo = timeout;
	//dev->flags           |= IFF_NOARP|IFF_PROMISC;

	dev->priv = kmalloc (sizeof (struct switch_priv), GFP_KERNEL);
	if (dev->priv == NULL)
	{
		printk("%s: Memory allocation Failed!!!\n",__FUNCTION__);
		return -ENOMEM;
	}
	memset (dev->priv, 0, sizeof (struct switch_priv));
	priv = dev->priv;

	/*initialize the dma device */
	priv->dma_device = dma_device_reserve ("PPE");
	if (!priv->dma_device)
	{
		return IFX_ERROR;
	}
	priv->dma_device->buffer_alloc = &etop_dma_buffer_alloc;
	priv->dma_device->buffer_free = &etop_dma_buffer_free;
	priv->dma_device->intr_handler = &dma_intr_handler;
	priv->dma_device->max_rx_chan_num = 4;	/*turn on all the receive channels */

	for (i = 0; i < priv->dma_device->max_rx_chan_num; i++) {
		priv->dma_device->rx_chan[i]->packet_size =
			ETHERNET_PACKET_DMA_BUFFER_SIZE;
		priv->dma_device->rx_chan[i]->control = AMAZON_SE_DMA_CH_ON;
	}

	for (i = 0; i < priv->dma_device->max_tx_chan_num; i++) {
		if (i == 0)
		{
			priv->dma_device->tx_chan[i]->control =
				AMAZON_SE_DMA_CH_ON;
		}
		else
			priv->dma_device->tx_chan[i]->control =
				AMAZON_SE_DMA_CH_OFF;
	}

	result = dma_device_register (priv->dma_device);


	/*read the mac address from the mac table and put them into the mac table. */
	for (i = 0; i < 6; i++) {
		retval += g_my_ethaddr[i];
	}
	/* ethaddr not set in u-boot ? */
	if (retval == 0) {
		TRACE ("use default MAC address\n");
		dev->dev_addr[0] = 0x00;
		dev->dev_addr[1] = 0x20;
		dev->dev_addr[2] = 0xda;
		dev->dev_addr[3] = 0x86;
		dev->dev_addr[4] = 0x23;
		dev->dev_addr[5] = 0x74 + (unsigned char) (dev - switch_devs);
	}
	else {
		for (i = 0; i < 6; i++) {
			dev->dev_addr[i] = g_my_ethaddr[i];
		}
		dev->dev_addr[5] += +(unsigned char) (dev - switch_devs);
	}
#ifdef AMAZON_SW_DMSG_MSG
	retval = 0;
	for (i = 0; i < 6; i++) {
		retval = (retval << 8) + dev->dev_addr[i];
		printk (" %2x ", dev->dev_addr[i]);
	}
	//add_mac_table_entry(retval);
#endif

	return 0;
}


void
configureMiiRxClk (int external, int loopback)
{
	if (external == 1) 
	{
		//Clock is from External (From PAD)	
		*AMAZON_SE_CGU_IFCCR = (*AMAZON_SE_CGU_IFCCR & ~(0x3 << 28));
	}
	else 
	{
		//Clock is from  CGU
		if (loopback == 0) 
		{
		
			*AMAZON_SE_CGU_IFCCR =
				(*AMAZON_SE_CGU_IFCCR | (0x3 << 28));
		}
		else 
		{
			*AMAZON_SE_CGU_IFCCR =
				((*AMAZON_SE_CGU_IFCCR | (0x1 << 28)) &
				 ~(0x1 << 29));
		}
	}
}


void
configureMiiTxClk (int external, int loopback)
{
	if (external == 1) 
	{
		//Clock is from External (From PAD)	
		*AMAZON_SE_CGU_IFCCR = (*AMAZON_SE_CGU_IFCCR & ~(0x3 << 26));
	}
	else 
	{
		//Clock is from  CGU
		if (loopback == 0) 
		{
			*AMAZON_SE_CGU_IFCCR =
				(*AMAZON_SE_CGU_IFCCR | (0x3 << 26));
		}
		else 
		{
			*AMAZON_SE_CGU_IFCCR =
				((*AMAZON_SE_CGU_IFCCR | (0x1 << 26)) &
				 ~(0x1 << 27));
		}
	}
}


void
configureRMiiRefClk (int external, int loopback)
{
	if (external == 1) 
	{
		//Clock is from External (From PAD)	
		*AMAZON_SE_CGU_IFCCR = (*AMAZON_SE_CGU_IFCCR & ~(0x3 << 24));
	}
	else 
	{
		//Clock is from  CGU
		if (loopback == 0)
		{
			*AMAZON_SE_CGU_IFCCR =
				(*AMAZON_SE_CGU_IFCCR | (0x3 << 24));
		}
		else
		{
			*AMAZON_SE_CGU_IFCCR =
				((*AMAZON_SE_CGU_IFCCR | (0x1 << 24)) &
				 ~(0x1 << 25));
		}
	}
}


void
configurePhyClk (int enable)
{
	if (enable == 1)
	{
		//Clock from External  
		*AMAZON_SE_CGU_IFCCR = (*AMAZON_SE_CGU_IFCCR & ~(0x1 << 4));
		*AMAZON_SE_CGU_OSC_CON = (*AMAZON_SE_CGU_OSC_CON ) | 0x18;
		printk(" External Clock\n");
	}	
	else
	{
		//Clock from CGU
		*AMAZON_SE_CGU_IFCCR = (*AMAZON_SE_CGU_IFCCR | (0x1 << 4));
		*AMAZON_SE_CGU_OSC_CON = (*AMAZON_SE_CGU_OSC_CON ) & ~(0x18);
		printk("Internal Clock\n");
	}

}

int
configureMiiGpioChan (int mode, int gpio)
{

	if (gpio == 0)
		return;

	if ( mode == MII_MODE )
	{

	} else if (mode == REV_MII_MODE )
	{
		//GPIO 0 (TXD0), 4(TXD1), 12 (RXDV) & 13(TXEN)
		*AMAZON_SE_GPIO_P0_ALTSEL0 =
			(*AMAZON_SE_GPIO_P0_ALTSEL0 & 0xCFFF ) | 0x0010;
		*AMAZON_SE_GPIO_P0_ALTSEL1 =
			(*AMAZON_SE_GPIO_P0_ALTSEL1 & 0xfffe) | 0x3011;
#if 1
		*AMAZON_SE_GPIO_P0_OD = *(AMAZON_SE_GPIO_P0_OD) | 0x3011;
		*AMAZON_SE_GPIO_P0_DIR = (*AMAZON_SE_GPIO_P0_DIR) & ~(0x3011);
		*AMAZON_SE_GPIO_P0_DIR = (*AMAZON_SE_GPIO_P0_DIR) | (0x1000);
		//GPIO 29 (TXD2), 28(TXD3) 
		*AMAZON_SE_GPIO_P1_ALTSEL0 =
			(*AMAZON_SE_GPIO_P1_ALTSEL0 & 0xCFFE ) ;
		*AMAZON_SE_GPIO_P1_ALTSEL1 =
			(*AMAZON_SE_GPIO_P1_ALTSEL1 & 0xffff) | 0x3000;
		*AMAZON_SE_GPIO_P1_DIR = (*AMAZON_SE_GPIO_P1_DIR) | (0x3000);

		//GPIO 19 (RXD3), 20(RXD2), 23 (RXD1), 25 (RXD0), 24 (MDIO) & 27 (MDC)
		*AMAZON_SE_GPIO_P1_ALTSEL0 =
			(*AMAZON_SE_GPIO_P1_ALTSEL0 & 0xFD67 )  | 0x0900;
		*AMAZON_SE_GPIO_P1_ALTSEL1 =
			(*AMAZON_SE_GPIO_P1_ALTSEL1 & 0xffff) | 0x0B98;
		*AMAZON_SE_GPIO_P1_OD = *(AMAZON_SE_GPIO_P1_OD) | 0x0998;
		*AMAZON_SE_GPIO_P1_DIR = (*AMAZON_SE_GPIO_P1_DIR) |(0x0998);
		*AMAZON_SE_GPIO_P1_DIR = (*AMAZON_SE_GPIO_P1_DIR) &(~0x0200);

		//GPIO 21 (CRS), 22(RX Clk) & 26 (TX Clk)  
		*AMAZON_SE_GPIO_P1_ALTSEL0 =
			(*AMAZON_SE_GPIO_P1_ALTSEL0 & 0xFB9F ) ;
		*AMAZON_SE_GPIO_P1_ALTSEL1 =
			(*AMAZON_SE_GPIO_P1_ALTSEL1 & 0xffff) | 0x0460;
		*AMAZON_SE_GPIO_P1_DIR =  ((*AMAZON_SE_GPIO_P1_DIR) & ~(0x0220) ) ;
#else
		*AMAZON_SE_GPIO_P1_ALTSEL0 =
			(*AMAZON_SE_GPIO_P1_ALTSEL0 & 0xc907 )| 0x0900 ;
		*AMAZON_SE_GPIO_P1_ALTSEL1 =
			(*AMAZON_SE_GPIO_P1_ALTSEL1 & 0xffff) | 0x3ff8;
		*AMAZON_SE_GPIO_P0_OD = *(AMAZON_SE_GPIO_P0_OD) | 0xffff;
		*AMAZON_SE_GPIO_P1_DIR =  ((*AMAZON_SE_GPIO_P1_DIR) & (0xff9f) ) ;
#endif
		printk("GPIO setting is done for REV MII\n");

	} else if (mode == RED_MII_MODE) 
	{
		// Red MII
		//Common  GPIO 0 (TXD0), 4(TXD1) & 13(TXEN)
		*AMAZON_SE_GPIO_P0_ALTSEL0 =
			(*AMAZON_SE_GPIO_P0_ALTSEL0 & 0xDFFE ) | 0x0010;
		*AMAZON_SE_GPIO_P0_ALTSEL1 =
			(*AMAZON_SE_GPIO_P0_ALTSEL1 & 0xffff) | 0x2011;
		*AMAZON_SE_GPIO_P0_OD = *(AMAZON_SE_GPIO_P0_OD) | 0x2011;
		 //GPIO24 (MDIO) & 27 (MDC)
                *AMAZON_SE_GPIO_P1_ALTSEL0 =
                        (*AMAZON_SE_GPIO_P1_ALTSEL0 & 0xFFFF )  | 0x0900;
                *AMAZON_SE_GPIO_P1_ALTSEL1 =
                        (*AMAZON_SE_GPIO_P1_ALTSEL1 & 0xffff) | 0x0900;
                *AMAZON_SE_GPIO_P1_DIR = (*AMAZON_SE_GPIO_P1_DIR) |(0x0900);

#if RED_MII_MUX_WITH_SPI
		//GPIO 7(RXDV),8(RXD1),9(RXD0) & 10(Ref Clock)
		*AMAZON_SE_GPIO_P0_ALTSEL0 =
			(*AMAZON_SE_GPIO_P0_ALTSEL0 & 0xF87F ) ;
		*AMAZON_SE_GPIO_P0_ALTSEL1 =
			(*AMAZON_SE_GPIO_P0_ALTSEL1 & 0xffff) | 0x0780;
		*AMAZON_SE_GPIO_P0_OD = *(AMAZON_SE_GPIO_P0_OD) | 0x0400;
	
		*AMAZON_SE_GPIO_P0_DIR = (*AMAZON_SE_GPIO_P0_DIR) & 0xFC7F;

		printk(" SPI  Inactive\n");
#else
		//GPIO 21,22,23 & 25
		*AMAZON_SE_GPIO_P1_ALTSEL0 =
			(*AMAZON_SE_GPIO_P1_ALTSEL0 & 0xFD1F) ;
		*AMAZON_SE_GPIO_P1_ALTSEL1 =
			(*AMAZON_SE_GPIO_P1_ALTSEL1 & 0xFFFF ) | 0x02E0;
		*AMAZON_SE_GPIO_P1_OD = *(AMAZON_SE_GPIO_P1_OD) | 0x0040;
		*AMAZON_SE_GPIO_P1_DIR = (*AMAZON_SE_GPIO_P1_DIR & 0xFD5F);
		printk(" FLASH Inactive\n");

#endif


	}
}



void
enableTurboMode ()
{
	*AMAZON_SE_PPE32_ETOP_CFG = *AMAZON_SE_PPE32_ETOP_CFG | 0x00000004;
}

void
disableTurboMode ()
{
	*AMAZON_SE_PPE32_ETOP_CFG = *AMAZON_SE_PPE32_ETOP_CFG & 0xfffffffb;
}



static int
amazon_se_sw_chip_init (int mode, int gpio)
{

	int i;
	unsigned short data=0;

/* 30 May 2007 start */
	*(volatile u32*)AMAZON_SE_PPE32_BASE=0x0;
	*AMAZON_SE_RCU_RST_REQ |= 0x100;   //Reset PPE module
/* 30 May 2007 end */


	*AMAZON_SE_PMU_PWDCR &= ~(1 << 13);	/*enable PPE  from PMU */
	for (i = 0; i < 0x5000; i++);

	g_ethernet_mode = mode;

	switch (mode) {

	case MII_MODE:
		configureMiiGpioChan (mode, gpio);
		configureMiiRxClk (1, 0);	//externel = 0, loop back 0
		configureMiiTxClk (1, 0);
		*AMAZON_SE_PPE32_ETOP_CFG =
			*AMAZON_SE_PPE32_ETOP_CFG & 0xfffffffe;
		printk("Selected MII_MODE  \n");
		break;
	case REV_MII_MODE:
		*AMAZON_SE_PPE32_ETOP_CFG =
			(*AMAZON_SE_PPE32_ETOP_CFG ) | 0x00000001;
		configureMiiRxClk (0, 0);	//externel = 0, loop back 0
		configureMiiTxClk (0, 0);
		configureMiiGpioChan (mode, gpio);
		*AMAZON_SE_PPE32_ETOP_CFG = (*AMAZON_SE_PPE32_ETOP_CFG) & ~(0x4000);
		*AMAZON_SE_CGU_OSC_CON = (*AMAZON_SE_CGU_OSC_CON ) & ~(0x18);
		*AMAZON_SE_PPE32_ETOP_CFG =
			(*AMAZON_SE_PPE32_ETOP_CFG ) | 0x00000002;
		*AMAZON_SE_PPE32_ETOP_CFG =
			*AMAZON_SE_PPE32_ETOP_CFG & 0xfffffffe;
		printk("Selected REV_MII_MODE \n");
		break;

	case RED_MII_MODE:
		*AMAZON_SE_PPE32_ETOP_CFG =
			(*AMAZON_SE_PPE32_ETOP_CFG ) | 0x00000001;
		configureRMiiRefClk (0, 0);	//clock from CGU
		configureMiiGpioChan (mode, gpio);
		*AMAZON_SE_PPE32_ETOP_CFG = (*AMAZON_SE_PPE32_ETOP_CFG) & ~(0x4000);
		//enable Red Mii mode and clock is from CGU
		*AMAZON_SE_PPE32_ETOP_CFG =
			(*AMAZON_SE_PPE32_ETOP_CFG)  | 0x00001000;
		*AMAZON_SE_CGU_OSC_CON = (*AMAZON_SE_CGU_OSC_CON ) & ~(0x18);
		//enable Red Mii mode and clock is from External 
//		*AMAZON_SE_PPE32_ETOP_CFG =
//			(*AMAZON_SE_PPE32_ETOP_CFG  | 0x00009000;
		*AMAZON_SE_PPE32_ETOP_CFG =
			*AMAZON_SE_PPE32_ETOP_CFG & 0xfffffffe;
		//Leon 07092805 disable MDIO auto-sensing mode
                *AMAZON_SE_PPE32_ETOP_MDIO_CFG &= ~(1 << 1);
		printk("Selected RED_MII_MODE \n");
		break;
	case EPHY_MODE:
		*AMAZON_SE_PMU_PWDCR &= ~(1 << 7);	/*enable EPHY module  from PMU */
		for (i = 0; i < 0x500; i++);
		*AMAZON_SE_PPE32_ETOP_CFG =
			(*AMAZON_SE_PPE32_ETOP_CFG ) | 0x00000001;
#if EPHY_CLOCK
		configurePhyClk (1);   //clock from External
#else
		configurePhyClk (0);   //clock from CGU 
#endif
//		*AMAZON_SE_PPE32_ETOP_CFG =  0x4141;
		*AMAZON_SE_PPE32_ETOP_CFG = (*AMAZON_SE_PPE32_ETOP_CFG) |0x4000;
		//unsigned short data;          
/* 08 June 2007 start */
#if SET_CLASS_A_VALUE
//		data = ephy_read_mdio_reg (EPHY_ADDRESS,  EPHY_SET_CLASS_VALUE_REG);
		ephy_write_mdio_reg (EPHY_ADDRESS, EPHY_SET_CLASS_VALUE_REG, 0xc020);
#endif
/* 08 June 2007 End*/
		printk("Selected EPHY_MODE \n");
		break;

	default:
		printk (" Switch Mode is not selected, Error!!! \n");

	}

/* 14 May 2007 Start */
//	*AMAZON_SE_PPE32_ETOP_IG_PLEN_CTRL = 0x4005ee;	// set packetlen.
	*AMAZON_SE_PPE32_ETOP_IG_PLEN_CTRL = 0x4005dc;	// set packetlen.
/* 14 May 2007 End*/
	*ENET_MAC_CFG |= 1 << 11;	/*enable the crc */

}


/* Initialize the rest of the LOOPBACK device. */
int __init
switch_init_module (void)
{

	int i = 0, result, device_present = 0;
	struct net_device *dev;
	g_amazon_se_etop_dir = proc_mkdir ("amazon_se_etop", NULL);

	create_proc_read_entry ("register",
				0,
				g_amazon_se_etop_dir,
				etop_register_proc_read, NULL);
	create_proc_read_entry ("DSCP_COS",
				0,
				g_amazon_se_etop_dir,
				etop_dscp_cos_proc_read, NULL);
	create_proc_read_entry ("VLAN_COS",
				0,
				g_amazon_se_etop_dir,
				etop_vlan_cos_proc_read, NULL);

	/*
	 *      Fill in the generic fields of the device structure. 
	 */

	for (i = 0; i < AMAZON_SE_SW_INT_NO; i++) {
		dev = switch_devs + i;
		strcpy (dev->name, "eth%d");
		SET_MODULE_OWNER (dev);
		if ((result = register_netdev (dev)))
			TRACE ("error %i registering device \"%s\"\n", result,
			       dev->name);
		else
			device_present++;
	}
	printk ("amazon_se ETOP driver loaded!\n");

//	amazon_se_sw_chip_init (REV_MII_MODE, SET_GPIO);
#ifdef CONFIG_AMAZON_SE_ETHERNET_RMII
        amazon_se_sw_chip_init(RED_MII_MODE, SET_GPIO);
#endif
#ifdef CONFIG_AMAZON_SE_ETHERNET_EPHY
        amazon_se_sw_chip_init(EPHY_MODE, 0);
#endif
	return (0);
};

void extern_sw_exit(void)
{
	int i, ret;
	struct net_device *dev;

	remove_proc_entry ("register",g_amazon_se_etop_dir);
        remove_proc_entry ("DSCP_COS", g_amazon_se_etop_dir);
        remove_proc_entry ("VLAN_COS",g_amazon_se_etop_dir);

	struct switch_priv *priv;
	for (i = 0; i < AMAZON_SE_SW_INT_NO; i++) {

		priv = switch_devs[i].priv;
		if (priv->dma_device) {
			ret=dma_device_unregister (priv->dma_device);
			if(ret != IFX_SUCCESS)
				printk("%s: dma_device_unregister failed", __FUNCTION__);
			dma_device_release (priv->dma_device);
			priv->dma_device = NULL;
//			kfree (priv);
			TRACE ("dma_device null now!\n");
		}
		kfree (switch_devs[i].priv);

		unregister_netdev (switch_devs + i);
	}
	printk("\namazon_se_sw EXITED\n");

	return;
}
EXPORT_SYMBOL(extern_sw_exit);

static void __exit
switch_cleanup (void)
{
	extern_sw_exit();
}

module_init (switch_init_module);
module_exit (switch_cleanup);

void amazon_se_sw_config(void)
{
#ifdef CONFIG_AMAZON_SE_ETHERNET_RMII
        amazon_se_sw_chip_init(RED_MII_MODE, SET_GPIO);
#endif
#ifdef CONFIG_AMAZON_SE_ETHERNET_EPHY
        amazon_se_sw_chip_init(EPHY_MODE, 0);
#endif
}
EXPORT_SYMBOL(amazon_se_sw_config);
EXPORT_SYMBOL(g_my_ethaddr);

