/******************************************************************************
**
** FILE NAME    : amazon_se_ptm.c
** PROJECT      : AMAZON_SE
** MODULES     	: DSL PTM
**
** DATE         : 2 OCT 2007
** AUTHOR       : Xu Liang
** DESCRIPTION  : DSL PTM Driver
** COPYRIGHT    : 	Copyright (c) 2006
**			Infineon Technologies AG
**			Am Campeon 1-12, 85579 Neubiberg, Germany
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
** HISTORY
** $Date        $Author         $Comment
**  2 OCT 2007  Xu Liang        Initiate Version
** 1 April      Huang Xiaogang  add amazon_se_sw_config to ppe reset function 
*******************************************************************************/



/*
 * ####################################
 *              Head File
 * ####################################
 */

/*
 *  Common Head File
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/atmdev.h>
#include <linux/init.h>
#include <linux/etherdevice.h>  /*  eth_type_trans  */
#include <linux/ethtool.h>      /*  ethtool_cmd     */
#include <asm/uaccess.h>
#include <asm/unistd.h>
#include <asm/irq.h>
#include <asm/delay.h>
#include <asm/io.h>
#include <linux/errno.h>

/*
 *  Chip Specific Head File
 */
#include <asm/amazon_se/irq.h>

//#include <asm/amazon_se/amazon_se_cgu.h>
#include <asm/amazon_se/amazon_se_ptm.h>


/*
 * ####################################
 *   Parameters to Configure PPE
 * ####################################
 */
static int write_descriptor_delay  = 0x20;      /*  Write descriptor delay                          */

static int rx_max_packet_size   = 0x05EE;       /*  Max packet size for RX                          */
static int rx_min_packet_size   = 0x0040;       /*  Min packet size for RX                          */
static int tx_max_packet_size   = 0x05EE;       /*  Max packet size for TX                          */
static int tx_min_packet_size   = 0x0040;       /*  Min packet size for TX                          */

static int dma_rx_descriptor_length = 24;       /*  Number of descriptors per DMA RX channel        */
static int dma_tx_descriptor_length = 24;       /*  Number of descriptors per DMA TX channel        */

static int eth_efmtc_crc_cfg = 0x03100710;      /*  default: tx_eth_crc_check: 1, tx_tc_crc_check: 1, tx_tc_crc_len = 16    */
                                                /*           rx_eth_crc_present: 1, rx_eth_crc_check: 1, rx_tc_crc_check: 1, rx_tc_crc_len = 16 */

module_param(write_descriptor_delay, int, 0600);
MODULE_PARM_DESC(write_descriptor_delay, "PPE core clock cycles between descriptor write and effectiveness in external RAM");

module_param(rx_max_packet_size, int, 0600);
MODULE_PARM_DESC(rx_max_packet_size, "Max packet size in byte for downstream ethernet frames");

module_param(rx_min_packet_size, int, 0600);
MODULE_PARM_DESC(rx_min_packet_size, "Min packet size in byte for downstream ethernet frames");

module_param(tx_max_packet_size, int, 0600);
MODULE_PARM_DESC(tx_max_packet_size, "Max packet size in byte for upstream ethernet frames");

module_param(tx_min_packet_size, int, 0600);
MODULE_PARM_DESC(tx_min_packet_size, "Min packet size in byte for upstream ethernet frames");

module_param(dma_rx_descriptor_length, int, 0600);
MODULE_PARM_DESC(dma_rx_descriptor_length, "Number of descriptor assigned to DMA RX channel (>16)");

module_param(dma_tx_descriptor_length, int, 0600);
MODULE_PARM_DESC(dma_tx_descriptor_length, "Number of descriptor assigned to DMA TX channel (>16)");

module_param(eth_efmtc_crc_cfg, int, 0600);
MODULE_PARM_DESC(eth_efmtc_crc_cfg, "Configuration for PTM TX/RX ethernet/efm-tc CRC");



/*
 * ####################################
 *              Definition
 * ####################################
 */

#define DISABLE_TASKLET                 1

#define ENABLE_ETH2_DEBUG               0

#define ENABLE_ETH2_ASSERT              0

#define ENABLE_DEBUG_COUNTER            0

#define DEBUG_DUMP_RX_SKB_BEFORE        0   //  before function eth_type_trans

#define DEBUG_DUMP_RX_SKB_AFTER         0   //  after function eth_type_trans

#define DEBUG_DUMP_TX_SKB               0

#define DEBUG_DUMP_ETOP_REGISTER        0

#define DEBUG_WRITE_GPIO_REGISTER       1

#define ENABLE_PROBE_TRANSCEIVER        0

#define ENABLE_ETH2_HW_FLOWCONTROL      1

#define DEBUG_MEM_PROC                  1

#define ENABLE_DBG_PROC                 0

#define PPE_MAILBOX_IGU1_INT            INT_NUM_IM2_IRL13

#define MY_ETHADDR                      my_ethaddr

  #include <asm/amazon_se/amazon_se_ptm_fw.h>

#if (defined(DEBUG_DUMP_RX_SKB_BEFORE) && DEBUG_DUMP_RX_SKB_BEFORE) || (defined(DEBUG_DUMP_RX_SKB_AFTER) && DEBUG_DUMP_RX_SKB_AFTER)
  #define DEBUG_DUMP_RX_SKB             1

  #undef  ENABLE_DBG_PROC
  #define ENABLE_DBG_PROC               1
#else
  #define DEBUG_DUMP_RX_SKB             0
#endif

#if defined(CONFIG_NET_HW_FLOWCONTROL) && (defined(ENABLE_ETH2_HW_FLOWCONTROL) && ENABLE_ETH2_HW_FLOWCONTROL)
  #define ETH2_HW_FLOWCONTROL           1
#else
  #define ETH2_HW_FLOWCONTROL           0
#endif

#if defined(ENABLE_ETH2_DEBUG) && ENABLE_ETH2_DEBUG
  #define ENABLE_DEBUG_PRINT            1
  #define DISABLE_INLINE                1
#else
  #define ENABLE_DEBUG_PRINT            0
  #define DISABLE_INLINE                0
#endif  //  defined(ENABLE_ETH2_DEBUG) && ENABLE_ETH2_DEBUG

#if !defined(DISABLE_INLINE) || !DISABLE_INLINE
  #define INLINE                        inline
#else
  #define INLINE
#endif  //  !defined(DISABLE_INLINE) || !DISABLE_INLINE

#if defined(ENABLE_DEBUG_PRINT) && ENABLE_DEBUG_PRINT
  #undef  dbg
  #define dbg(format, arg...)           printk(KERN_WARNING __FILE__ ":%d:%s: " format "\n", __LINE__, __FUNCTION__, ##arg)
#else
  #if !defined(dbg)
    #define dbg(format, arg...)         do {} while (0)
  #endif    //  !defined(dbg)
#endif  //  defined(ENABLE_DEBUG_PRINT) && ENABLE_DEBUG_PRINT

#if defined(ENABLE_ETH2_ASSERT) && ENABLE_ETH2_ASSERT
  #define ETH2_ASSERT(cond, format, arg...)  \
                                        do { if ( !(cond) ) printk(KERN_ERR __FILE__ ":%d:%s: " format "\n", __LINE__, __FUNCTION__, ##arg); } while ( 0 )
#else
  #define ETH2_ASSERT(cond, format, arg...)  \
                                        do { } while ( 0 )
#endif  //  defined(ENABLE_ETH2_ASSERT) && ENABLE_ETH2_ASSERT

/*
 *  Eth Mode
 */
#define MII_MODE                        1
#define REV_MII_MODE                    2

/*
 *  Default Eth Hardware Configuration
 */
#define CDM_CFG_DEFAULT                 0x00000000
#define SB_MST_SEL_DEFAULT              0x00000003
#if 0
  #define ENETS1_CFG_DEFAULT            (0x00007037 | (RX_HEAD_MAC_ADDR_ALIGNMENT << 18) | (1 << 30))
#else
  #define ENETS1_CFG_DEFAULT            (0x00007037 | (RX_HEAD_MAC_ADDR_ALIGNMENT << 18))
#endif
#define ENETF1_CFG_DEFAULT              0x00007010
#define ENETS1_PGCNT_DEFAULT            0x00020000
#define ENETS1_PKTCNT_DEFAULT           0x00000200
#define ENETF1_PGCNT_DEFAULT            0x00020000
#define ENETF1_PKTCNT_DEFAULT           0x00000200
#define ENETS1_COS_CFG_DEFAULT          0x00000002  // This enables multiple DMA channels when packets with VLAN is injected. COS mapping is through ETOP_IG_VLAN_COS; It is already preconfigured.
#define ENETS1_DBA_DEFAULT              0x00000400
#define ENETS1_CBA_DEFAULT              0x00000AE0
#define ENETF1_DBA_DEFAULT              0x00001600
#define ENETF1_CBA_DEFAULT              0x00001800

/*
 *  Constant Definition
 */
#define ETH2_WATCHDOG_TIMEOUT           (10 * HZ)
#define ETOP_MDIO_PHY1_ADDR             1
#define ETOP_MDIO_DELAY                 1
#define IDLE_CYCLE_NUMBER               30000
#define RX_TOTAL_CHANNEL_USED           MAX_RX_DMA_CHANNEL_NUMBER
#define TX_TOTAL_CHANNEL_USED           MAX_TX_DMA_CHANNEL_NUMBER

/*
 *  Tasklet Parameters
 */
#if 0
  #define TASKLET_MAX_EMPTY_LOOP        3
#endif
#define TASKLET_MAX_RX_CHECK_LOOP       100

/*
 *  DMA RX/TX Channel Parameters
 */
#define MAX_RX_DMA_CHANNEL_NUMBER       2
#define MAX_TX_DMA_CHANNEL_NUMBER       2
#define DMA_ALIGNMENT                   4

/*
 *  Ethernet Frame Definitions
 */
#define ETH_MAC_HEADER_LENGTH           18
#define ETH_CRC_LENGTH                  4
#define ETH_MIN_FRAME_LENGTH            64
#define ETH_MAX_FRAME_LENGTH            1518

/*
 *  RX Frame Definitions
 */
#define RX_HEAD_MAC_ADDR_ALIGNMENT      2
//#define RX_TAIL_CRC_LENGTH              ETH_CRC_LENGTH
#define RX_TAIL_CRC_LENGTH              0   //  PTM firmware does not have ethernet frame CRC
                                            //  The len in descriptor doesn't include ETH_CRC
                                            //  because ETH_CRC may not present in some configuration

/*
 *  TX Frame Definitions
 */
#define MAX_TX_PACKET_ALIGN_BYTES       3
#define MAX_TX_PACKET_PADDING_BYTES     3
#define MIN_TX_PACKET_LENGTH            (ETH_MIN_FRAME_LENGTH - ETH_CRC_LENGTH)

// testing purpose, EFM supposes CRC is already attached to the frame
//#define MIN_TX_PACKET_LENGTH            (ETH_MIN_FRAME_LENGTH)

/*
 *  EMA Settings
 */
#define EMA_CMD_BUF_LEN      0x0040
#define EMA_CMD_BASE_ADDR    (0x00001580 << 2)
#define EMA_DATA_BUF_LEN     0x0100
#define EMA_DATA_BASE_ADDR   (0x0000B00 << 2)
#define EMA_WRITE_BURST      0x2
#define EMA_READ_BURST       0x2

/*
 *  Bits Operation
 */
#define GET_BITS(x, msb, lsb)           (((x) & ((1 << ((msb) + 1)) - 1)) >> (lsb))
#define SET_BITS(x, msb, lsb, value)    (((x) & ~(((1 << ((msb) + 1)) - 1) ^ ((1 << (lsb)) - 1))) | (((value) & ((1 << (1 + (msb) - (lsb))) - 1)) << (lsb)))

/*
 *  FPI Configuration Bus Register and Memory Address Mapping
 */
#define DANUBE_PPE                      (KSEG1 + 0x1E180000)
#define PP32_DEBUG_REG_ADDR(x)          ((volatile u32*)(DANUBE_PPE + (((x) + 0x0000) << 2)))
#define PPM_INT_REG_ADDR(x)             ((volatile u32*)(DANUBE_PPE + (((x) + 0x0030) << 2)))
#define PP32_INTERNAL_RES_ADDR(x)       ((volatile u32*)(DANUBE_PPE + (((x) + 0x0040) << 2)))
//#define PPE_CLOCK_CONTROL_ADDR(x)       ((volatile u32*)(DANUBE_PPE + (((x) + 0x0100) << 2)))
#define CDM_CODE_MEMORY_RAM0_ADDR(x)    ((volatile u32*)(DANUBE_PPE + (((x) + 0x1000) << 2)))
#define CDM_CODE_MEMORY_RAM1_ADDR(x)    ((volatile u32*)(DANUBE_PPE + (((x) + 0x2000) << 2)))
#define PPE_REG_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x4000) << 2)))
#define PP32_DATA_MEMORY_RAM1_ADDR(x)   ((volatile u32*)(DANUBE_PPE + (((x) + 0x5000) << 2)))
#define PPM_INT_UNIT_ADDR(x)            ((volatile u32*)(DANUBE_PPE + (((x) + 0x6000) << 2)))
#define PPM_TIMER0_ADDR(x)              ((volatile u32*)(DANUBE_PPE + (((x) + 0x6100) << 2)))
#define PPM_TASK_IND_REG_ADDR(x)        ((volatile u32*)(DANUBE_PPE + (((x) + 0x6200) << 2)))
#define PPS_BRK_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x6300) << 2)))
#define PPM_TIMER1_ADDR(x)              ((volatile u32*)(DANUBE_PPE + (((x) + 0x6400) << 2)))
//#define SB_RAM0_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x8000) << 2)))
//#define SB_RAM1_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x8400) << 2)))
//#define SB_RAM2_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x8C00) << 2)))
//#define SB_RAM3_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x9600) << 2)))
#define SB_RAM0_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x8200) << 2)))
#define SB_RAM1_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x8C00) << 2)))
//#define SB_RAM2_ADDR(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0x9600) << 2)))
#define QSB_CONF_REG(x)                 ((volatile u32*)(DANUBE_PPE + (((x) + 0xC000) << 2)))

#define FW_SB_MAPPING(x)                ((volatile u32*)(DANUBE_PPE + ((((x) - 0x2000) + 0x8000) << 2)))

/*
 *  DWORD-Length of Memory Blocks
 */
#define PP32_DEBUG_REG_DWLEN            0x0030
#define PPM_INT_REG_DWLEN               0x0010
#define PP32_INTERNAL_RES_DWLEN         0x00C0
//#define PPE_CLOCK_CONTROL_DWLEN         0x0F00
#define CDM_CODE_MEMORY_RAM0_DWLEN      0x1000
#define CDM_CODE_MEMORY_RAM1_DWLEN      0x0800
#define PPE_REG_DWLEN                   0x1000
#define PP32_DATA_MEMORY_RAM1_DWLEN     0x0800
#define PPM_INT_UNIT_DWLEN              0x0100
#define PPM_TIMER0_DWLEN                0x0100
#define PPM_TASK_IND_REG_DWLEN          0x0100
#define PPS_BRK_DWLEN                   0x0100
#define PPM_TIMER1_DWLEN                0x0100
//#define SB_RAM0_DWLEN                   0x0400
//#define SB_RAM1_DWLEN                   0x0800
//#define SB_RAM2_DWLEN                   0x0A00
//#define SB_RAM3_DWLEN                   0x0400
#define SB_RAM0_DWLEN                   0x0A00
#define SB_RAM1_DWLEN                   0x0A00
//#define SB_RAM2_DWLEN                   0x1000
#define QSB_CONF_REG_DWLEN              0x0100

/*
 *  Share Buffer Registers
 */
#define SB_MST_PRI0                     PPE_REG_ADDR(0x0300)
#define SB_MST_PRI1                     PPE_REG_ADDR(0x0301)
#define SB_MST_SEL                      PPE_REG_ADDR(0x0304)

/*
 *    ETOP MDIO Registers
 */
#define ETOP_MDIO_CFG                   PPE_REG_ADDR(0x0600)
#define ETOP_MDIO_ACC                   PPE_REG_ADDR(0x0601)
#define ETOP_CFG                        PPE_REG_ADDR(0x0602)
#define ETOP_IG_VLAN_COS                PPE_REG_ADDR(0x0603)
#define ETOP_IG_DSCP_COSx(x)            PPE_REG_ADDR(0x0607 - ((x) & 0x03))
#define ETOP_IG_PLEN_CTRL0              PPE_REG_ADDR(0x0608)
//#define ETOP_IG_PLEN_CTRL1              PPE_REG_ADDR(0x0609)
#define ETOP_ISR                        PPE_REG_ADDR(0x060A)
#define ETOP_IER                        PPE_REG_ADDR(0x060B)
#define ETOP_VPID                       PPE_REG_ADDR(0x060C)
#define ENET_MAC_CFG(i)                 PPE_REG_ADDR(0x0610 + ((i) ? 0x40 : 0x00))
#define ENETS_DBA(i)                    PPE_REG_ADDR(0x0612 + ((i) ? 0x40 : 0x00))
#define ENETS_CBA(i)                    PPE_REG_ADDR(0x0613 + ((i) ? 0x40 : 0x00))
#define ENETS_CFG(i)                    PPE_REG_ADDR(0x0614 + ((i) ? 0x40 : 0x00))
#define ENETS_PGCNT(i)                  PPE_REG_ADDR(0x0615 + ((i) ? 0x40 : 0x00))
#define ENETS_PKTCNT(i)                 PPE_REG_ADDR(0x0616 + ((i) ? 0x40 : 0x00))
#define ENETS_BUF_CTRL(i)               PPE_REG_ADDR(0x0617 + ((i) ? 0x40 : 0x00))
#define ENETS_COS_CFG(i)                PPE_REG_ADDR(0x0618 + ((i) ? 0x40 : 0x00))
#define ENETS_IGDROP(i)                 PPE_REG_ADDR(0x0619 + ((i) ? 0x40 : 0x00))
#define ENETS_IGERR(i)                  PPE_REG_ADDR(0x061A + ((i) ? 0x40 : 0x00))
#define ENETS_MAC_DA0(i)                PPE_REG_ADDR(0x061B + ((i) ? 0x40 : 0x00))
#define ENETS_MAC_DA1(i)                PPE_REG_ADDR(0x061C + ((i) ? 0x40 : 0x00))
#define ENETF_DBA(i)                    PPE_REG_ADDR(0x0630 + ((i) ? 0x40 : 0x00))
#define ENETF_CBA(i)                    PPE_REG_ADDR(0x0631 + ((i) ? 0x40 : 0x00))
#define ENETF_CFG(i)                    PPE_REG_ADDR(0x0632 + ((i) ? 0x40 : 0x00))
#define ENETF_PGCNT(i)                  PPE_REG_ADDR(0x0633 + ((i) ? 0x40 : 0x00))
#define ENETF_PKTCNT(i)                 PPE_REG_ADDR(0x0634 + ((i) ? 0x40 : 0x00))
#define ENETF_HFCTRL(i)                 PPE_REG_ADDR(0x0635 + ((i) ? 0x40 : 0x00))
#define ENETF_TXCTRL(i)                 PPE_REG_ADDR(0x0636 + ((i) ? 0x40 : 0x00))
#define ENETF_VLCOS0(i)                 PPE_REG_ADDR(0x0638 + ((i) ? 0x40 : 0x00))
#define ENETF_VLCOS1(i)                 PPE_REG_ADDR(0x0639 + ((i) ? 0x40 : 0x00))
#define ENETF_VLCOS2(i)                 PPE_REG_ADDR(0x063A + ((i) ? 0x40 : 0x00))
#define ENETF_VLCOS3(i)                 PPE_REG_ADDR(0x063B + ((i) ? 0x40 : 0x00))
#define ENETF_EGCOL(i)                  PPE_REG_ADDR(0x063C + ((i) ? 0x40 : 0x00))
#define ENETF_EGDROP(i)                 PPE_REG_ADDR(0x063D + ((i) ? 0x40 : 0x00))
//#define ENET_MAC_DA0(i)                 PPE_REG_ADDR(0x063E + ((i) ? 0x40 : 0x00))
//#define ENET_MAC_DA1(i)                 PPE_REG_ADDR(0x063F + ((i) ? 0x40 : 0x00))

/*
 *  DPlus Registers
 */
#define DPLUS_RXCFG                     PPE_REG_ADDR(0x0712)
#define DPLUS_RXPGCNT                   PPE_REG_ADDR(0x0713)

/*
 *  EMA Registers
 */
#define EMA_CMDCFG                      PPE_REG_ADDR(0x0A00)
#define EMA_DATACFG                     PPE_REG_ADDR(0x0A01)
#define EMA_CMDCNT                      PPE_REG_ADDR(0x0A02)
#define EMA_DATACNT                     PPE_REG_ADDR(0x0A03)
#define EMA_ISR                         PPE_REG_ADDR(0x0A04)
#define EMA_IER                         PPE_REG_ADDR(0x0A05)
#define EMA_CFG                         PPE_REG_ADDR(0x0A06)
#define EMA_SUBID                       PPE_REG_ADDR(0x0A07)

/*
 *  EFM mode related HW register
 */
#define AT_CTRL                         PPE_REG_ADDR(0xD02)
#define AR_CTRL                         PPE_REG_ADDR(0xD08)
#define AT_IDLE0                        PPE_REG_ADDR(0xD28)
#define AT_IDLE1                        PPE_REG_ADDR(0xD29)
#define AR_IDLE0                        PPE_REG_ADDR(0xD74)
#define AR_IDLE1                        PPE_REG_ADDR(0xD75)
#define RFBI_CFG                        PPE_REG_ADDR(0x400)
#define SFSM_DBA0                       PPE_REG_ADDR(0x412)
#define SFSM_DBA1                       PPE_REG_ADDR(0x413)
#define SFSM_CBA0                       PPE_REG_ADDR(0x414)
#define SFSM_CBA1                       PPE_REG_ADDR(0x415)
#define SFSM_CFG0                       PPE_REG_ADDR(0x416)
#define SFSM_CFG1                       PPE_REG_ADDR(0x417)
#define FFSM_DBA0                       PPE_REG_ADDR(0x508)
#define FFSM_DBA1                       PPE_REG_ADDR(0x509)
#define FFSM_CFG0                       PPE_REG_ADDR(0x50A)
#define FFSM_CFG1                       PPE_REG_ADDR(0x50B)
#define FFSM_IDLE_HEAD_BC0              PPE_REG_ADDR(0x50E)
#define FFSM_IDLE_HEAD_BC1              PPE_REG_ADDR(0x50F)
//#define EMA_DATACFG                     PPE_REG_ADDR(0xA01)


/*
 *  Host-PPE Communication Data Address Mapping
 */
#define CFG_WAN_WRDES_DELAY             FW_SB_MAPPING(0x2404)
#define CFG_WRX_DMACH_ON                FW_SB_MAPPING(0x2405)
#define CFG_WTX_DMACH_ON                FW_SB_MAPPING(0x2406)
#define CFG_WRX_LOOK_BITTH              FW_SB_MAPPING(0x2407)

#define CFG_ETH_EFMTC_CRC               ((volatile struct eth_efmtc_crc_cfg *)FW_SB_MAPPING(0x2408))

#define WAN_MIB_TABLE_SIZE              (sizeof(struct efm_wan_mib_table) * 4)
#define PTM0_MIB_IDX                    0
#define PTMFAST_MIB_IDX                 1
#define PTM0HI_MIB_IDX                  2
#define PTMFASTHI_MIB_IDX               3

#define WAN_MIB_TABLE                   ((volatile struct efm_wan_mib_table*)      FW_SB_MAPPING(0x2440))
#define WRX_PORT_CONFIG(i)              ((volatile struct wrx_port_cfg_status*)    FW_SB_MAPPING(0x2500 + (i) * 18))
#define WRX_DMA_CHANNEL_CONFIG(i)       ((volatile struct wrx_dma_channel_config*) FW_SB_MAPPING(0x2640 + (i) * 7))
#define WTX_PORT_CONFIG(i)              ((volatile struct wtx_port_cfg*)           FW_SB_MAPPING(0x2710 + (i) * 30))
#define WTX_DMA_CHANNEL_CONFIG(i)       ((volatile struct wtx_dma_channel_config*) FW_SB_MAPPING(0x2711 + (i) * 30))

/*
 *  PP32 Debug Control Register
 */
#define PP32_DBG_CTRL                   PP32_DEBUG_REG_ADDR(0x0000)

//#define DBG_CTRL_START_SET(value)       ((value) ? (1 << 0) : 0)
//#define DBG_CTRL_STOP_SET(value)        ((value) ? (1 << 1) : 0)
//#define DBG_CTRL_STEP_SET(value)        ((value) ? (1 << 2) : 0)
#define DBG_CTRL_RESTART                0
#define DBG_CTRL_STOP                   1

/*
 *  PP32 Registers
 */
//#define DBG_BRK_SRC				PP32_DEBUG_REG_ADDR(0x0002)
//#define DBG_PC_MIN0				PP32_DEBUG_REG_ADDR(0x0010)
//#define DBG_PC_MIN1				PP32_DEBUG_REG_ADDR(0x0011)
//#define DBG_PC_MIN2				PP32_DEBUG_REG_ADDR(0x0012)
//#define DBG_PC_MIN3				PP32_DEBUG_REG_ADDR(0x0013)
//#define DBG_PC_MAX0				PP32_DEBUG_REG_ADDR(0x0014)
//#define DBG_PC_MAX1				PP32_DEBUG_REG_ADDR(0x0015)
//#define DBG_PC_MAX2				PP32_DEBUG_REG_ADDR(0x0016)
//#define DBG_PC_MAX3				PP32_DEBUG_REG_ADDR(0x0017)
//#define DBG_DATA_MIN0				PP32_DEBUG_REG_ADDR(0x0018)
//#define DBG_DATA_MIN1				PP32_DEBUG_REG_ADDR(0x0019)
//#define DBG_DATA_MAX0				PP32_DEBUG_REG_ADDR(0x001A)
//#define DBG_DATA_MAX1				PP32_DEBUG_REG_ADDR(0x001B)
//#define DBG_DATA_VAL0				PP32_DEBUG_REG_ADDR(0x001C)
//#define DBG_DATA_VAL1				PP32_DEBUG_REG_ADDR(0x001D)
//
//#define DBG_ENABLE_PC(x)			( *DBG_BRK_SRC = (*DBG_BRK_SRC) | (1 << (x)) )
//#define DBG_DISABLE_PC(x)			( *DBG_BRK_SRC = (*DBG_BRK_SRC) & (~ (1 << (x)) ))
//
//#define DBG_SETDAEN_0(val)			( *DBG_BRK_SRC = ((*DBG_BRK_SRC) & (~ ( 7 << 8 ))) | ( ((val) & 0x7) << 8 ))
//#define DBG_SETDAEN_1(val)			( *DBG_BRK_SRC = ((*DBG_BRK_SRC) & (~ ( 7 << 11))) | ( ((val) & 0x7) << 11))
//
//#define DBG_DABP_DISABLE			0x0
//#define DBG_DABP_READ				0x1
//#define DBG_DABP_WRITE				0x2
//#define DBG_DABP_READ_WRITE			0x3
//#define DBG_DABP_WRITE_VALUE        0x4
#define PP32_HALT_STAT                          PP32_DEBUG_REG_ADDR(0x0D00)
#define PP32_BREAKPOINT_REASONS                 PP32_DEBUG_REG_ADDR(0x0A00)

#define PP32_BRK_SRC                            PP32_DEBUG_REG_ADDR(0x0F00)

#define PP32_DBG_PC_MIN(i)                      PP32_DEBUG_REG_ADDR(0x0010 + (i))
#define PP32_DBG_PC_MAX(i)                      PP32_DEBUG_REG_ADDR(0x0014 + (i))
#define PP32_DBG_DATA_MIN(i)                    PP32_DEBUG_REG_ADDR(0x0018 + (i))
#define PP32_DBG_DATA_MAX(i)                    PP32_DEBUG_REG_ADDR(0x001A + (i))
#define PP32_DBG_DATA_VAL(i)                    PP32_DEBUG_REG_ADDR(0x001C + (i))

#define PP32_DBG_CUR_PC                         PP32_DEBUG_REG_ADDR(0x0F80)

#define PP32_DBG_TASK_NO                        PP32_DEBUG_REG_ADDR(0x0F81)

/*
 *    Code/Data Memory (CDM) Interface Configuration Register
 */
#define CDM_CFG                         PPE_REG_ADDR(0x0100)

#define CDM_CFG_RAM1                    GET_BITS(*CDM_CFG, 3, 2)
#define CDM_CFG_RAM0                    (*CDM_CFG & (1 << 1))

#define CDM_CFG_RAM1_SET(value)         SET_BITS(0, 3, 2, value)
#define CDM_CFG_RAM0_SET(value)         ((value) ? (1 << 1) : 0)

/*
 *  ETOP MDIO Configuration Register
 */
#define ETOP_MDIO_CFG_SMRST(value)      ((value) ? (1 << 13) : 0)
#define ETOP_MDIO_CFG_PHYA(i, value)    ((i) ? SET_BITS(0, 12, 8, (value)) : SET_BITS(0, 7, 3, (value)))
#define ETOP_MDIO_CFG_UMM(i, value)     ((value) ? ((i) ? (1 << 2) : (1 << 1)) : 0)

#define ETOP_MDIO_CFG_MASK(i)           (ETOP_MDIO_CFG_SMRST(1) | ETOP_MDIO_CFG_PHYA(i, 0x1F) | ETOP_MDIO_CFG_UMM(i, 1))

/*
 *  ETOP Configuration Register
 */
#define ETOP_CFG_MII1_OFF(value)        ((value) ? (1 << 3) : 0)
#define ETOP_CFG_REV_MII1_ON(value)     ((value) ? (1 << 4) : 0)
#define ETOP_CFG_TURBO_MII1_ON(value)   ((value) ? (1 << 5) : 0)
#define ETOP_CFG_SEN1_ON(value)         ((value) ? (1 << 7) : 0)
#define ETOP_CFG_FEN1_ON(value)         ((value) ? (1 << 9) : 0)
#define ETOP_CFG_TCKINV1_ON(value)      ((value) ? (1 << 11) : 0)

#define ETOP_CFG_MII1_MASK              (ETOP_CFG_MII1_OFF(1) | ETOP_CFG_REV_MII1_ON(1) | ETOP_CFG_TURBO_MII1_ON(1) | ETOP_CFG_SEN1_ON(1) | ETOP_CFG_FEN1_ON(1))

/*
 *  ETOP IG VLAN Priority CoS Mapping
 */
#define ETOP_IG_VLAN_COS_Px_SET(org, x, value)   \
                                        SET_BITS((org), ((x) << 1) + 1, (x) << 1, (value))

/*
 *  ETOP IG DSCP CoS Mapping Register x
 */
#define ETOP_IG_DSCP_COS_SET(x, value)  do {        \
                                            *ETOP_IG_DSCP_COSx((x) >> 4) = SET_BITS(*ETOP_IG_DSCP_COSx((x) >> 4), (((x) & 0x0F) << 1) + 1, ((x) & 0x0F) << 1, (value)); \
                                        } while ( 0 )

/*
 * ETOP Ingress Packet Length Control 0
 */
#define ETOP_IG_PLEN_CTRL0_UNDER_SET(value)     \
                                        (((value) & 0x7F) << 16)
#define ETOP_IG_PLEN_CTRL0_OVER_SET(value)      \
                                        ((value) & 0x3FFF)


/*
 *  ENet MAC Configuration Register
 */
#define ENET_MAC_CFG_CRC(i)             (*ENET_MAC_CFG(i) & (0x01 << 11))
#define ENET_MAC_CFG_DUPLEX(i)          (*ENET_MAC_CFG(i) & (0x01 << 2))
#define ENET_MAC_CFG_SPEED(i)           (*ENET_MAC_CFG(i) & (0x01 << 1))
#define ENET_MAC_CFG_LINK(i)            (*ENET_MAC_CFG(i) & 0x01)

#define ENET_MAC_CFG_CRC_OFF(i)         do { *ENET_MAC_CFG(i) &= ~(1 << 11); } while (0)
#define ENET_MAC_CFG_CRC_ON(i)          do { *ENET_MAC_CFG(i) |= 1 << 11; } while (0)
#define ENET_MAC_CFG_DUPLEX_HALF(i)     do { *ENET_MAC_CFG(i) &= ~(1 << 2); } while (0)
#define ENET_MAC_CFG_DUPLEX_FULL(i)     do { *ENET_MAC_CFG(i) |= 1 << 2; } while (0)
#define ENET_MAC_CFG_SPEED_10M(i)       do { *ENET_MAC_CFG(i) &= ~(1 << 1); } while (0)
#define ENET_MAC_CFG_SPEED_100M(i)      do { *ENET_MAC_CFG(i) |= 1 << 1; } while (0)
#define ENET_MAC_CFG_LINK_NOT_OK(i)     do { *ENET_MAC_CFG(i) &= ~1; } while (0)
#define ENET_MAC_CFG_LINK_OK(i)         do { *ENET_MAC_CFG(i) |= 1; } while (0)

/*
 *  ENets Configuration Register
 */
#define ENETS_CFG_VL2_SET               (1 << 29)
#define ENETS_CFG_VL2_CLEAR             ~(1 << 29)
#define ENETS_CFG_FTUC_SET              (1 << 28)
#define ENETS_CFG_FTUC_CLEAR            ~(1 << 28)
#define ENETS_CFG_DPBC_SET              (1 << 27)
#define ENETS_CFG_DPBC_CLEAR            ~(1 << 27)
#define ENETS_CFG_DPMC_SET              (1 << 26)
#define ENETS_CFG_DPMC_CLEAR            ~(1 << 26)

/*
 *  ENets Classification Configuration Register
 */
#define ENETS_COS_CFG_VLAN_SET          (1 << 1)
#define ENETS_COS_CFG_VLAN_CLEAR        ~(1 << 1)
#define ENETS_COS_CFG_DSCP_SET          (1 << 0)
#define ENETS_COS_CFG_DSCP_CLEAR        ~(1 << 0)

/*
 *  Mailbox IGU1 Registers
 */
#define MBOX_IGU1_ISRS                  PPE_REG_ADDR(0x0204)
#define MBOX_IGU1_ISRC                  PPE_REG_ADDR(0x0205)
#define MBOX_IGU1_ISR                   PPE_REG_ADDR(0x0206)
#define MBOX_IGU1_IER                   PPE_REG_ADDR(0x0207)

#define MBOX_IGU1_ISRS_SET(n)           (1 << (n))
#define MBOX_IGU1_ISRC_CLEAR(n)         (1 << (n))
#define MBOX_IGU1_ISR_ISR(n)            (*MBOX_IGU1_ISR & (1 << (n)))
#define MBOX_IGU1_IER_EN(n)             (*MBOX_IGU1_IER & (1 << (n)))
#define MBOX_IGU1_IER_EN_SET(n)         (1 << (n))

/*
 *  Mailbox IGU3 Registers
 */
#define MBOX_IGU3_ISRS                  PPE_REG_ADDR(0x0214)
#define MBOX_IGU3_ISRC                  PPE_REG_ADDR(0x0215)
#define MBOX_IGU3_ISR                   PPE_REG_ADDR(0x0216)
#define MBOX_IGU3_IER                   PPE_REG_ADDR(0x0217)

#define MBOX_IGU3_ISRS_SET(n)           (1 << (n))
#define MBOX_IGU3_ISRC_CLEAR(n)         (1 << (n))
#define MBOX_IGU3_ISR_ISR(n)            (*MBOX_IGU3_ISR & (1 << (n)))
#define MBOX_IGU3_IER_EN(n)             (*MBOX_IGU3_IER & (1 << (n)))
#define MBOX_IGU3_IER_EN_SET(n)         (1 << (n))


#define FW_VER_ID                      ((volatile struct fw_ver_id *)FW_SB_MAPPING(0x2401))

/*
 * ####################################
 *  Preparation of Hardware Emulation
 * ####################################
 */



/*
 * ####################################
 *              Data Type
 * ####################################
 */

  struct fw_ver_id {
    unsigned int    family              :4;
    unsigned int    fwtype              :4;
    unsigned int    interface           :4;
    unsigned int    fwmode              :4;
    unsigned int    major               :8;
    unsigned int    minor               :8;
  };



/*
 *  MIB Table Maintained by Firmware
 */
struct eth_mib_table {
    u32                     erx_dropdes_pdu;
    u32                     erx_pass_pdu;
    u32                     erx_pass_bytes;
    u32                     res1;
    u32                     etx_total_pdu;
    u32                     etx_total_bytes;
};

struct efm_wan_mib_table {
    u32                     wrx_correct_pdu;            // 0
    u32                     wrx_correct_pdu_bytes;      // 1
    u32                     wrx_tccrc_err_pdu;          // 2
    u32                     wrx_tccrc_err_pdu_bytes;    // 3
    u32                     wrx_ethcrc_err_pdu;         // 4
    u32                     wrx_ethcrc_err_pdu_bytes;   // 5
    u32                     wrx_nodesc_drop_pdu;        // 6
    u32                     wrx_len_violation_drop_pdu; // 7
    u32                     wrx_idle_bytes;             // 8
    u32                     wrx_nonidle_cw;             // 9
    u32                     wrx_idle_cw;                // A
    u32                     wrx_err_cw;                 // B
    u32                     wtx_total_pdu;              // C
    u32                     wtx_total_bytes;            // D
    u32                     res0;                       // E
    u32                     res1;                       // F
};

/*
 *  Host-PPE Communication Data Structure
 */
#if defined(__BIG_ENDIAN)
    struct wrx_port_cfg_status {
        /* 0h */
        unsigned int mfs            :16;
        unsigned int res0           :12;
        unsigned int dmach          :3;
        unsigned int res1           :1;

        /* 1h */
        unsigned int res2           :14;
        unsigned int local_state    :2;
        unsigned int res3           :15;
        unsigned int partner_state  :1;

    };

    struct wrx_dma_channel_config {
        /*  0h  */
        unsigned int    res3        :1;
        unsigned int    res4        :2;
        unsigned int    res5        :1;
        unsigned int    desba       :28;
//        unsigned int    res6        :3;
        /*  1h  */
        unsigned int    res1        :16;
        unsigned int    res2        :16;
        /*  2h  */
        unsigned int    deslen      :16;
        unsigned int    vlddes      :16;
    };

    struct wtx_port_cfg {
        /* 0h */
        unsigned int tx_cwth2       :8;
        unsigned int tx_cwth1       :8;
        unsigned int res0           :16;
    };


    struct wtx_dma_channel_config {
        /*  0h  */
        unsigned int    res3        :1;
        unsigned int    res4        :2;
        unsigned int    res5        :1;
        unsigned int    desba       :28;
//        unsigned int    res6        :3;

        /*  1h  */
        unsigned int    res1        :16;
        unsigned int    res2        :16;

        /*  2h  */
        unsigned int    deslen      :16;
        unsigned int    vlddes      :16;
    };

    struct eth_efmtc_crc_cfg {
        /*  0h  */
        unsigned int    res0                :6;
        unsigned int    tx_eth_crc_gen      :1;
        unsigned int    tx_tc_crc_gen       :1;
        unsigned int    tx_tc_crc_len       :8;
        unsigned int    res1                :5;
        unsigned int    rx_eth_crc_present  :1;
        unsigned int    rx_eth_crc_check    :1;
        unsigned int    rx_tc_crc_check     :1;
        unsigned int    rx_tc_crc_len       :8;
    };
#else   //  defined(__BIG_ENDIAN)

    struct wrx_port_cfg_status {
        /* 0h */
        unsigned int res1           :1;
        unsigned int dmach          :3;
        unsigned int res0           :12;
        unsigned int mfs            :16;

        /* 1h */
        unsigned int partner_state  :1;
        unsigned int res3           :15;
        unsigned int local_state    :2;
        unsigned int res2           :14;

    };


    struct wrx_dma_channel_config {
        /*  0h  */
//        unsigned int    res6        :3;
        unsigned int    desba       :28;
        unsigned int    res5        :1;
        unsigned int    res4        :2;
        unsigned int    res3        :1;
        /*  1h  */
        unsigned int    res2        :16;
        unsigned int    res1        :16;
        /*  2h  */
        unsigned int    vlddes      :16;
        unsigned int    deslen      :16;
    };

    struct wtx_port_cfg {
        /* 0h */
        unsigned int res0           :16;
        unsigned int tx_cwth1       :8;
        unsigned int tx_cwth2       :8;
    };

    struct wtx_dma_channel_config {
        /*  0h  */
 //       unsigned int    res6        :3;
        unsigned int    desba       :28;
        unsigned int    res5        :1;
        unsigned int    res4        :2;
        unsigned int    res3        :1;
        /*  1h  */
        unsigned int    res2        :16;
        unsigned int    res1        :16;
        /*  2h  */
        unsigned int    vlddes      :16;
        unsigned int    deslen      :16;
    };

    struct eth_efmtc_crc_cfg {
        /*  0h  */
        unsigned int    rx_tc_crc_len       :8;
        unsigned int    rx_tc_crc_check     :1;
        unsigned int    rx_eth_crc_check    :1;
        unsigned int    rx_eth_crc_present  :1;
        unsigned int    res1                :5;
        unsigned int    tx_tc_crc_len       :8;
        unsigned int    tx_tc_crc_gen       :1;
        unsigned int    tx_eth_crc_gen      :1;
        unsigned int    res0                :6;
    };
#endif  //  defined(__BIG_ENDIAN)

/*
 *  Descirptor
 */
#if defined(__BIG_ENDIAN)
    struct rx_descriptor {
        /*  0 - 3h  */
        unsigned int    own         :1;
        unsigned int    c           :1;
        unsigned int    sop         :1;
        unsigned int    eop         :1;
        unsigned int    res1        :3;
        unsigned int    byteoff     :2;
        unsigned int    res2        :2;
        unsigned int    id          :4;
        unsigned int    err         :1;
        unsigned int    datalen     :16;
        /*  4 - 7h  */
        unsigned int    res3        :4;
        unsigned int    dataptr     :28;
    };

    struct tx_descriptor {
        /*  0 - 3h  */
        unsigned int    own         :1;
        unsigned int    c           :1;
        unsigned int    sop         :1;
        unsigned int    eop         :1;
        unsigned int    byteoff     :5;
        unsigned int    res1        :5;
        unsigned int    iscell      :1;
        unsigned int    clp         :1;
        unsigned int    datalen     :16;
        /*  4 - 7h  */
        unsigned int    res2        :4;
        unsigned int    dataptr     :28;
    };
#else
    struct rx_descriptor {
        /*  4 - 7h  */
        unsigned int    dataptr     :28;
        unsigned int    res3        :4;
        /*  0 - 3h  */
        unsigned int    datalen     :16;
        unsigned int    err         :1;
        unsigned int    id          :4;
        unsigned int    res2        :2;
        unsigned int    byteoff     :2;
        unsigned int    res1        :3;
        unsigned int    eop         :1;
        unsigned int    sop         :1;
        unsigned int    c           :1;
        unsigned int    own         :1;
    };

    struct tx_descriptor {
        /*  4 - 7h  */
        unsigned int    dataptr     :28;
        unsigned int    res2        :4;
        /*  0 - 3h  */
        unsigned int    datalen     :16;
        unsigned int    clp         :1;
        unsigned int    iscell      :1;
        unsigned int    res1        :5;
        unsigned int    byteoff     :5;
        unsigned int    eop         :1;
        unsigned int    sop         :1;
        unsigned int    c           :1;
        unsigned int    own         :1;
    };
#endif  //  defined(__BIG_ENDIAN)

/*
 *  Eth2 Structure
 */
struct eth2_dev {
    u32                 rx_buffer_size;             /*  max memory allocated for a RX packet    */

    u32                 rx_descriptor_number;       /*  number of RX descriptors                */
    u32                 tx_descriptor_number;       /*  number of TX descriptors                */

    u32                 write_descriptor_delay;     /*  delay on descriptor write path          */

    void                *rx_descriptor_addr;        /*  base address of memory allocated for    */
                                                    /*  RX descriptors                          */
    struct rx_descriptor
                        *rx_descriptor_base;        /*  base address of RX descriptors          */
    struct rx_descriptor
                        *rx_desc_ch_base[RX_TOTAL_CHANNEL_USED];

//    struct rx_descriptor
//                        rx_desc_ch_base_dbg_copy[RX_TOTAL_CHANNEL_USED * 24];

    int                 rx_desc_read_pos[RX_TOTAL_CHANNEL_USED];        /*  first RX descriptor */
                                                                        /*  to be read          */

    void                *tx_descriptor_addr;        /*  base address of memory allocated for    */
                                                    /*  TX descriptors                          */
    struct tx_descriptor
                        *tx_descriptor_base;        /*  base address of TX descriptors          */
    int                 tx_desc_alloc_pos[TX_TOTAL_CHANNEL_USED];       /*  first TX descriptor */
                                                                        /*  could be allocated  */
    struct sk_buff      **tx_skb_pointers;          /*  base address of TX sk_buff pointers     */

    u32                 rx_drop_counter;            /*  number of dropped RX packet             */
//    struct net_device_stats
//                        stats;                      /*  net device stats                        */

#if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL
    int                 fc_bit;                     /*  net wakeup callback                     */
#endif

    u32                 rx_irq;                     /*  RX channel IRQ enabled                  */
    u32                 tx_irq;                     /*  TX channel IRQ enabled                  */
    int                 irq_handling_flag;
#if 0
    int                 empty_loop;                 /*  tasklet empty loop counter              */
#endif

#if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
    u32                 rx_success;                 /*  number of packet succeeded to push up   */
    u32                 rx_fail;                    /*  number of packet failed in pushing up   */
    u32                 tx_success;                 /*  number of packet succeeded to transmit  */

    u32                 rx_driver_level_error;      /*  number of rx error at driver level      */
    u32                 rx_driver_level_drop;       /*  number of rx drop at driver level       */
    u32                 tx_driver_level_drop;       /*  number of tx drop at driver level       */

    u32                 rx_desc_update_wait_loop;
    u32                 tx_desc_update_wait_loop;
#endif
};


/*
 * ####################################
 *             Declaration
 * ####################################
 */

/*
 *  Network Operations
 */
static int eth2_init(struct net_device *);
static struct net_device_stats *eth2_get_stats(struct net_device *);
static int eth2_open(struct net_device *);
static int eth2_stop(struct net_device *);
static int eth2_hard_start_xmit(struct sk_buff *, struct net_device *);
static int eth2_set_mac_address(struct net_device *, void *);
static int eth2_ioctl(struct net_device *, struct ifreq *, int);
static int eth2_change_mtu(struct net_device *, int);
static void eth2_tx_timeout(struct net_device *);

/*
 *  Channel Determination
 */
static INLINE int pick_up_tx_channel(struct sk_buff *, struct net_device *);

/*
 *  Buffer manage functions
 */
static INLINE struct sk_buff* alloc_skb_rx(void);
static INLINE struct sk_buff* alloc_skb_tx(unsigned int);
static INLINE int alloc_tx_channel(int, int *);

/*
 *  Mailbox handler and signal function
 */
#if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL
  static void eth2_xon(struct net_device *);
#endif
#if !defined(DISABLE_TASKLET) || !DISABLE_TASKLET
  static void do_eth2_tasklet(unsigned long);
#endif
static INLINE int mailbox_rx_irq_handler(unsigned int);
//static void mailbox_irq_handler(int, void *, struct pt_regs *);
static irqreturn_t mailbox_irq_handler(int, void *);
static INLINE void mailbox_signal(unsigned int, int);

/*
 *  ioctl help functions
 */
static INLINE int ethtool_ioctl(struct net_device *, struct ifreq *);
static INLINE void set_vlan_cos(struct vlan_cos_req *);
static INLINE void set_dscp_cos(struct dscp_cos_req *);

/*
 *  Init & clean-up functions
 */
#if defined(ENABLE_PROBE_TRANSCEIVER) && ENABLE_PROBE_TRANSCEIVER
  static INLINE int probe_transceiver(struct net_device *);
#endif
#ifdef MODULE
  static INLINE void reset_ppe(void);
#endif
static INLINE int init_local_variables(void);
static INLINE void check_parameters(void);
static INLINE int init_eth2_dev(void);
static INLINE void clear_share_buffer(void);
static INLINE void config_ppe_hw(void);
static INLINE void init_tables(void);
static INLINE void clear_eth2_dev(void);

/*
 *  PP32 specific init functions
 */
static INLINE void init_ema(void);
static INLINE void init_chip(int);
static INLINE int pp32_download_code(const u32 *, unsigned int, const u32 *, unsigned int);
static INLINE int pp32_specific_init(void *);
static INLINE int pp32_start(void);
static INLINE void pp32_stop(void);
static INLINE void ppe_bp_setup(void);
/*
 *  Proc File
 */
static INLINE void proc_file_create(void);
static INLINE void proc_file_delete(void);
static int proc_read_stats(char *, char **, off_t, int, int *, void *);
static int proc_read_mib(char *, char **, off_t, int, int *, void *);
static int proc_write_mib(struct file *, const char *, unsigned long, void *);
#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
  static int proc_read_dbg(char *, char **, off_t, int, int *, void *);
  static int proc_write_dbg(struct file *, const char *, unsigned long, void *);
#endif
#if defined(DEBUG_MEM_PROC) && DEBUG_MEM_PROC
  static int proc_write_mem(struct file *, const char *, unsigned long, void *);
#endif
static INLINE int stricmp(const char *, const char *);
static INLINE int strincmp(const char *, const char *, int);
static INLINE int get_token(char **, char **, int *, int *);
static INLINE int get_number(char **, int *, int);
static INLINE void ignore_space(char **, int *);

/*
 *  Debug functions
 */
#if (defined(DEBUG_DUMP_RX_SKB) && DEBUG_DUMP_RX_SKB) || (defined(DEBUG_DUMP_TX_SKB) && DEBUG_DUMP_TX_SKB)
  static INLINE void dump_skb(struct sk_buff *, int);
#endif  //  (defined(DEBUG_DUMP_RX_SKB) && DEBUG_DUMP_RX_SKB) || (defined(DEBUG_DUMP_TX_SKB) && DEBUG_DUMP_TX_SKB)
#if defined(DEBUG_DUMP_ETOP_REGISTER) && DEBUG_DUMP_ETOP_REGISTER
  static INLINE void dump_etop0_reg(void);
  static INLINE void dump_etop1_reg(void);
  static INLINE void dump_etop_reg(void);
#endif  //  defined(DEBUG_DUMP_ETOP_REGISTER) && DEBUG_DUMP_ETOP_REGISTER



/*
 * ####################################
 *            Local Variable
 * ####################################
 */

static struct net_device eth2_net_dev[] = {
    {
        name:   "ptm0",
        init:   eth2_init,
    },
    //{
    //    name:   "ptmhi0",
    //    init:   eth2_init,
    //},
    {
        name:   "ptmfast0",
        init:   eth2_init,
    },
    //{
    //    name:   "ptmfasthi0",
    //    init:   eth2_init,
    //}
};

static struct net_device_stats eth2_net_stats[sizeof(eth2_net_dev) / sizeof(*eth2_net_dev)] = {{0}};

static struct eth2_dev eth2_dev;

static u8 my_ethaddr[MAX_ADDR_LEN] = {0x00, 0x02, 0x03, 0x04, 0xDB, 0x30, 0x00, 0x00};

static struct proc_dir_entry *g_eth2_proc_dir;

#if !defined(DISABLE_TASKLET) || !DISABLE_TASKLET
  static DECLARE_TASKLET(eth2_tasklet, do_eth2_tasklet, 0);
#endif

#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
  static u32 dbg_enable;
  static int eth2_tx_quota = -1;
//  static int eth2_tx_crc_append = 0;
#endif


/*
 * ####################################
 *           Global Variable
 * ####################################
 */


/*
 * ####################################
 *            Local Function
 * ####################################
 */

static int eth2_init(struct net_device *dev)
{
    u64 retval;
    int ndev = dev - eth2_net_dev;
    int i;

//    dbg("invoked");

    ETH2_ASSERT(ndev >= 0 && ndev < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev), "ndev = %d (wrong value)", ndev);

    ether_setup(dev);   /*  assign some members */

    /*  hook network operations */
    dev->get_stats       = eth2_get_stats;
    dev->open            = eth2_open;
    dev->stop            = eth2_stop;
    dev->hard_start_xmit = eth2_hard_start_xmit;
    dev->set_mac_address = eth2_set_mac_address;
    dev->do_ioctl        = eth2_ioctl;
    dev->change_mtu      = eth2_change_mtu;
    dev->tx_timeout      = eth2_tx_timeout;
    dev->watchdog_timeo  = ETH2_WATCHDOG_TIMEOUT;
    dev->priv            = &eth2_dev;

    SET_MODULE_OWNER(dev);

    /*  read MAC address from the MAC table and put them into device    */
    retval = 0;
    for ( i = 0; i < 6; i++ )
        retval += MY_ETHADDR[i];
    if ( retval == 0 )
    {
        /*  ethaddr not set in u-boot   */
        dev->dev_addr[0] = 0x00;
		dev->dev_addr[1] = 0x20;
		dev->dev_addr[2] = 0xda;
		dev->dev_addr[3] = 0x86;
		dev->dev_addr[4] = 0x23;
		dev->dev_addr[5] = 0x74 + 1 + ndev; /*  eth2    */
	}
	else
	{
	    for ( i = 0; i < 6; i++ )
	        dev->dev_addr[i] = MY_ETHADDR[i];
	    dev->dev_addr[5] += 1 + ndev;       /*  eth2    */
	}

	return 0;   //  Qi Ming set 1 here in Switch driver
}

static struct net_device_stats *eth2_get_stats(struct net_device *dev)
{
    int ndev;
//    dbg("invoked");

    if ( dev->priv != &eth2_dev )
        return NULL;

    ndev = dev - eth2_net_dev;
    ETH2_ASSERT(ndev >= 0 && ndev < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev), "ndev = %d (wrong value)", ndev);

    eth2_net_stats[ndev].rx_packets  = WAN_MIB_TABLE[ndev].wrx_correct_pdu;
    eth2_net_stats[ndev].rx_bytes    = WAN_MIB_TABLE[ndev].wrx_correct_pdu_bytes;
    eth2_net_stats[ndev].rx_errors   = WAN_MIB_TABLE[ndev].wrx_tccrc_err_pdu + WAN_MIB_TABLE[ndev].wrx_ethcrc_err_pdu;
    eth2_net_stats[ndev].rx_dropped  = WAN_MIB_TABLE[ndev].wrx_nodesc_drop_pdu + WAN_MIB_TABLE[ndev].wrx_len_violation_drop_pdu + eth2_dev.rx_drop_counter;

    eth2_net_stats[ndev].tx_packets  = WAN_MIB_TABLE[ndev].wtx_total_pdu;
    eth2_net_stats[ndev].tx_bytes    = WAN_MIB_TABLE[ndev].wtx_total_bytes;
    //eth2_net_stats[ndev].tx_errors  += *ENETF_EGCOL(1);               /*  eth2_dev.stats.tx_errors is also updated in function eth2_hard_start_xmit   */
    //eth2_net_stats[ndev].tx_dropped += *ENETF_EGDROP(1);              /*  eth2_dev.stats.tx_dropped is updated in function eth2_hard_start_xmit       */

    return eth2_net_stats + ndev;
}

static int eth2_open(struct net_device *dev)
{
    dbg("invoked");

//    MOD_INC_USE_COUNT;

#if defined(ENABLE_PROBE_TRANSCEIVER) && ENABLE_PROBE_TRANSCEIVER
    if ( !probe_transceiver(dev) )
    {
        printk("%s cannot work because of hardware problem\n", dev->name);
//        MOD_DEC_USE_COUNT;
        return -1;
    }
#endif  //  defined(ENABLE_PROBE_TRANSCEIVER) && ENABLE_PROBE_TRANSCEIVER
    dbg(" %s", dev->name);

#if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL
    if ( (eth2_dev.fc_bit = netdev_register_fc(dev, eth2_xon)) == 0 )
    {
        printk("Hardware Flow Control register fails\n");
    }
#endif  //  defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL

    //  enable RX DMA
    *CFG_WRX_DMACH_ON = (1 << RX_TOTAL_CHANNEL_USED) - 1;

    netif_start_queue(dev);

    return 0;
}

static int eth2_stop(struct net_device *dev)
{
    dbg("invoked");

    //  disable RX DMA
    *CFG_WRX_DMACH_ON = 0x00;

#if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL
    if ( eth2_dev.fc_bit )
        netdev_unregister_fc(eth2_dev.fc_bit);
#endif  //  defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL

    netif_stop_queue(dev);
//    MOD_DEC_USE_COUNT;

    return 0;
}

//#define CRC_LE_BITS 1
//#define CRC_BE_BITS 1
#include <linux/crc32.h>


#if 0
/*
  * There are multiple 16-bit CRC polynomials in common use, but this is
  * *the* standard CRC-32 polynomial, first popularized by Ethernet.
  * x^32+x^26+x^23+x^22+x^16+x^12+x^11+x^10+x^8+x^7+x^5+x^4+x^2+x^1+x^0
  */
#define CRCPOLY_LE 0xedb88320
#define CRCPOLY_BE 0x04c11db7

/**
  * crc32_le() - Calculate bitwise little-endian Ethernet AUTODIN II CRC32
  * @crc - seed value for computation.  ~0 for Ethernet, sometimes 0 for
  *        other uses, or the previous crc32 value if computing incrementally.
  * @p   - pointer to buffer over which CRC is run
  * @len - length of buffer @p
  *
  */
static u32 crc32_le(u32 crc, unsigned char const *p, size_t len)
{
        int i;
        printk("crc32_le() BEFORE: crc = %08x, p = %08x, len = %d\n", crc, p, len);
        while (len--) {
                crc ^= *p++;
                for (i = 0; i < 8; i++)
                        crc = (crc >> 1) ^ ((crc & 1) ? CRCPOLY_LE : 0);
        }
        printk("crc32_le() AFTER: crc = %08x, p = %08x, len = %d\n", crc, p, len);
        return crc;
}

static u32 bitreverse(u32 x)
{
        x = (x >> 16) | (x << 16);
        x = (x >> 8 & 0x00ff00ff) | (x << 8 & 0xff00ff00);
        x = (x >> 4 & 0x0f0f0f0f) | (x << 4 & 0xf0f0f0f0);
        x = (x >> 2 & 0x33333333) | (x << 2 & 0xcccccccc);
        x = (x >> 1 & 0x55555555) | (x << 1 & 0xaaaaaaaa);
        return x;
}

#define ether_crc(data, length)    bitreverse(crc32_le(~0, data, length))
#endif


//  TODO: OAM Support
struct ptm_ch {
    int ch;
};
int danube_ppe_send_eoam(struct ptm_ch *ch, void *frame, int flags)
{
    printk(KERN_ERR __FILE__ ":%d:%s: not implemented yet\n", __LINE__, __FUNCTION__);
    return -1;
}

static int eth2_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
//    static int called_num = 0;
    int ret = 0;
    int ndev = dev - eth2_net_dev;
    int ch;
    int f_full;
    int desc_base;
    register struct tx_descriptor reg_desc;
    struct tx_descriptor *desc;

    ETH2_ASSERT(ndev >= 0 && ndev < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev), "ndev = %d (wrong value)", ndev);

#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
    // tx quota control
    if(eth2_tx_quota == 0) {
        dev_kfree_skb_any(skb);
        eth2_net_stats[ndev].tx_dropped++;
        return 0;
    }

    if(eth2_tx_quota > 0)
        eth2_tx_quota --;
#endif


    /*  By default, channel 0 is used to transmit data. If want to take full  *
     *  advantage of 4 channels, please implement a strategy to select a      *
     *  channel for transmission.                                             */
    ch = pick_up_tx_channel(skb, dev);

    /*  allocate descriptor */
    desc_base = alloc_tx_channel(ch, &f_full);
    if ( desc_base < 0 )
    {
        u32 sys_flag;

//        printk("*** dscriptor used up!!!\n");

        dev->trans_start = jiffies;
        //netif_stop_queue(eth2_net_dev + ndev);
        netif_stop_queue(dev);

        local_irq_save(sys_flag);
        eth2_dev.tx_irq |= 1 << ch;
        *MBOX_IGU1_ISRC = 1 << (ch + 16);
        *MBOX_IGU1_IER = (eth2_dev.tx_irq << 16) | eth2_dev.rx_irq;
        local_irq_restore(sys_flag);


#if 0
        ret = -EIO;
#else
        ret = 0;
#endif
        goto ALLOC_TX_CONNECTION_FAIL;
    }
    else if ( f_full )
    {
        u32 sys_flag;

        dev->trans_start = jiffies;
        //netif_stop_queue(eth2_net_dev + ndev);
        netif_stop_queue(dev);

        local_irq_save(sys_flag);
        eth2_dev.tx_irq |= 1 << ch;
        *MBOX_IGU1_ISRC = 1 << (ch + 16);
        *MBOX_IGU1_IER = (eth2_dev.tx_irq << 16) | eth2_dev.rx_irq;
        local_irq_restore(sys_flag);

   //     printk("** f_full = 1\n");
    }

    if ( eth2_dev.tx_skb_pointers[desc_base] )
        dev_kfree_skb_any(eth2_dev.tx_skb_pointers[desc_base]);

    desc = &eth2_dev.tx_descriptor_base[desc_base];

    /*  load descriptor from memory */
    reg_desc = *desc;

    /*  if data pointer is not aligned, allocate new sk_buff    */
    if ( ((u32)skb->data & ~(DMA_ALIGNMENT - 1)) < (u32)skb->head )
    {
        struct sk_buff *new_skb;

        dbg("skb->data is not aligned");

        new_skb = alloc_skb_tx(skb->len);
        if ( new_skb == NULL )
        {
#if 0
            ret = -ENOMEM;
#else
            ret = 0;
#endif
            eth2_dev.tx_skb_pointers[desc_base] = NULL;
            goto ALLOC_SKB_TX_FAIL;
        }
        skb_put(new_skb, skb->len);
        memcpy(new_skb->data, skb->data, skb->len);
        dev_kfree_skb_any(skb);
        skb = new_skb;
    }

    /*  if packet length is less than 60, pad it to 60 bytes    */
//    if ( skb->len < MIN_TX_PACKET_LENGTH )
//        skb_put(skb, MIN_TX_PACKET_LENGTH - skb->len);
    ETH2_ASSERT(skb->len >= MIN_TX_PACKET_LENGTH, "packet length (%d) is less than 60 bytes\n", skb->len);

    reg_desc.dataptr = (u32)skb->data >> 2;
    reg_desc.datalen = skb->len < MIN_TX_PACKET_LENGTH ? MIN_TX_PACKET_LENGTH : skb->len;
    reg_desc.byteoff = (u32)skb->data & (DMA_ALIGNMENT - 1);
    reg_desc.own     = 1;
    reg_desc.c       = 1;

    // reg_desc.datalen += ETH_CRC_LENGTH;
#if 0
    if(eth2_tx_crc_append) {
      u32 crc;
      int pkt_len;
      struct sk_buff * expanded_skb;

      //printk("\nhead = %08x\ndata = %08x\ntail = %08x\nend  = %08x\n",
      //      skb->head, skb->data, skb->tail, skb->end);

      pkt_len = (skb->len < MIN_TX_PACKET_LENGTH ? MIN_TX_PACKET_LENGTH : skb->len ) + ETH_CRC_LENGTH;


      if( skb->end - skb->data < pkt_len) {
          // expand data buffer to accomerdate CRC
            if(!(expanded_skb = skb_copy_expand(skb, skb_headroom(skb), (pkt_len - skb->len + 31) /16 * 16, GFP_ATOMIC )) ) {
                printk("ETH2 appending CRC Error! cannot alloce new data buffer to accomerdate CRC\n");
                goto ALLOC_TX_CONNECTION_FAIL;
            }
            dev_kfree_skb_any(skb);
            skb = expanded_skb;
            //printk("After reallocate data buffer:\nhead = %08x\ndata = %08x\ntail = %08x\nend  = %08x\n",
            //        skb->head, skb->data, skb->tail, skb->end);
            //printk("pkt_len = %d, skb->len = %d\n", pkt_len, skb->len);

            {
                // clear appended data
                unsigned char * p = (unsigned char *)skb->tail;
                int i;

                for(i = 0; i < pkt_len - skb-> len; i ++) {
                    *p ++ = 0;
                }

            }
            skb_put(skb, pkt_len - skb->len);

            //printk("After put:\nhead = %08x\ndata = %08x\ntail = %08x\nend  = %08x\n",
            //        skb->head, skb->data, skb->tail, skb->end);
            //printk("pkt_len = %d, skb->len = %d\n", pkt_len, skb->len);

      }
      crc = ether_crc_le(pkt_len, (unsigned char const *)skb->data);
      printk("ether_crc_le = %08x\n", crc);
      crc = ether_crc(pkt_len, (unsigned char const *)skb->data);
      printk("ether_crc = %08x\n", crc);
      * (u32 *) ( (unsigned char * ) skb->tail  - ETH_CRC_LENGTH) = crc;

      reg_desc.datalen = pkt_len;
    }
#endif

    /*  update descriptor send pointer  */
    eth2_dev.tx_skb_pointers[desc_base] = skb;

#if defined(DEBUG_DUMP_TX_SKB) && DEBUG_DUMP_TX_SKB
    dump_skb(skb, 0);
#endif  //  defined(DEBUG_DUMP_TX_SKB) && DEBUG_DUMP_TX_SKB

    /*  write discriptor to memory and write back cache */
    *desc = reg_desc;
    dma_cache_wback((unsigned long)skb->data, skb->len);

    /*  signal firmware */
    dev->trans_start = jiffies;
    mailbox_signal(ch, 1);

#if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
    eth2_dev.tx_success++;
#endif

    return 0;

ALLOC_SKB_TX_FAIL:
    dbg("ALLOC_SKB_TX_FAIL");
ALLOC_TX_CONNECTION_FAIL:
    dev_kfree_skb_any(skb);
//    eth2_dev.stats.tx_errors++;
    eth2_net_stats[ndev].tx_dropped++;
    return ret;
}

static int eth2_set_mac_address(struct net_device *dev, void *p)
{
    struct sockaddr *addr = (struct sockaddr *)p;

    dbg("invoked");

    printk("%s: change MAC from %02X:%02X:%02X:%02X:%02X:%02X to %02X:%02X:%02X:%02X:%02X:%02X\n", dev->name,
        dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2], dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5],
        addr->sa_data[0], addr->sa_data[1], addr->sa_data[2], addr->sa_data[3], addr->sa_data[4], addr->sa_data[5]);

    memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);

    return 0;
}

static int eth2_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
    int ndev;

    dbg("invoked");

    ndev = dev - eth2_net_dev;
    ETH2_ASSERT(ndev >= 0 && ndev < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev), "ndev = %d (wrong value)", ndev);

    switch ( cmd )
    {
#if 1
    case IFX_PTM_MIB_CW_GET:
        ((PTM_CW_IF_ENTRY_T *)ifr->ifr_data)->ifRxNoIdleCodewords   = WAN_MIB_TABLE[ndev].wrx_nonidle_cw;
        ((PTM_CW_IF_ENTRY_T *)ifr->ifr_data)->ifRxIdleCodewords     = WAN_MIB_TABLE[ndev].wrx_idle_cw;
        ((PTM_CW_IF_ENTRY_T *)ifr->ifr_data)->ifRxCodingViolation   = WAN_MIB_TABLE[ndev].wrx_err_cw;
        ((PTM_CW_IF_ENTRY_T *)ifr->ifr_data)->ifTxNoIdleCodewords   = 0;
        ((PTM_CW_IF_ENTRY_T *)ifr->ifr_data)->ifTxIdleCodewords     = 0;
        break;
    case IFX_PTM_MIB_FRAME_GET:
        ((PTM_FRAME_MIB_T *)ifr->ifr_data)->RxCorrect   = WAN_MIB_TABLE[ndev].wrx_correct_pdu;
        ((PTM_FRAME_MIB_T *)ifr->ifr_data)->TC_CrcError = WAN_MIB_TABLE[ndev].wrx_tccrc_err_pdu;
        ((PTM_FRAME_MIB_T *)ifr->ifr_data)->RxDropped   = WAN_MIB_TABLE[ndev].wrx_nodesc_drop_pdu + WAN_MIB_TABLE[ndev].wrx_len_violation_drop_pdu;
        ((PTM_FRAME_MIB_T *)ifr->ifr_data)->TxSend      = WAN_MIB_TABLE[ndev].wtx_total_pdu;
        break;
    case IFX_PTM_CFG_GET:
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxEthCrcPresent = CFG_ETH_EFMTC_CRC->rx_eth_crc_present;
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxEthCrcCheck   = CFG_ETH_EFMTC_CRC->rx_eth_crc_check;
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxTcCrcCheck    = CFG_ETH_EFMTC_CRC->rx_tc_crc_check;
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxTcCrcLen      = CFG_ETH_EFMTC_CRC->rx_tc_crc_len;
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxEthCrcGen     = CFG_ETH_EFMTC_CRC->tx_eth_crc_gen;
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxTcCrcGen      = CFG_ETH_EFMTC_CRC->tx_tc_crc_gen;
        ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxTcCrcLen      = CFG_ETH_EFMTC_CRC->tx_tc_crc_len;
        break;
    case IFX_PTM_CFG_SET:
        CFG_ETH_EFMTC_CRC->rx_eth_crc_present   = ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxEthCrcPresent ? 1 : 0;
        CFG_ETH_EFMTC_CRC->rx_eth_crc_check     = ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxEthCrcCheck ? 1 : 0;
        if ( ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxTcCrcCheck && (((IFX_PTM_CFG_T *)ifr->ifr_data)->RxTcCrcLen == 16 || ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxTcCrcLen == 32) )
        {
            CFG_ETH_EFMTC_CRC->rx_tc_crc_check  = 1;
            CFG_ETH_EFMTC_CRC->rx_tc_crc_len    = ((IFX_PTM_CFG_T *)ifr->ifr_data)->RxTcCrcLen;
        }
        else
        {
            CFG_ETH_EFMTC_CRC->rx_tc_crc_check  = 0;
            CFG_ETH_EFMTC_CRC->rx_tc_crc_len    = 0;
        }
        CFG_ETH_EFMTC_CRC->tx_eth_crc_gen       = ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxEthCrcGen ? 1 : 0;
        if ( ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxTcCrcGen && (((IFX_PTM_CFG_T *)ifr->ifr_data)->TxTcCrcLen == 16 || ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxTcCrcLen == 32) )
        {
            CFG_ETH_EFMTC_CRC->tx_tc_crc_gen    = 1;
            CFG_ETH_EFMTC_CRC->tx_tc_crc_len    = ((IFX_PTM_CFG_T *)ifr->ifr_data)->TxTcCrcLen;
        }
        else
        {
            CFG_ETH_EFMTC_CRC->tx_tc_crc_gen    = 0;
            CFG_ETH_EFMTC_CRC->tx_tc_crc_len    = 0;
        }
        break;
#else
    /*
     *  following are standard implementation for ETH2 ethernet device
     */
    case SIOCETHTOOL:
        return ethtool_ioctl(dev, ifr);
    case SET_VLAN_COS:
        {
            struct vlan_cos_req vlan_cos_req;

            if ( copy_from_user(&vlan_cos_req, ifr->ifr_data, sizeof(struct vlan_cos_req)) )
                return -EFAULT;
            set_vlan_cos(&vlan_cos_req);
        }
        break;
    case SET_DSCP_COS:
        {
            struct dscp_cos_req dscp_cos_req;

            if ( copy_from_user(&dscp_cos_req, ifr->ifr_data, sizeof(struct dscp_cos_req)) )
                return -EFAULT;
            set_dscp_cos(&dscp_cos_req);
        }
        break;
    case ENABLE_VLAN_CLASSIFICATION:
        *ENETS_COS_CFG(1) |= ENETS_COS_CFG_VLAN_SET;    break;
    case DISABLE_VLAN_CLASSIFICATION:
        *ENETS_COS_CFG(1) &= ENETS_COS_CFG_VLAN_CLEAR;  break;
    case ENABLE_DSCP_CLASSIFICATION:
        *ENETS_COS_CFG(1) |= ENETS_COS_CFG_DSCP_SET;    break;
    case DISABLE_DSCP_CLASSIFICATION:
        *ENETS_COS_CFG(1) &= ENETS_COS_CFG_DSCP_CLEAR;  break;
    case VLAN_CLASS_FIRST:
        *ENETS_CFG(1) &= ENETS_CFG_FTUC_CLEAR;          break;
    case VLAN_CLASS_SECOND:
        *ENETS_CFG(1) |= ENETS_CFG_VL2_SET;             break;
    case PASS_UNICAST_PACKETS:
        *ENETS_CFG(1) &= ENETS_CFG_FTUC_CLEAR;          break;
    case FILTER_UNICAST_PACKETS:
        *ENETS_CFG(1) |= ENETS_CFG_FTUC_SET;            break;
    case KEEP_BROADCAST_PACKETS:
        *ENETS_CFG(1) &= ENETS_CFG_DPBC_CLEAR;          break;
    case DROP_BROADCAST_PACKETS:
        *ENETS_CFG(1) |= ENETS_CFG_DPBC_SET;            break;
    case KEEP_MULTICAST_PACKETS:
         *ENETS_CFG(1) &= ENETS_CFG_DPMC_CLEAR;         break;
    case DROP_MULTICAST_PACKETS:
        *ENETS_CFG(1) |= ENETS_CFG_DPMC_SET;
#endif
    default:
        return -EOPNOTSUPP;
    }

    return 0;
}

static int eth2_change_mtu(struct net_device *dev, int new_mtu)
{
    dbg("not implemented");

    /*  not implemented */
    return 0;
}

static void eth2_tx_timeout(struct net_device *dev)
{
    u32 sys_flag;
    int ndev = dev - eth2_net_dev;

    dbg("invoked");

    ETH2_ASSERT(ndev >= 0 && ndev < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev), "ndev = %d (wrong value)", ndev);

    /*  must restart TX channel (pending)   */

    /*  disable TX irq, release skb when sending new packet */
    local_irq_save(sys_flag);
    eth2_dev.tx_irq &= ~(1 << ndev);
    *MBOX_IGU1_IER = eth2_dev.tx_irq ? ((eth2_dev.tx_irq << 16) | eth2_dev.rx_irq) : eth2_dev.rx_irq;
    local_irq_restore(sys_flag);

    /*  wake up TX queue    */
    netif_wake_queue(dev);

    return;
}

/*
 *  Description:
 *    Implement strategy to pick up TX DMA channel to transmit packet. Not
 *    implemented yet.
 *  Input:
 *    skb --- struct sk_buff *, packet waiting to be transmitted.
 *    dev --- struct net_device *, device used to transmit packet.
 *  Output:
 *    int --- 0:    Success
 *            else: Error Code
 */
static INLINE int pick_up_tx_channel(struct sk_buff *skb, struct net_device *dev)
{
#if 0
    static int count = 0;

    return count++ % TX_TOTAL_CHANNEL_USED;
#else
    return dev - eth2_net_dev;
#endif
}

/*
 *  Description:
 *    Allocate a sk_buff for RX path using. The size is maximum packet size
 *    plus maximum overhead size.
 *  Input:
 *    none
 *  Output:
 *    sk_buff* --- 0:    Failed
 *                 else: Pointer to sk_buff
 */
static INLINE struct sk_buff* alloc_skb_rx(void)
{
    struct sk_buff *skb;

    /*  allocate memroy including trailer and padding   */
    skb = dev_alloc_skb(eth2_dev.rx_buffer_size + RX_HEAD_MAC_ADDR_ALIGNMENT + DMA_ALIGNMENT);
    if ( skb )
    {
        /*  must be burst length alignment and reserve two more bytes for MAC address alignment  */
        if ( ((u32)skb->data & (DMA_ALIGNMENT - 1)) != 0 )
            skb_reserve(skb, ~((u32)skb->data + (DMA_ALIGNMENT - 1)) & (DMA_ALIGNMENT - 1));
        /*  invalidate cache    */
        *((u32*)skb->data - 1) = (u32)skb;

        dma_cache_inv((unsigned long)skb->head, (u32)skb->end - (u32)skb->head);
        /*  put skb in reserved area "skb->data - 4"    */
    }
    return skb;
}

/*
 *  Description:
 *    Allocate a sk_buff for TX path using.
 *  Input:
 *    size     --- unsigned int, size of the buffer
 *  Output:
 *    sk_buff* --- 0:    Failed
 *                 else: Pointer to sk_buff
 */
static INLINE struct sk_buff* alloc_skb_tx(unsigned int size)
{
    struct sk_buff *skb;

    /*  allocate memory including padding   */
    size = (size + DMA_ALIGNMENT - 1) & ~(DMA_ALIGNMENT - 1);
    skb = dev_alloc_skb(size + DMA_ALIGNMENT);
    /*  must be burst length alignment  */
    if ( skb && ((u32)skb->data & (DMA_ALIGNMENT - 1)) != 0 )
        skb_reserve(skb, ~((u32)skb->data + (DMA_ALIGNMENT - 1)) & (DMA_ALIGNMENT - 1));
    return skb;
}

/*
 *  Description:
 *    Allocate a TX descriptor for DMA channel.
 *  Input:
 *    ch     --- int, connection ID
 *    f_full --- int *, a pointer to get descriptor full flag
 *               1: full, 0: not full
 *  Output:
 *    int    --- negative value: descriptor is used up.
 *               else:           index of descriptor relative to the first one
 *                               of this channel.
 */
static INLINE int alloc_tx_channel(int ch, int *f_full)
{
    u32 sys_flag;
    int desc_base;

    local_irq_save(sys_flag);

    desc_base = eth2_dev.tx_descriptor_number * ch + eth2_dev.tx_desc_alloc_pos[ch];
    if ( !eth2_dev.tx_descriptor_base[desc_base].own )  //  hold by MIPS
    {
        if ( ++eth2_dev.tx_desc_alloc_pos[ch] == eth2_dev.tx_descriptor_number )
            eth2_dev.tx_desc_alloc_pos[ch] = 0;

        ETH2_ASSERT(f_full, "pointer \"f_full\" must be valid!");
        if ( eth2_dev.tx_descriptor_base[eth2_dev.tx_descriptor_number * ch + eth2_dev.tx_desc_alloc_pos[ch]].own ) //  hold by PP32
            *f_full = 1;
        else
            *f_full = 0;
    }
    else
        desc_base = -1;

    local_irq_restore(sys_flag);

    return desc_base;
}

#if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL
static void eth2_xon(struct net_device *dev)
{
//    printk("danube_eth2.c: trying to recover RX_DMACH\n");
    clear_bit(eth2_dev.fc_bit, &netdev_fc_xoff);
    if ( netif_running(dev) )
    {
        *CFG_WRX_DMACH_ON = (1 << RX_TOTAL_CHANNEL_USED) - 1;
    }
}
#endif

#if !defined(DISABLE_TASKLET) || !DISABLE_TASKLET
static void do_eth2_tasklet(unsigned long arg)
{
  #if 0
    if ( !(*MBOX_IGU1_ISR & ((1 << RX_TOTAL_CHANNEL_USED) - 1)) )
    {
        if ( ++eth2_dev.empty_loop >= TASKLET_MAX_EMPTY_LOOP )
        {
            u32 sys_flag;

            local_irq_save(sys_flag);
            eth2_dev.irq_handling_flag = 0;
            //local_irq_save(sys_flag);
            eth2_dev.rx_irq = (1 << RX_TOTAL_CHANNEL_USED) - 1; //  enable RX interrupt
            *MBOX_IGU1_IER = (eth2_dev.tx_irq << 16) | eth2_dev.rx_irq;
            local_irq_restore(sys_flag);

            return;
        }
    }
    else
    {
        int rx_check_loop_counter = 0;
        u32 f_quit = 0;
        u32 channel_mask = 1;
        int channel = 0;
        u32 vlddes;

        while ( 1 )
        {
            if ( (*MBOX_IGU1_ISR & channel_mask) )
            {
                *MBOX_IGU1_ISRC = channel_mask;
                vlddes = ERX_DMA_CHANNEL_CONFIG(channel)->vlddes;

                while ( vlddes )
                {
                    if ( mailbox_rx_irq_handler(channel) == 0 )
                    {
                        /*  signal firmware that descriptor is updated  */
                        mailbox_signal(channel, 0);

                        vlddes--;
                        rx_check_loop_counter++;

                        f_quit = 0;
                    }
                    else
                    {
                        dbg("mailbox_rx_irq_handler(%d) = -EAGAIN", channel);

                        *MBOX_IGU1_ISRS = channel_mask;

                        f_quit |= channel_mask;

                        break;
                    }
                }
            }
            else
                f_quit |= channel_mask;

            if ( rx_check_loop_counter > TASKLET_MAX_RX_CHECK_LOOP || f_quit == (1 << RX_TOTAL_CHANNEL_USED) - 1 )
                break;

            /*  update channel, channel_mask    */
            if ( ++channel == RX_TOTAL_CHANNEL_USED )
            {
                channel_mask = 1;
                channel = 0;
            }
            else
                channel_mask <<= 1;
        }
    }

    tasklet_schedule(&eth2_tasklet);
 #else
    u32 sys_flag;
    int rx_check_loop_counter;
    u32 channel_mask;
    int channel;

    rx_check_loop_counter = 0;
    channel = RX_TOTAL_CHANNEL_USED - 1;
    channel_mask = 0;
    do
    {
        if ( ++channel == RX_TOTAL_CHANNEL_USED )
        {
            channel_mask = 1;
            channel = 0;
        }
        else
            channel_mask <<= 1;

        if ( !(*MBOX_IGU1_ISR & channel_mask) )
            continue;

        *MBOX_IGU1_ISRC = channel_mask;

        while ( mailbox_rx_irq_handler(channel) == 0 )
        {
            /*  signal firmware that descriptor is updated  */
            mailbox_signal(channel, 0);

            ++rx_check_loop_counter;
        }

        if ( rx_check_loop_counter >= TASKLET_MAX_RX_CHECK_LOOP )
        {
            tasklet_schedule(&eth2_tasklet);
            return;
        }
    } while ( (*MBOX_IGU1_ISR & ((1 << RX_TOTAL_CHANNEL_USED) - 1)) );

    local_irq_save(sys_flag);
    eth2_dev.irq_handling_flag = 0;
   // printk("eth2_dev.irq_handling_flag = 0; in %s", __FUNCTION__);
    //local_irq_save(sys_flag);
    eth2_dev.rx_irq = (1 << RX_TOTAL_CHANNEL_USED) - 1; //  enable RX interrupt
    *MBOX_IGU1_IER = eth2_dev.tx_irq ? ((eth2_dev.tx_irq << 16) | eth2_dev.rx_irq) : eth2_dev.rx_irq;
    local_irq_restore(sys_flag);
  #endif
}
#endif

/*
 *  Description:
 *    Handle IRQ triggered by received packet (RX).
 *  Input:
 *    channel --- unsigned int, channel ID which triggered IRQ
 *  Output:
 *    int     --- 0:    Success
 *                else: Error Code (-EAGAIN, retry until owner flag set)
 */
//static int rx_dmach_stopped = 0;
//volatile int loop = 0;
static INLINE int mailbox_rx_irq_handler(unsigned int channel)
{
//    u32 sys_flag;
//    int skb_base;
//    static int called_num;
    struct sk_buff *skb;
//    struct sk_buff *skb_dbg_copy;
    register struct rx_descriptor reg_desc;
    struct rx_descriptor *desc;
//    struct rx_descriptor *desc_dbg_copy;

    ETH2_ASSERT(channel >= 0 && channel < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev), "channel = %d (wrong value)", channel);

//    local_irq_save(sys_flag);

    /*  get sk_buff pointer and descriptor  */
//    skb_base = eth2_dev.rx_descriptor_number * channel + eth2_dev.rx_desc_read_pos[channel];

    /*  load descriptor from memory */
//    desc = &eth2_dev.rx_descriptor_base[skb_base];
    desc = eth2_dev.rx_desc_ch_base[channel] + eth2_dev.rx_desc_read_pos[channel];
//    desc_dbg_copy = eth2_dev.rx_desc_ch_base_dbg_copy + channel * 24 + eth2_dev.rx_desc_read_pos[channel];

//    called_num ++;

//    if(desc->dataptr != desc_dbg_copy->dataptr) {
//        loop = 1;
//        printk("** Error!, called_num = %d, (%08x, %08x), descriptor data pointer mismatch (at fn: %s() ln: %d\n",
//                called_num, desc->dataptr, desc_dbg_copy->dataptr, __FUNCTION__, __LINE__);
//        while(loop);
//    }
    reg_desc = *desc;

    /*  if PP32 hold descriptor or show not completed   */
    if ( reg_desc.own || !reg_desc.c )
    {
//        local_irq_restore(sys_flag);
        return -EAGAIN;
    }

    /*  update read position    */
    if ( ++eth2_dev.rx_desc_read_pos[channel] == eth2_dev.rx_descriptor_number )
        eth2_dev.rx_desc_read_pos[channel] = 0;

//    local_irq_restore(sys_flag);

    /*  get skb address */
    skb = *(struct sk_buff **)((((u32)reg_desc.dataptr << 2) | KSEG0) - 4);

//    skb_dbg_copy = *(struct sk_buff **)((((u32)desc_dbg_copy->dataptr << 2) | KSEG0) - 4);

//    if(skb != skb_dbg_copy) {
//        loop = 1;
//        printk("** Error!, called_num = %d, (%08x, %08x), saved skb pointer mismatch (at fn: %s() ln: %d\n",
//                called_num, skb, skb_dbg_copy, __FUNCTION__, __LINE__);
//        while(loop);
//    }

    ETH2_ASSERT((u32)skb >= 0x80000000, "mailbox_rx_irq_handler: skb = 0x%08X", (u32)skb);

    if ( !reg_desc.err )
    {
        struct sk_buff *new_skb;

        new_skb = alloc_skb_rx();
        if ( new_skb )
        {
//            reset_skb_rx(skb);
            skb_reserve(skb, reg_desc.byteoff);
            skb_put(skb, reg_desc.datalen - RX_TAIL_CRC_LENGTH);

  #if defined(DEBUG_DUMP_RX_SKB_BEFORE) && DEBUG_DUMP_RX_SKB_BEFORE
            dump_skb(skb, 1);
  #endif

            ETH2_ASSERT(skb->len >= 60, "mailbox_rx_irq_handler: skb->len = %d, reg_desc.datalen = %d", skb->len, reg_desc.datalen);

            /*  parse protocol header   */
            skb->dev = eth2_net_dev + channel;
            skb->protocol = eth_type_trans(skb, eth2_net_dev + channel);

  #if defined(DEBUG_DUMP_RX_SKB_AFTER) && DEBUG_DUMP_RX_SKB_AFTER
            dump_skb(skb, 1);
  #endif

  #if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL
            if ( netif_rx(skb) == NET_RX_DROP )
            {
                if ( eth2_dev.fc_bit && !test_and_set_bit(eth2_dev.fc_bit, &netdev_fc_xoff) )
                {
//                    printk("danube_eth2.c: stop RX_DMACH\n");
//                    rx_dmach_stopped = 1;
    #if defined(DEBUG_DUMP_RX_SKB_AFTER) && DEBUG_DUMP_RX_SKB_AFTER
                    dump_skb(skb, 1);
    #endif
                    *CFG_WRX_DMACH_ON = 0;
                }

    #if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
                eth2_dev.rx_fail++;
    #endif
            }
            else
            {
    #if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
                eth2_dev.rx_success++;
    #endif
            }
  #else
    #if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
            if ( netif_rx(skb) != NET_RX_DROP )
                eth2_dev.rx_success++;
            else
                eth2_dev.rx_fail++;
    #else
            netif_rx(skb);
    #endif
  #endif    //  #if defined(ETH2_HW_FLOWCONTROL) && ETH2_HW_FLOWCONTROL

            /*  update descriptor with new sk_buff  */
            reg_desc.dataptr = (u32)new_skb->data >> 2;
            reg_desc.byteoff = RX_HEAD_MAC_ADDR_ALIGNMENT;
        }
        else
        {
            dbg("null sk_buff");

            /*  no sk buffer    */
//            eth2_dev.stats.rx_errors++;
            eth2_dev.rx_drop_counter++;
  #if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
            eth2_dev.rx_driver_level_drop++;
  #endif
        }
    }
    else
    {
        dbg("rx_error");
        eth2_net_stats[channel].rx_errors++;
//        eth2_dev.rx_drop_counter++;
  #if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
        eth2_dev.rx_driver_level_error++;
  #endif
    }

    /*  update descriptor   */
    reg_desc.datalen = eth2_dev.rx_buffer_size;
    reg_desc.own = 1;
    reg_desc.c   = 0;

    /*  write descriptor to memory  */
    *desc = reg_desc;

//    *desc_dbg_copy = reg_desc;
    return 0;
}

/*
 *  Description:
 *    Handle IRQ of mailbox and despatch to relative handler.
 *  Input:
 *    irq    --- int, IRQ number
 *    dev_id --- void *, argument passed when registering IRQ handler
 *    regs   --- struct pt_regs *, registers' value before jumping into handler
 *  Output:
 *    none
 */
//static void mailbox_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
static irqreturn_t mailbox_irq_handler(int irq, void *dev_id)
{
//    static int called_num = 0;
    //u32 sys_flag;


#if 1
    if ( !*MBOX_IGU1_ISR )
    {
        //local_irq_save(sys_flag);
        //eth2_dev.irq_handling_flag = 0;
        //local_irq_restore(sys_flag);
        return IRQ_HANDLED;
    }
#endif

    /*  TX  */
    if ( eth2_dev.tx_irq )
    {
        u32 f_set;
        u32 bit;
        int i;

        f_set = eth2_dev.tx_irq & (*MBOX_IGU1_ISR >> 16);
        if ( f_set )
        {
            if(f_set != 1) {
                printk("Error: tx_irq != !\n");
            }
            eth2_dev.tx_irq &= ~f_set;
            *MBOX_IGU1_IER = (eth2_dev.tx_irq << 16) | eth2_dev.rx_irq;
//            *MBOX_IGU1_ISRC = (f_set << 16);

            for ( i = 0, bit = 1; i < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev); i++, bit <<= 1 )
                if ( (f_set & bit) )
                    netif_wake_queue(eth2_net_dev + i);
        }
    }

    //local_irq_save(sys_flag);
    if ( eth2_dev.irq_handling_flag++ )
    {
        //local_irq_restore(sys_flag);
        return IRQ_HANDLED;
    }
    //else
        //local_irq_restore(sys_flag);

#if !defined(DISABLE_TASKLET) || !DISABLE_TASKLET
    /*  RX  */
    if ( eth2_dev.rx_irq )
    {
//        printk("** fn:%s: rx_irq = %d, tasklet_schedule() called\n", __FUNCTION__, eth2_dev.rx_irq);
        eth2_dev.rx_irq = 0;
        *MBOX_IGU1_IER = eth2_dev.tx_irq << 16; //  disable RX irq
        tasklet_schedule(&eth2_tasklet);

        return IRQ_HANDLED;
    }
    //else

    //local_irq_save(sys_flag);
        //eth2_dev.irq_handling_flag = 0;
//    printk("** fn: %s, eth2_dev.irq_handling_flag = 0;\n", __FUNCTION__);
    //local_irq_restore(sys_flag);
#else
    u32 channel_mask;
    int channel;

    channel = RX_TOTAL_CHANNEL_USED - 1;
    channel_mask = 0;
    do
    {
        if ( ++channel == RX_TOTAL_CHANNEL_USED )
        {
            channel_mask = 1;
            channel = 0;
        }
        else
            channel_mask <<= 1;

        if ( !(*MBOX_IGU1_ISR & channel_mask) )
            continue;

        *MBOX_IGU1_ISRC = channel_mask;

        while ( mailbox_rx_irq_handler(channel) == 0 )
        {
            /*  signal firmware that descriptor is updated  */
            mailbox_signal(channel, 0);
        }
    } while ( (*MBOX_IGU1_ISR & ((1 << RX_TOTAL_CHANNEL_USED) - 1)) );

    eth2_dev.irq_handling_flag = 0;

    return IRQ_HANDLED;

#endif
}

/*
 *  Description:
 *    Signal PPE firmware a TX packet ready or RX descriptor updated.
 *  Input:
 *    connection --- unsigned int, connection ID
 *    is_tx      --- int, 0 means RX, else means TX
 *  Output:
 *    none
 */
static INLINE void mailbox_signal(unsigned int connection, int is_tx)
{
    if ( is_tx )
    {
#if !defined(ENABLE_DEBUG_COUNTER) || !ENABLE_DEBUG_COUNTER
        while ( MBOX_IGU3_ISR_ISR(connection + 16) );
#else
        while ( MBOX_IGU3_ISR_ISR(connection + 16) )
            eth2_dev.tx_desc_update_wait_loop++;
#endif
        *MBOX_IGU3_ISRS = MBOX_IGU3_ISRS_SET(connection + 16);
    }
    else
    {
#if !defined(ENABLE_DEBUG_COUNTER) || !ENABLE_DEBUG_COUNTER
        while ( MBOX_IGU3_ISR_ISR(connection) );
#else
        while ( MBOX_IGU3_ISR_ISR(connection) )
            eth2_dev.rx_desc_update_wait_loop++;
#endif
        *MBOX_IGU3_ISRS = MBOX_IGU3_ISRS_SET(connection);
    }
}

/*
 *  Description:
 *    Handle ioctl command SIOCETHTOOL.
 *  Input:
 *    dev --- struct net_device *, device responsing to the command.
 *    ifr --- struct ifreq *, interface request structure to pass parameters
 *            or result.
 *  Output:
 *    int --- 0:    Success
 *            else: Error Code (-EFAULT, -EOPNOTSUPP)
 */
static INLINE int ethtool_ioctl(struct net_device *dev, struct ifreq *ifr)
{
    struct ethtool_cmd cmd;

    if ( copy_from_user(&cmd, ifr->ifr_data, sizeof(cmd)) )
        return -EFAULT;

    switch ( cmd.cmd )
    {
    case ETHTOOL_GSET:      /*  get hardware information        */
        {
            memset(&cmd, 0, sizeof(cmd));

            cmd.supported   = SUPPORTED_Autoneg | SUPPORTED_TP | SUPPORTED_MII |
                              SUPPORTED_10baseT_Half | SUPPORTED_10baseT_Full |
                              SUPPORTED_100baseT_Half | SUPPORTED_100baseT_Full;
            cmd.port        = PORT_MII;
            cmd.transceiver = XCVR_EXTERNAL;
            cmd.phy_address = 1;
            cmd.speed       = ENET_MAC_CFG_SPEED(1) ? SPEED_100 : SPEED_10;
            cmd.duplex      = ENET_MAC_CFG_DUPLEX(1) ? DUPLEX_FULL : DUPLEX_HALF;

            if ( (*ETOP_MDIO_CFG & ETOP_MDIO_CFG_UMM(1, 1)) )
            {
                /*  auto negotiate  */
                cmd.autoneg = AUTONEG_ENABLE;
                cmd.advertising |= ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full |
                                   ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full;
            }
            else
            {
                cmd.autoneg = AUTONEG_DISABLE;
                cmd.advertising &= ~(ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full |
                                     ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full);
            }

            if ( copy_to_user(ifr->ifr_data, &cmd, sizeof(cmd)) )
                return -EFAULT;
        }
        break;
    case ETHTOOL_SSET:      /*  force the speed and duplex mode */
        {
            if ( !capable(CAP_NET_ADMIN) )
                return -EPERM;

            if ( cmd.autoneg == AUTONEG_ENABLE )
            {
                /*  set property and start autonegotiation                                  */
                /*  have to set mdio advertisement register and restart autonegotiation     */
                /*  which is a very rare case, put it to future development if necessary.   */
            }
            else
            {
                /*  set property without autonegotiation    */
                *ETOP_MDIO_CFG &= ~ETOP_MDIO_CFG_UMM(1, 1);

                /*  set speed   */
                if ( cmd.speed == SPEED_10 )
                    ENET_MAC_CFG_SPEED_10M(1);
                else if ( cmd.speed == SPEED_100 )
                    ENET_MAC_CFG_SPEED_100M(1);

                /*  set duplex  */
                if ( cmd.duplex == DUPLEX_HALF )
                    ENET_MAC_CFG_DUPLEX_HALF(1);
                else if ( cmd.duplex == DUPLEX_FULL )
                    ENET_MAC_CFG_DUPLEX_FULL(1);

                ENET_MAC_CFG_LINK_OK(1);
            }
        }
        break;
    case ETHTOOL_GDRVINFO:  /*  get driver information          */
        {
            struct ethtool_drvinfo info;

            memset(&info, 0, sizeof(info));
            strncpy(info.driver, "Danube Eth2 Driver", sizeof(info.driver) - 1);
            strncpy(info.fw_version, "0.0.1", sizeof(info.fw_version) - 1);
            strncpy(info.bus_info, "N/A", sizeof(info.bus_info) - 1);
            info.regdump_len = 0;
            info.eedump_len = 0;
            info.testinfo_len = 0;
            if ( copy_to_user(ifr->ifr_data, &info, sizeof(info)) )
                return -EFAULT;
        }
        break;
    case ETHTOOL_NWAY_RST:  /*  restart auto negotiation        */
        *ETOP_MDIO_CFG |= ETOP_MDIO_CFG_SMRST(1) | ETOP_MDIO_CFG_UMM(1, 1);
        break;
    default:
        return -EOPNOTSUPP;
    }

    return 0;
}

/*
 *  Description:
 *    Specify ETOP ingress VLAN priority's class of service mapping.
 *  Input:
 *    req --- struct vlan_cos_req *, pass parameters such as priority and class
 *            of service mapping.
 *  Output:
 *    none
 */
static INLINE void set_vlan_cos(struct vlan_cos_req *req)
{
     *ETOP_IG_VLAN_COS = ETOP_IG_VLAN_COS_Px_SET(*ETOP_IG_VLAN_COS, req->pri, req->cos_value);
}

/*
 *  Description:
 *    Specify ETOP ingress VLAN differential service control protocol's class of
 *    service mapping.
 *  Input:
 *    req --- struct dscp_cos_req *, pass parameters such as differential
 *            service control protocol and class of service mapping.
 *  Output:
 *    none
 */
static INLINE void set_dscp_cos(struct dscp_cos_req *req)
{
    ETOP_IG_DSCP_COS_SET(req->dscp, req->cos_value);
}

#if defined(ENABLE_PROBE_TRANSCEIVER) && ENABLE_PROBE_TRANSCEIVER
/*
 *  Description:
 *    Setup ethernet hardware in init process.
 *  Input:
 *    dev --- struct net_device *, device to be setup.
 *  Output:
 *    int --- 0:    Success
 *            else: Error Code (-EIO, link is not OK)
 */
static INLINE int probe_transceiver(struct net_device *dev)
{
    *ENETS_MAC_DA0(1) = (dev->dev_addr[0] << 24) | (dev->dev_addr[1] << 16) | (dev->dev_addr[2] << 8) | dev->dev_addr[3];
    *ENETS_MAC_DA1(1) = (dev->dev_addr[4] << 24) | (dev->dev_addr[3] << 16);

    if ( !ENET_MAC_CFG_LINK(1) )
    {
        *ETOP_MDIO_CFG = (*ETOP_MDIO_CFG & ~ETOP_MDIO_CFG_MASK(1))
                         | ETOP_MDIO_CFG_SMRST(1)
                         | ETOP_MDIO_CFG_PHYA(1, ETOP_MDIO_PHY1_ADDR)
                         | ETOP_MDIO_CFG_UMM(1, 1);

        udelay(ETOP_MDIO_DELAY);

        if ( !ENET_MAC_CFG_LINK(1) )
            return -EIO;
    }

    return 0;
}
#endif

/*
 *  Description:
 *    Check parameters passed by command "insmod" and amend them.
 *  Input:
 *    none
 *  Output:
 *    none
 */
static INLINE void check_parameters(void)
{
    /*  There is a delay between PPE write descriptor and descriptor is       */
    /*  really stored in memory. Host also has this delay when writing        */
    /*  descriptor. So PPE will use this value to determine if the write      */
    /*  operation makes effect.                                               */
    if ( write_descriptor_delay < 0 )
        write_descriptor_delay = 0;

    /*  Because of the limitation of length field in descriptors, the packet  */
    /*  size could not be larger than 64K minus overhead size.                */
    if ( rx_max_packet_size < ETH_MIN_FRAME_LENGTH )
        rx_max_packet_size = ETH_MIN_FRAME_LENGTH;
    else if ( rx_max_packet_size > ETH_MAX_FRAME_LENGTH )
        rx_max_packet_size = ETH_MAX_FRAME_LENGTH;
    if ( rx_min_packet_size < ETH_MIN_FRAME_LENGTH )
        rx_min_packet_size = ETH_MIN_FRAME_LENGTH;
    else if ( rx_min_packet_size > rx_max_packet_size )
        rx_min_packet_size = rx_max_packet_size;
    if ( tx_max_packet_size < ETH_MIN_FRAME_LENGTH )
        tx_max_packet_size = ETH_MIN_FRAME_LENGTH;
    else if ( tx_max_packet_size > ETH_MAX_FRAME_LENGTH )
        tx_max_packet_size = ETH_MAX_FRAME_LENGTH;
    if ( tx_min_packet_size < ETH_MIN_FRAME_LENGTH )
        tx_min_packet_size = ETH_MIN_FRAME_LENGTH;
    else if ( tx_min_packet_size > tx_max_packet_size )
        tx_min_packet_size = tx_max_packet_size;

    if ( dma_rx_descriptor_length < 2 )
        dma_rx_descriptor_length = 2;
    if ( dma_tx_descriptor_length < 2 )
        dma_tx_descriptor_length = 2;
}

/*
 *  Description:
 *    Setup variable eth2_dev and allocate memory.
 *  Input:
 *    none
 *  Output:
 *    int --- 0:    Success
 *            else: Error Code
 */
static INLINE int init_eth2_dev(void)
{
    int rx_desc;
    struct rx_descriptor rx_descriptor = {  own:    1,
                                            c:      0,
                                            sop:    1,
                                            eop:    1,
                                            res1:   0,
                                            byteoff:RX_HEAD_MAC_ADDR_ALIGNMENT,
                                            res2:   0,
                                            id:     0,
                                            err:    0,
                                            datalen:0,
                                            res3:   0,
                                            dataptr:0};
    int i;

    int tx_desc;
    struct tx_descriptor tx_descriptor = {  own:    0,  //  it's hold by MIPS
                                            c:      0,
                                            sop:    1,
                                            eop:    1,
                                            byteoff:0,
                                            res1:   0,
                                            iscell: 0,
                                            clp:    0,
                                            datalen:0,
                                            res2:   0,
                                            dataptr:0};

    memset(&eth2_dev, 0, sizeof(eth2_dev));
    memset(eth2_net_stats, 0, sizeof(eth2_net_stats));

    eth2_dev.rx_buffer_size = (rx_max_packet_size + DMA_ALIGNMENT - 1) & ~(DMA_ALIGNMENT - 1);

    /*
     *  dma
     */
    /*  descriptor number of RX DMA channel */
    eth2_dev.rx_descriptor_number   = dma_rx_descriptor_length;
    /*  descriptor number of TX DMA channel */
    eth2_dev.tx_descriptor_number   = dma_tx_descriptor_length;

    /*  delay on descriptor write path  */
    eth2_dev.write_descriptor_delay = write_descriptor_delay;

    /*  allocate memory for RX descriptors  */
    eth2_dev.rx_descriptor_addr = kmalloc(RX_TOTAL_CHANNEL_USED * eth2_dev.rx_descriptor_number * sizeof(struct rx_descriptor) + DMA_ALIGNMENT, GFP_KERNEL | GFP_DMA);
    if ( !eth2_dev.rx_descriptor_addr )
        goto RX_DESCRIPTOR_BASE_ALLOCATE_FAIL;
    /*  do alignment (DWORD)    */
    eth2_dev.rx_descriptor_base = (struct rx_descriptor *)(((u32)eth2_dev.rx_descriptor_addr + (DMA_ALIGNMENT - 1)) & ~(DMA_ALIGNMENT - 1));
    eth2_dev.rx_descriptor_base = (struct rx_descriptor *)((u32)eth2_dev.rx_descriptor_base | KSEG1);   //  no cache
    printk("\nETH2 rx_descriptor_base = %08x\n", (u32)eth2_dev.rx_descriptor_base);
    for ( i = 0; i < RX_TOTAL_CHANNEL_USED; i++ ) {
        eth2_dev.rx_desc_ch_base[i] = eth2_dev.rx_descriptor_base + eth2_dev.rx_descriptor_number * i;
    }

    /*  allocate memory for TX descriptors  */
    eth2_dev.tx_descriptor_addr = kmalloc(TX_TOTAL_CHANNEL_USED * eth2_dev.tx_descriptor_number * sizeof(struct tx_descriptor) + DMA_ALIGNMENT, GFP_KERNEL | GFP_DMA);
    if ( !eth2_dev.tx_descriptor_addr )
       goto TX_DESCRIPTOR_BASE_ALLOCATE_FAIL;
    /*  do alignment (DWORD)    */
    eth2_dev.tx_descriptor_base = (struct tx_descriptor *)(((u32)eth2_dev.tx_descriptor_addr + (DMA_ALIGNMENT - 1)) & ~(DMA_ALIGNMENT - 1));
    eth2_dev.tx_descriptor_base = (struct tx_descriptor *)((u32)eth2_dev.tx_descriptor_base | KSEG1);   //  no cache
    printk("tx_descriptor_base = %08x\n", (u32)eth2_dev.tx_descriptor_base);

    /*  allocate pointers to TX sk_buff */
    eth2_dev.tx_skb_pointers = kmalloc(TX_TOTAL_CHANNEL_USED * eth2_dev.tx_descriptor_number * sizeof(struct sk_buff *), GFP_KERNEL);
    if ( !eth2_dev.tx_skb_pointers )
        goto TX_SKB_POINTER_ALLOCATE_FAIL;
    memset(eth2_dev.tx_skb_pointers, 0, TX_TOTAL_CHANNEL_USED * eth2_dev.tx_descriptor_number * sizeof(struct sk_buff *));

    /*  Allocate RX sk_buff and fill up RX descriptors. */
    rx_descriptor.datalen = eth2_dev.rx_buffer_size;
    for ( rx_desc = RX_TOTAL_CHANNEL_USED * eth2_dev.rx_descriptor_number - 1; rx_desc >= 0; rx_desc-- )
    {
        struct sk_buff *skb;

        skb = alloc_skb_rx();
        if ( skb == NULL )
            panic("sk buffer is used up\n");
        rx_descriptor.dataptr = (u32)skb->data >> 2;
        eth2_dev.rx_descriptor_base[rx_desc] = rx_descriptor;
//        eth2_dev.rx_desc_ch_base_dbg_copy[rx_desc] = rx_descriptor;
    }

    /*  Fill up TX descriptors. */
    for ( tx_desc = TX_TOTAL_CHANNEL_USED * eth2_dev.tx_descriptor_number - 1; tx_desc >= 0; tx_desc-- )
        eth2_dev.tx_descriptor_base[tx_desc] = tx_descriptor;

    return 0;

TX_SKB_POINTER_ALLOCATE_FAIL:
    kfree(eth2_dev.tx_descriptor_addr);
TX_DESCRIPTOR_BASE_ALLOCATE_FAIL:
    kfree(eth2_dev.rx_descriptor_addr);
RX_DESCRIPTOR_BASE_ALLOCATE_FAIL:
    return -ENOMEM;
}

#ifdef MODULE
static INLINE void reset_ppe(void)
{
    int i;

    //  reset ETOP
    *((volatile u32 *)(0xBF203000 + 0x0010)) |= (1 << 8);
    for ( i = 0; i < 0x1000; i++ );

    amazon_se_sw_config();
}
#endif

static INLINE int init_local_variables(void)
{
#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
    dbg_enable = 0;
#endif

    return 0;
}

/*
 *  Description:
 *    Fill up share buffer with 0.
 *  Input:
 *    none
 *  Output:
 *    none
 */
static INLINE void clear_share_buffer(void)
{
    volatile u32 *p = SB_RAM0_ADDR(0);
    unsigned int i;

    /*  write all zeros only    */
    for ( i = 0; i < SB_RAM0_DWLEN + SB_RAM1_DWLEN; i++ )
        *p++ = 0;
}

static INLINE void config_ppe_hw(void)
{
	*AT_CTRL            = 0x0F00;
    *AR_CTRL            = 0x3C00;
	*AT_IDLE0           = 0x0;
	*AT_IDLE1           = 0x0;
	*AR_IDLE0           = 0x0;
	*AR_IDLE1           = 0x0;
	*RFBI_CFG           = 0x0;
	*SFSM_DBA0          = 0x0200;
	*SFSM_DBA1          = 0x0800;
	*SFSM_CBA0          = 0x0321;
	*SFSM_CBA1          = 0x0921;
	*SFSM_CFG0          = 0x14011;
	*SFSM_CFG1          = 0x14011;
	*FFSM_DBA0          = 0x0332;
	*FFSM_DBA1          = 0x0932;
	*FFSM_CFG0          = 0x3000C;
	*FFSM_CFG1          = 0x3000C;
	*FFSM_IDLE_HEAD_BC0 = 0xF0D10000;
	*FFSM_IDLE_HEAD_BC1 = 0xF0D10000;
	*EMA_DATACFG        = 0x1000B00;
}

/*
 *  Description:
 *    Setup RX/TX relative registers and tables, including HTU table. All
 *    parameters are taken from eth2_dev.
 *  Input:
 *    none
 *  Output:
 *    none
 */
static INLINE void init_tables(void)
{
    int i;
    volatile u32 *p;
    struct wrx_dma_channel_config rx_config = {0};
    struct wtx_dma_channel_config tx_config = {0};
    struct wrx_port_cfg_status rx_port_cfg = { 0 };
    struct wtx_port_cfg        tx_port_cfg = { 0 };

    /*
     *  CDM Block 1
     */
    *CDM_CFG = CDM_CFG_RAM1_SET(0x00) | CDM_CFG_RAM0_SET(0x00); //  CDM block 1 must be data memory and mapped to 0x5000 (dword addr)
    p = PP32_DATA_MEMORY_RAM1_ADDR(0);                          //  Clear CDM block 1
    for ( i = 0; i < PP32_DATA_MEMORY_RAM1_DWLEN; i++ )
        *p++ = 0;

    /*
     *  General Registers
     */
    *CFG_WAN_WRDES_DELAY    = eth2_dev.write_descriptor_delay;
    *CFG_WRX_DMACH_ON       = (1 << RX_TOTAL_CHANNEL_USED) - 1;
    *CFG_WTX_DMACH_ON       = (1 << TX_TOTAL_CHANNEL_USED) - 1 ;

    *CFG_WRX_LOOK_BITTH     = 8;  // WAN RX EFM-TC Looking Threshold

    *CFG_ETH_EFMTC_CRC      = *(struct eth_efmtc_crc_cfg *)&eth_efmtc_crc_cfg;

    /*
     *  WRX DMA Channel Configuration Table
     */
    rx_config.deslen = eth2_dev.rx_descriptor_number;
    rx_port_cfg.mfs = 1526;
    rx_port_cfg.local_state = 0;     // looking for sync
    rx_port_cfg.partner_state = 0;   // parter receiver is out of sync

    for ( i = 0; i < RX_TOTAL_CHANNEL_USED; i++ )
    {
        rx_config.desba = (((u32)eth2_dev.rx_descriptor_base  >> 2 ) & 0x0FFFFFFF) + eth2_dev.rx_descriptor_number * i * (sizeof(struct  rx_descriptor) >> 2);
        *WRX_DMA_CHANNEL_CONFIG(i) = rx_config;


        rx_port_cfg.dmach = i;
        *WRX_PORT_CONFIG(i) = rx_port_cfg;
    }

    /*
     *  WTX DMA Channel Configuration Table
     */
    tx_config.deslen = eth2_dev.tx_descriptor_number;

    tx_port_cfg.tx_cwth1 = 5;
    tx_port_cfg.tx_cwth2 = 4;

    for ( i = 0; i < TX_TOTAL_CHANNEL_USED; i++ )
    {
        tx_config.desba = (((u32)eth2_dev.tx_descriptor_base >> 2 ) & 0x0FFFFFFF) + eth2_dev.tx_descriptor_number * i * (sizeof(struct  tx_descriptor) >> 2);
        *WTX_DMA_CHANNEL_CONFIG(i) = tx_config;


        *WTX_PORT_CONFIG(i) = tx_port_cfg;
    }
}

/*
 *  Description:
 *    Clean-up eth2_dev and release memory.
 *  Input:
 *    none
 *  Output:
 *    none
 */
static INLINE void clear_eth2_dev(void)
{
    int desc_base;
    int i, j;

    /*
     *  free memory allocated for RX/TX descriptors and RX sk_buff
     */
    desc_base = 0;
    for ( i = 0; i < TX_TOTAL_CHANNEL_USED; i++ )
        for ( j = 0; j < eth2_dev.tx_descriptor_number; j++ )
        {
            if ( eth2_dev.tx_skb_pointers[desc_base] )
                dev_kfree_skb_any(eth2_dev.tx_skb_pointers[desc_base]);
            desc_base++;
        }

    for ( i = RX_TOTAL_CHANNEL_USED * eth2_dev.rx_descriptor_number - 1; i >= 0; i-- )
        dev_kfree_skb_any(*(struct sk_buff **)(((eth2_dev.rx_descriptor_base[i].dataptr << 2) | KSEG0) - 4));

    kfree(eth2_dev.tx_skb_pointers);
    kfree(eth2_dev.tx_descriptor_addr);
    kfree(eth2_dev.rx_descriptor_addr);
}

static INLINE void init_ema(void)
{
    *EMA_CMDCFG  = (EMA_CMD_BUF_LEN << 16) | (EMA_CMD_BASE_ADDR >> 2);
    *EMA_DATACFG = (EMA_DATA_BUF_LEN << 16) | (EMA_DATA_BASE_ADDR >> 2);
    *EMA_IER     = 0x000000FF;
	*EMA_CFG     = EMA_READ_BURST | (EMA_WRITE_BURST << 2);
}

static INLINE void init_chip(int mode)
{
    /*  enable PPE module in PMU    */
    *(unsigned long *)0xBF10201C &= ~((1 << 15) | (1 << 13) | (1 << 9));

    //  Configure share buffer master selection
    //*SB_MST_SEL |= 0x03;  //  no eth1
    *SB_MST_PRI0 = 1;
    *SB_MST_PRI1 = 1;

    //  Init EMA
    init_ema();

    //  Enable mailbox
    *MBOX_IGU1_ISRC = 0xFFFFFFFF;
    *MBOX_IGU1_IER  = (1 << RX_TOTAL_CHANNEL_USED) - 1; //  enable RX interrupt only
    *MBOX_IGU3_ISRC = 0xFFFFFFFF;
    *MBOX_IGU3_IER  = ((1 << RX_TOTAL_CHANNEL_USED) - 1) | (((1 << TX_TOTAL_CHANNEL_USED) - 1) << 16);

    dbg("ENETS_DBA(0)     = 0x%08X, ENETS_DBA(1)     = 0x%08X", *ENETS_DBA(0),    *ENETS_DBA(1));
    dbg("ENETS_CBA(0)     = 0x%08X, ENETS_CBA(1)     = 0x%08X", *ENETS_CBA(0),    *ENETS_CBA(1));
    dbg("ENETS_CFG(0)     = 0x%08X, ENETS_CFG(1)     = 0x%08X", *ENETS_CFG(0),    *ENETS_CFG(1));
    dbg("ENETS_PGCNT(0)   = 0x%08X, ENETS_PGCNT(1)   = 0x%08X", *ENETS_PGCNT(0),  *ENETS_PGCNT(1));
    dbg("ENETS_PKTCNT(0)  = 0x%08X, ENETS_PKTCNT(1)  = 0x%08X", *ENETS_PKTCNT(0), *ENETS_PKTCNT(1));
    dbg("ENETS_COS_CFG(0) = 0x%08X, ENETS_COS_CFG(1) = 0x%08X", *ENETS_COS_CFG(0),*ENETS_COS_CFG(1));
    dbg("ENETF_DBA(0)     = 0x%08X, ENETF_DBA(1)     = 0x%08X", *ENETF_DBA(0),    *ENETF_DBA(1));
    dbg("ENETF_CBA(0)     = 0x%08X, ENETF_CBA(1)     = 0x%08X", *ENETF_CBA(0),    *ENETF_CBA(1));
    dbg("ENETF_PGCNT(0)   = 0x%08X, ENETF_PGCNT(1)   = 0x%08X", *ENETF_PGCNT(0),  *ENETF_PGCNT(1));
    dbg("ENETF_PKTCNT(0)  = 0x%08X, ENETF_PKTCNT(1)  = 0x%08X", *ENETF_PKTCNT(0), *ENETF_PKTCNT(1));

    dbg("SB_MST_SEL = 0x%08X", *SB_MST_SEL);

    dbg("ETOP_CFG = 0x%08X", *ETOP_CFG);

    dbg("ETOP_IG_PLEN_CTRL0 = 0x%08X", *ETOP_IG_PLEN_CTRL0);

    dbg("ENET_MAC_CFG(0) = 0x%08X, ENET_MAC_CFG(1) = 0x%08X", *ENET_MAC_CFG(0), *ENET_MAC_CFG(1));
}

/*
 *  Description:
 *    Download PPE firmware binary code.
 *  Input:
 *    src       --- u32 *, binary code buffer
 *    dword_len --- unsigned int, binary code length in DWORD (32-bit)
 *  Output:
 *    int       --- 0:    Success
 *                  else: Error Code
 */
static INLINE int pp32_download_code(const u32 *code_src, unsigned int code_dword_len, const u32 *data_src, unsigned int data_dword_len)
{
    u32 reg_old_value;
    volatile u32 *dest;

    if ( code_src == 0 || ((unsigned long)code_src & 0x03) != 0
        || data_src == 0 || ((unsigned long)data_src & 0x03) )
        return -EINVAL;

    /*  save the old value of CDM_CFG and set PPE code memory to FPI bus access mode    */
    reg_old_value = *CDM_CFG;
    if ( code_dword_len <= 4096 )
        *CDM_CFG = CDM_CFG_RAM1_SET(0x00) | CDM_CFG_RAM0_SET(0x00);
    else
        *CDM_CFG = CDM_CFG_RAM1_SET(0x01) | CDM_CFG_RAM0_SET(0x00);

    /*  copy code   */
    dest = CDM_CODE_MEMORY_RAM0_ADDR(0);
    while ( code_dword_len-- > 0 )
        *dest++ = *code_src++;

    /*  copy data   */
    dest = PP32_DATA_MEMORY_RAM1_ADDR(0);
    while ( data_dword_len-- > 0 )
        *dest++ = *data_src++;

    /*  restore old configuration   */
//    *CDM_CFG = reg_old_value;

    return 0;
}

/*
 *  Description:
 *    Do PP32 specific initialization.
 *  Input:
 *    data --- void *, specific parameter passed in.
 *  Output:
 *    int  --- 0:    Success
 *             else: Error Code
 */
static INLINE int pp32_specific_init(void *data)
{
    return 0;
}

/*
 *  Description:
 *    Initialize and start up PP32.
 *  Input:
 *    none
 *  Output:
 *    int  --- 0:    Success
 *             else: Error Code
 */
static INLINE int pp32_start(void)
{
    int ret;
    register int i;

    /*  download firmware   */
    ret = pp32_download_code(firmware_binary_code, sizeof(firmware_binary_code) / sizeof(*firmware_binary_code), firmware_binary_data, sizeof(firmware_binary_data) / sizeof(*firmware_binary_data));
   if ( ret )
        return ret;

    /*  firmware specific initialization    */
    ret = pp32_specific_init(NULL);
    if ( ret )
        return ret;

    /*  run PP32    */
    *PP32_DBG_CTRL = DBG_CTRL_RESTART;
    /*  idle for a while to let PP32 init itself    */
    for ( i = 0; i < IDLE_CYCLE_NUMBER; i++ );

    return 0;
}

static INLINE void ppe_bp_setup(void)
{
//	*DBG_DATA_MIN0 = 0x2001;
//	*DBG_DATA_MAX0 = 0x2001;
//	DBG_SETDAEN_0(DBG_DABP_WRITE);


//	*DBG_DATA_MIN1 = 0x1092;	// DPLUS_PKT_CNT0
//	*DBG_DATA_MAX1 = 0x1092;
//	DBG_SETDAEN_1(DBG_DABP_WRITE);

//	*DBG_PC_MIN0 = *DBG_PC_MAX0 = 0x0231;
//	DBG_ENABLE_PC(0);

}

/*
 *  Description:
 *    Halt PP32.
 *  Input:
 *    none
 *  Output:
 *    none
 */
static INLINE void pp32_stop(void)
{
    /*  halt PP32   */
    *PP32_DBG_CTRL = DBG_CTRL_STOP;
}

static INLINE void proc_file_create(void)
{
    struct proc_dir_entry *res;

    g_eth2_proc_dir = proc_mkdir("eth2", NULL);

    create_proc_read_entry("stats",
                           0,
                           g_eth2_proc_dir,
                           proc_read_stats,
                           NULL);

    res = create_proc_read_entry("mib",
                                  0,
                                  g_eth2_proc_dir,
                                  proc_read_mib,
                                  NULL);
    if ( res )
        res->write_proc = proc_write_mib;

#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
    res = create_proc_read_entry("dbg",
                                  0,
                                  g_eth2_proc_dir,
                                  proc_read_dbg,
                                  NULL);
    if ( res )
        res->write_proc = proc_write_dbg;
#endif

#if defined(DEBUG_MEM_PROC) && DEBUG_MEM_PROC
    res = create_proc_read_entry("mem",
                                  0,
                                  g_eth2_proc_dir,
                                  NULL,
                                  NULL);
    if ( res )
        res->write_proc = proc_write_mem;
#endif

}

static INLINE void proc_file_delete(void)
{
    remove_proc_entry("stats",
                      g_eth2_proc_dir);

    remove_proc_entry("mib",
                      g_eth2_proc_dir);

#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
    remove_proc_entry("dbg",
                      g_eth2_proc_dir);
#endif

#if defined(DEBUG_MEM_PROC) && DEBUG_MEM_PROC
    remove_proc_entry("mem",
                      g_eth2_proc_dir);
#endif


    remove_proc_entry("eth2", NULL);
}

static int proc_read_mib(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;
//    int i;

//    MOD_INC_USE_COUNT;

    len += sprintf(page + off + len,     "ptm0\n");
    len += sprintf(page + off + len,     "  RX\n");
    len += sprintf(page + off + len,     "    correct      : %u (%u)\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_correct_pdu, WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_correct_pdu_bytes);
    len += sprintf(page + off + len,     "    tccrc_err    : %u (%u)\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_tccrc_err_pdu, WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_tccrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    ethcrc_err   : %u (%u)\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_ethcrc_err_pdu, WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_ethcrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    nodesc_drop  : %u\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_nodesc_drop_pdu);
    len += sprintf(page + off + len,     "    len_viol_drop: %u\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_len_violation_drop_pdu);
    len += sprintf(page + off + len,     "    idle_bytes   : %u\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_idle_bytes);
    len += sprintf(page + off + len,     "    nonidle_cw   : %u\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_nonidle_cw);
    len += sprintf(page + off + len,     "    idle_cw      : %u\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_idle_cw);
    len += sprintf(page + off + len,     "    err_cw       : %u\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wrx_err_cw);
    len += sprintf(page + off + len,     "  TX \n");
    len += sprintf(page + off + len,     "    total_pdu    : %u (%u)\n", WAN_MIB_TABLE[PTM0_MIB_IDX].wtx_total_pdu, WAN_MIB_TABLE[PTM0_MIB_IDX].wtx_total_bytes);

    len += sprintf(page + off + len,     "ptmfast0\n");
    len += sprintf(page + off + len,     "  RX\n");
    len += sprintf(page + off + len,     "    correct      : %u (%u)\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_correct_pdu, WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_correct_pdu_bytes);
    len += sprintf(page + off + len,     "    tccrc_err    : %u (%u)\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_tccrc_err_pdu, WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_tccrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    ethcrc_err   : %u (%u)\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_ethcrc_err_pdu, WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_ethcrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    nodesc_drop  : %u\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_nodesc_drop_pdu);
    len += sprintf(page + off + len,     "    len_viol_drop: %u\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_len_violation_drop_pdu);
    len += sprintf(page + off + len,     "    idle_bytes   : %u\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_idle_bytes);
    len += sprintf(page + off + len,     "    nonidle_cw   : %u\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_nonidle_cw);
    len += sprintf(page + off + len,     "    idle_cw      : %u\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_idle_cw);
    len += sprintf(page + off + len,     "    err_cw       : %u\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wrx_err_cw);
    len += sprintf(page + off + len,     "  TX \n");
    len += sprintf(page + off + len,     "    total_pdu    : %u (%u)\n", WAN_MIB_TABLE[PTMFAST_MIB_IDX].wtx_total_pdu, WAN_MIB_TABLE[PTMFAST_MIB_IDX].wtx_total_bytes);

#if 0
    len += sprintf(page + off + len,     "ptm0hi\n");
    len += sprintf(page + off + len,     "  RX\n");
    len += sprintf(page + off + len,     "    correct      : %u (%u)\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_correct_pdu, WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_correct_pdu_bytes);
    len += sprintf(page + off + len,     "    tccrc_err    : %u (%u)\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_tccrc_err_pdu, WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_tccrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    ethcrc_err   : %u (%u)\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_ethcrc_err_pdu, WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_ethcrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    nodesc_drop  : %u\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_nodesc_drop_pdu);
    len += sprintf(page + off + len,     "    len_viol_drop: %u\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_len_violation_drop_pdu);
    len += sprintf(page + off + len,     "    idle_bytes   : %u\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_idle_bytes);
    len += sprintf(page + off + len,     "    nonidle_cw   : %u\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_nonidle_cw);
    len += sprintf(page + off + len,     "    idle_cw      : %u\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_idle_cw);
    len += sprintf(page + off + len,     "    err_cw       : %u\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wrx_err_cw);
    len += sprintf(page + off + len,     "  TX \n");
    len += sprintf(page + off + len,     "    total_pdu    : %u (%u)\n", WAN_MIB_TABLE[PTM0HI_MIB_IDX].wtx_total_pdu, WAN_MIB_TABLE[PTM0HI_MIB_IDX].wtx_total_bytes);

    len += sprintf(page + off + len,     "ptmfast0hi\n");
    len += sprintf(page + off + len,     "  RX\n");
    len += sprintf(page + off + len,     "    correct      : %u (%u)\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_correct_pdu, WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_correct_pdu_bytes);
    len += sprintf(page + off + len,     "    tccrc_err    : %u (%u)\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_tccrc_err_pdu, WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_tccrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    ethcrc_err   : %u (%u)\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_ethcrc_err_pdu, WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_ethcrc_err_pdu_bytes);
    len += sprintf(page + off + len,     "    nodesc_drop  : %u\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_nodesc_drop_pdu);
    len += sprintf(page + off + len,     "    len_viol_drop: %u\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_len_violation_drop_pdu);
    len += sprintf(page + off + len,     "    idle_bytes   : %u\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_idle_bytes);
    len += sprintf(page + off + len,     "    nonidle_cw   : %u\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_nonidle_cw);
    len += sprintf(page + off + len,     "    idle_cw      : %u\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_idle_cw);
    len += sprintf(page + off + len,     "    err_cw       : %u\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wrx_err_cw);
    len += sprintf(page + off + len,     "  TX \n");
    len += sprintf(page + off + len,     "    total_pdu    : %u (%u)\n", WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wtx_total_pdu, WAN_MIB_TABLE[PTMFASTHI_MIB_IDX].wtx_total_bytes);
#endif

//    MOD_DEC_USE_COUNT;

    *eof = 1;

    return len;
}

static int proc_write_mib(struct file *file, const char *buf, unsigned long count, void *data)
{
    char str[64];
    char *p;
    int len, rlen;
//    u32 eth_clear;

//    MOD_INC_USE_COUNT;

    len = count < sizeof(str) ? count : sizeof(str) - 1;
    rlen = len - copy_from_user(str, buf, len);
    while ( rlen && str[rlen - 1] <= ' ' )
        rlen--;
    str[rlen] = 0;
    for ( p = str; *p && *p <= ' '; p++, rlen-- );
    if ( !*p )
    {
//        MOD_DEC_USE_COUNT;
        return 0;
    }

    if ( stricmp(p, "clear") == 0 || stricmp(p, "clear all") == 0
        || stricmp(p, "clean") == 0 || stricmp(p, "clean all") == 0 ) {
        memset((void *)WAN_MIB_TABLE, 0, WAN_MIB_TABLE_SIZE);
    }

//    MOD_DEC_USE_COUNT;

    return count;
}


static int proc_read_stats(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;

//    MOD_INC_USE_COUNT;

#if 0
    {
        unsigned int drop = *ENETS_IGDROP(1);
        unsigned int err  = *ENETS_IGERR(1);
        len += sprintf(page + off + len, "drop = %u, err = %u\n", drop, err);
    }
#endif

#if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
    /*
     *  update counters (eth2_get_stats)
     */

    eth2_dev.rx_drop_counter  += *ENETS_IGDROP(1);

    eth2_dev.stats.rx_packets  = ETH_MIB_TABLE->erx_pass_pdu;
    eth2_dev.stats.rx_bytes    = ETH_MIB_TABLE->erx_pass_bytes;
    eth2_dev.stats.rx_errors  += *ENETS_IGERR(1);
    eth2_dev.stats.rx_dropped  = ETH_MIB_TABLE->erx_dropdes_pdu + eth2_dev.rx_drop_counter;

    eth2_dev.stats.tx_packets  = ETH_MIB_TABLE->etx_total_pdu;
    eth2_dev.stats.tx_bytes    = ETH_MIB_TABLE->etx_total_bytes;
    eth2_dev.stats.tx_errors  += *ENETF_EGCOL(1);               /*  eth2_dev.stats.tx_errors is also updated in function eth2_hard_start_xmit   */
    eth2_dev.stats.tx_dropped += *ENETF_EGDROP(1);              /*  eth2_dev.stats.tx_dropped is updated in function eth2_hard_start_xmit       */
#endif

    /*
     *  print counters
     */

    len += sprintf(page + off + len, "ETH2 Stats\n");

#if defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER
    len += sprintf(page + off + len, "  Total\n");
    len += sprintf(page + off + len, "    rx_success        = %u\n", eth2_dev.rx_success);
    len += sprintf(page + off + len, "    rx_fail           = %u\n", eth2_dev.rx_fail);
    len += sprintf(page + off + len, "    rx_desc_read_pos  = %u, %u, %u, %u\n", eth2_dev.rx_desc_read_pos[0], eth2_dev.rx_desc_read_pos[1], eth2_dev.rx_desc_read_pos[2], eth2_dev.rx_desc_read_pos[3]);
    len += sprintf(page + off + len, "    tx_success        = %u\n", eth2_dev.tx_success);
    len += sprintf(page + off + len, "    tx_desc_alloc_pos = %u, %u, %u, %u\n", eth2_dev.tx_desc_alloc_pos[0], eth2_dev.tx_desc_alloc_pos[1], eth2_dev.tx_desc_alloc_pos[2], eth2_dev.tx_desc_alloc_pos[3]);
    len += sprintf(page + off + len, "  Driver\n");
    len += sprintf(page + off + len, "    rx_error          = %u\n", eth2_dev.rx_driver_level_error);
    len += sprintf(page + off + len, "    rx_drop           = %u\n", eth2_dev.rx_driver_level_drop);
    len += sprintf(page + off + len, "    tx_drop           = %u\n", eth2_dev.tx_driver_level_drop);
    len += sprintf(page + off + len, "  Firmware MIB\n");
    len += sprintf(page + off + len, "    erx_pass_pdu      = %u\n", ETH_MIB_TABLE->erx_pass_pdu);
    len += sprintf(page + off + len, "    erx_pass_bytes    = %u\n", ETH_MIB_TABLE->erx_pass_bytes);
    len += sprintf(page + off + len, "    erx_dropdes_pdu   = %u\n", ETH_MIB_TABLE->erx_dropdes_pdu);
    len += sprintf(page + off + len, "    etx_total_pdu     = %u\n", ETH_MIB_TABLE->etx_total_pdu);
    len += sprintf(page + off + len, "    etx_total_bytes   = %u\n", ETH_MIB_TABLE->etx_total_bytes);
    len += sprintf(page + off + len, "  Hardware\n");
    len += sprintf(page + off + len, "    ENETS_IGERR       = %lu\n", eth2_dev.stats.rx_errors - eth2_dev.rx_driver_level_error);
    len += sprintf(page + off + len, "    ENETS_IGDROP      = %u\n", eth2_dev.rx_drop_counter - eth2_dev.rx_driver_level_drop);
    len += sprintf(page + off + len, "    ENETS_PGCNT:UPAGE = %u\n", *ENETS_PGCNT(1) & 0xFF);
    len += sprintf(page + off + len, "    ENETF_EGCOL       = %lu\n", eth2_dev.stats.tx_errors);
    len += sprintf(page + off + len, "    ENETF_EGDROP      = %lu\n", eth2_dev.stats.tx_dropped - eth2_dev.tx_driver_level_drop);
    len += sprintf(page + off + len, "    ENETF_PGCNT:UPAGE = %u\n", *ENETF_PGCNT(1) & 0xFF);
    len += sprintf(page + off + len, "  EMA\n");
    len += sprintf(page + off + len, "    CMDBUF_VCNT       = %u\n", *EMA_CMDCNT & 0xFF);
    len += sprintf(page + off + len, "    DATABUF_UCNT      = %u\n", *EMA_DATACNT & 0x3FF);
    len += sprintf(page + off + len, "  Some Switches\n");
    len += sprintf(page + off + len, "    TX_CLK_INV        = %s\n", (*ETOP_CFG & ETOP_CFG_TCKINV1_ON(1)) ? "INV" : "NORM");
    len += sprintf(page + off + len, "    netif_Q_stopped   = %s\n", netif_queue_stopped(&eth2_net_dev) ? "stopped" : "running");
    len += sprintf(page + off + len, "    netif_running     = %s\n", netif_running(&eth2_net_dev) ? "running" : "stopped");
    len += sprintf(page + off + len, "    netdev_fc_xoff    = %s\n", (eth2_dev.fc_bit && test_bit(eth2_dev.fc_bit, &netdev_fc_xoff)) ? "off" : "on");
    len += sprintf(page + off + len, "    ERX_DMACH_ON/ETX_DMACH_ON\n");
    len += sprintf(page + off + len, "      ERX_DMACH_ON    = %04X\n", *CFG_ERX_DMACH_ON & 0xFFFF);
    len += sprintf(page + off + len, "      ETX_DMACH_ON    = %04X\n", *CFG_ETX_DMACH_ON & 0xFFFF);
    len += sprintf(page + off + len, "    IGU1_IER\n");
    len += sprintf(page + off + len, "      rx_irq = %04X, IER = %04X\n", eth2_dev.rx_irq & 0xFFFF, *MBOX_IGU1_IER & 0xFFFF);
    len += sprintf(page + off + len, "      tx_irq = %04X, IER = %04X\n", eth2_dev.tx_irq & 0xFFFF, *MBOX_IGU1_IER >> 16);
    len += sprintf(page + off + len, "  MDIO and ENET Configure\n");
    len += sprintf(page + off + len, "    ETOP_MDIO_CFG     = %08X\n", *ETOP_MDIO_CFG);
    len += sprintf(page + off + len, "    ETOP_CFG          = %08X\n", *ETOP_CFG);
    len += sprintf(page + off + len, "    ENET_MAC_CFG      = %08X\n", *ENET_MAC_CFG(1));
    len += sprintf(page + off + len, "  Mailbox Signal Wait Loop\n");
    len += sprintf(page + off + len, "    RX Wait Loop      = %d\n", eth2_dev.rx_desc_update_wait_loop);
    len += sprintf(page + off + len, "    TX Wait Loop      = %d\n", eth2_dev.tx_desc_update_wait_loop);
#endif  //  defined(ENABLE_DEBUG_COUNTER) && ENABLE_DEBUG_COUNTER

//    MOD_DEC_USE_COUNT;

    return len;
}

#if defined(ENABLE_DBG_PROC) && ENABLE_DBG_PROC
static int proc_read_dbg(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;

//    MOD_INC_USE_COUNT;

    if ( dbg_enable )
        len += sprintf(page + off + len, "debug enabled (dbg_enable = %08X)\n", dbg_enable);
    else
        len += sprintf(page + off + len, "debug disabled\n");

    len += sprintf(page + off + len, "PTM tx quota: %d\n", eth2_tx_quota);
//    len += sprintf(page + off + len, "ETH2 tx CRC append: %s\n", eth2_tx_crc_append ? "enable" : "disable");
    len += sprintf(page + off + len, "CRC settings:\n");
    len += sprintf(page + off + len, "  rx_eth_crc_present: %s\n", CFG_ETH_EFMTC_CRC->rx_eth_crc_present ? "yes" : "no");
    len += sprintf(page + off + len, "  rx_eth_crc_check:   %s\n", CFG_ETH_EFMTC_CRC->rx_eth_crc_check ? "on" : "off");
    len += sprintf(page + off + len, "  rx_tc_crc_check:    %s\n", CFG_ETH_EFMTC_CRC->rx_tc_crc_check ? "on" : "off");
    len += sprintf(page + off + len, "  rx_tc_crc_len:      %d\n", CFG_ETH_EFMTC_CRC->rx_tc_crc_len);
    len += sprintf(page + off + len, "  tx_eth_crc_gen:     %s\n", CFG_ETH_EFMTC_CRC->tx_eth_crc_gen ? "on" : "off");
    len += sprintf(page + off + len, "  tx_tc_crc_gen:      %s\n", CFG_ETH_EFMTC_CRC->tx_tc_crc_gen ? "on" : "off");
    len += sprintf(page + off + len, "  tx_tc_crc_len:      %d\n", CFG_ETH_EFMTC_CRC->tx_tc_crc_len);

//    MOD_DEC_USE_COUNT;

    *eof = 1;

    return len;
}

static int proc_write_dbg(struct file *file, const char *buf, unsigned long count, void *data)
{
    char str[64];
    char *p, *p1;

    int len, rlen;

//    MOD_INC_USE_COUNT;

    len = count < sizeof(str) ? count : sizeof(str) - 1;
    rlen = len - copy_from_user(str, buf, len);
    while ( rlen && str[rlen - 1] <= ' ' )
        rlen--;
    str[rlen] = 0;
    for ( p = str; *p && *p <= ' '; p++, rlen-- );
    if ( !*p )
    {
//        MOD_DEC_USE_COUNT;
        return 0;
    }

    p1 = p;
    while (*p1 > ' ')
        *p1 ++;

    if(*p1) {
        *p1 = 0;
        *p1 ++;
    }

    if ( stricmp(p, "enable") == 0 )
        dbg_enable = 1;
    else if ( stricmp(p, "disable") == 0 )
        dbg_enable = 0;
    else if ( stricmp(p, "notxq") == 0 )
        eth2_tx_quota = -1;
//    else if ( stricmp(p, "enable_crc") == 0 )
//        eth2_tx_crc_append = 1;
//    else if ( stricmp(p, "disable_crc") == 0 )
//        eth2_tx_crc_append = 0;
    else if ( stricmp(p, "txq") == 0 ) {
        p = p1;
        int new_q = -1;

        while (*p && *p <= ' ')
            p++;

        while( *p >= '0' && *p <= '9') {
            if( new_q < 0 )
                new_q = 0;
            new_q = new_q * 10 + *p - '0';
            p ++;
        }

        if(new_q >= 0)
            eth2_tx_quota = new_q;
    }


//    MOD_DEC_USE_COUNT;

    return count;
}
#endif

static INLINE int stricmp(const char *p1, const char *p2)
{
    int c1, c2;

    while ( *p1 && *p2 )
    {
        c1 = *p1 >= 'A' && *p1 <= 'Z' ? *p1 + 'a' - 'A' : *p1;
        c2 = *p2 >= 'A' && *p2 <= 'Z' ? *p2 + 'a' - 'A' : *p2;
        if ( (c1 -= c2) )
            return c1;
        p1++;
        p2++;
    }

    return *p1 - *p2;
}

static INLINE int strincmp(const char *p1, const char *p2, int n)
{
    int c1 = 0, c2;

    while ( n && *p1 && *p2 )
    {
        c1 = *p1 >= 'A' && *p1 <= 'Z' ? *p1 + 'a' - 'A' : *p1;
        c2 = *p2 >= 'A' && *p2 <= 'Z' ? *p2 + 'a' - 'A' : *p2;
        if ( (c1 -= c2) )
            return c1;
        p1++;
        p2++;
        n--;
    }

    return n ? *p1 - *p2 : c1;
}

static INLINE int get_token(char **p1, char **p2, int *len, int *colon)
{
    int tlen = 0;

    while ( *len && !((**p1 >= 'A' && **p1 <= 'Z') || (**p1 >= 'a' && **p1<= 'z')) )
    {
        (*p1)++;
        (*len)--;
    }
    if ( !*len )
        return 0;

    if ( *colon )
    {
        *colon = 0;
        *p2 = *p1;
        while ( *len && **p2 > ' ' && **p2 != ',' )
        {
            if ( **p2 == ':' )
            {
                *colon = 1;
                break;
            }
            (*p2)++;
            (*len)--;
            tlen++;
        }
        **p2 = 0;
    }
    else
    {
        *p2 = *p1;
        while ( *len && **p2 > ' ' && **p2 != ',' )
        {
            (*p2)++;
            (*len)--;
            tlen++;
        }
        **p2 = 0;
    }

    return tlen;
}

static INLINE int get_number(char **p, int *len, int is_hex)
{
    int ret = 0;
    int n = 0;

    if ( is_hex )
    {
        while ( *len && ((**p >= '0' && **p <= '9') || (**p >= 'a' && **p <= 'f') || (**p >= 'A' && **p <= 'F')) )
        {
            if ( **p >= '0' && **p <= '9' )
                n = **p - '0';
            else if ( **p >= 'a' && **p <= 'f' )
                n = **p - 'a' + 10;
            else if ( **p >= 'A' && **p <= 'F' )
                n = **p - 'A' + 10;
            ret = (ret << 4) | n;
            (*p)++;
            (*len)--;
        }
    }
    else
    {
        while ( *len && **p >= '0' && **p <= '9' )
        {
            n = **p - '0';
            ret = ret * 10 + n;
            (*p)++;
            (*len)--;
        }
    }

    return ret;
}

static INLINE void ignore_space(char **p, int *len)
{
    while ( *len && (**p <= ' ' || **p == ':' || **p == '.' || **p == ',') )
    {
        (*p)++;
        (*len)--;
    }
}



#if defined(DEBUG_MEM_PROC) && DEBUG_MEM_PROC
static int proc_write_mem(struct file *file, const char *buf, unsigned long count, void *data)
{
    char *p1, *p2;
    int len;
    int colon;
    unsigned long *p;
    char local_buf[1024];
    int i, n;

//    MOD_INC_USE_COUNT;

    len = sizeof(local_buf) < count ? sizeof(local_buf) : count;
    len = len - copy_from_user(local_buf, buf, len);
    local_buf[len] = 0;

    p1 = local_buf;
    colon = 1;
   while ( get_token(&p1, &p2, &len, &colon) )
    {
        if ( stricmp(p1, "w") == 0 || stricmp(p1, "write") == 0 || stricmp(p1, "r") == 0 || stricmp(p1, "read") == 0 )
            break;

        p1 = p2;
        colon = 1;
    }

    if ( *p1 == 'w' )
    {
        ignore_space(&p2, &len);
        p = (unsigned long *)get_number(&p2, &len, 1);
        if ( (u32)p >= KSEG0 )
            while ( 1 )
            {
                ignore_space(&p2, &len);
                if ( !len || !((*p2 >= '0' && *p2 <= '9') || (*p2 >= 'a' && *p2 <= 'f') || (*p2 >= 'A' && *p2 <= 'F')) )
                    break;

                *p++ = (u32)get_number(&p2, &len, 1);
            }
    }
    else if ( *p1 == 'r' )
    {
        ignore_space(&p2, &len);
        p = (unsigned long *)get_number(&p2, &len, 1);
        if ( (u32)p >= KSEG0 )
        {
            ignore_space(&p2, &len);
            n = (int)get_number(&p2, &len, 0);
            if ( n )
            {
                n += (((int)p >> 2) & 0x03);
                p = (unsigned long *)((u32)p & ~0x0F);
                for ( i = 0; i < n; i++ )
                {
                    if ( (i & 0x03) == 0 )
                        printk("%08X:", (u32)p);
                    printk(" %08X", (u32)*p++);
                    if ( (i & 0x03) == 0x03 )
                        printk("\n");
                }
                if ( (n & 0x03) != 0x00 )
                    printk("\n");
            }
        }
    }

//    MOD_DEC_USE_COUNT;

    return count;
}
#endif


#if (defined(DEBUG_DUMP_RX_SKB) && DEBUG_DUMP_RX_SKB) || (defined(DEBUG_DUMP_TX_SKB) && DEBUG_DUMP_TX_SKB)
static INLINE void dump_skb(struct sk_buff *skb, int is_rx)
{
    int i;


//    if ( !dbg_enable && ! rx_dmach_stopped )
    if ( !dbg_enable )
        return;

//    if(rx_dmach_stopped)
//        rx_dmach_stopped = 0;

   printk(is_rx ? "RX path --- sk_buff\n" : "TX path --- sk_buff\n");
    printk("  skb->data = %08X, skb->tail = %08X, skb->len = %d\n", (u32)skb->data, (u32)skb->tail, (int)skb->len);
    for ( i = 1; i <= skb->len; i++ )
    {
        if ( i % 16 == 1 )
            printk("  %4d:", i - 1);
        printk(" %02X", (int)(*((char*)skb->data + i - 1) & 0xFF));
        if ( i % 16 == 0 )
            printk("\n");
    }
    if ( (i - 1) % 16 != 0 )
        printk("\n");

}
#endif

#if defined(DEBUG_DUMP_ETOP_REGISTER) && DEBUG_DUMP_ETOP_REGISTER
static INLINE void dump_etop0_reg()
{
    printk("ETOP0 Registers:\n");

    printk("  ETOP_CFG\n");
    printk("    MII0:    %s\n", (char*)(!(*ETOP_CFG & 0x0001) ? "ON" : "OFF"));
    printk("    Mode:    %s\n", (char*)(!(*ETOP_CFG & 0x0002) ? "MII" : "Rev MII"));
    printk("    Turbo:   %s\n", (char*)(!(*ETOP_CFG & 0x0004) ? "Normal" : "Turbo"));
    printk("    Ingress: %s\n", (char*)(!(*ETOP_CFG & 0x0040) ? "Disable" : "Enable"));
    printk("    Egress:  %s\n", (char*)(!(*ETOP_CFG & 0x0100) ? "Disable" : "Enable"));
    printk("    Clock:   %s\n", (char*)(!(*ETOP_CFG & 0x0400) ? "Normal" : "Inversed"));

    printk("  ENETS_CFG\n");
    printk("    HDLEN:   %d\n", (*ENETS_CFG(0) >> 18) & 0x7F);
    printk("    PNUM:    %d\n", *ENETS_CFG(0) & 0xFF);

    printk("  ENETS_PGCNT\n");
    printk("    DSRC:    %s\n", (char*)((*ENETS_PGCNT(0) & 0x040000) ? ((*ENETS_PGCNT(0) & 0x020000) ? "Cross" : "Local") : ((*ENETS_PGCNT(0) & 0x020000) ? "PP32" : "DPLUS")));
    printk("    UPAGE:   %d\n", *ENETS_PGCNT(0) & 0xFF);

    printk("  ENETS_PKTCNT\n");
    printk("    UPKT:    %d\n", *ENETS_PKTCNT(0) & 0xFF);
}

static INLINE void dump_etop1_reg()
{
    printk("ETOP1 Registers:\n");

    printk("  ETOP_CFG\n");
    printk("    MII0:     %s\n", (char*)(!(*ETOP_CFG & 0x0008) ? "ON" : "OFF"));
    printk("    Mode:     %s\n", (char*)(!(*ETOP_CFG & 0x0010) ? "MII" : "Rev MII"));
    printk("    Turbo:    %s\n", (char*)(!(*ETOP_CFG & 0x0020) ? "Normal" : "Turbo"));
    printk("    Ingress:  %s\n", (char*)(!(*ETOP_CFG & 0x0080) ? "Disable" : "Enable"));
    printk("    Egress:   %s\n", (char*)(!(*ETOP_CFG & 0x0200) ? "Disable" : "Enable"));
    printk("    Clock:    %s\n", (char*)(!(*ETOP_CFG & 0x0800) ? "Normal" : "Inversed"));

    printk("  ENETS_CFG\n");
    printk("    HDLEN:   %d\n", (*ENETS_CFG(1) >> 18) & 0x7F);
    printk("    PNUM:    %d\n", *ENETS_CFG(1) & 0xFF);

    printk("  ENETS_PGCNT\n");
    printk("    DSRC:    %s\n", (char*)((*ENETS_PGCNT(1) & 0x040000) ? ((*ENETS_PGCNT(1) & 0x020000) ? "Cross" : "Local") : ((*ENETS_PGCNT(1) & 0x020000) ? "PP32" : "DPLUS")));
    printk("    UPAGE:   %d\n", *ENETS_PGCNT(1) & 0xFF);

    printk("  ENETS_PKTCNT\n");
    printk("    UPKT:    %d\n", *ENETS_PKTCNT(1) & 0xFF);
}

static INLINE void dump_etop_reg()
{
    dump_etop0_reg();
    dump_etop1_reg();
}
#endif



/*
 * ####################################
 *           Global Function
 * ####################################
 */


/*
 * ####################################
 *           Init/Cleanup API
 * ####################################
 */

/*
 *  Description:
 *    Initialize global variables, PP32, comunication structures, register IRQ
 *    and register device.
 *  Input:
 *    none
 *  Output:
 *    0    --- successful
 *    else --- failure, usually it is negative value of error code
 */
int __init danube_ppe_ptm_init(void)
{
    int ret;
    int i = 0;

    printk("Loading 2nd ETH driver... ");

#ifdef MODULE
    reset_ppe();
#endif

//    {
//        loop = 1;
//        while(loop);
//    }
    ret = init_local_variables();

    check_parameters();

    ret = init_eth2_dev();
    if ( ret )
        goto INIT_ETH2_DEV_FAIL;

    clear_share_buffer();

    config_ppe_hw();

    init_tables();

    /*  Disable RX  */
//    *CFG_ERX_DMACH_ON = 0x00;

    /*  create device   */
    for ( i = 0; i < sizeof(eth2_net_dev) / sizeof(*eth2_net_dev); i++ )
    {
        ret = register_netdev(eth2_net_dev + i);
        if ( ret )
            goto REGISTER_NETDEV_FAIL;
    }

    /*  register interrupt handler  */
    ret = request_irq(PPE_MAILBOX_IGU1_INT, mailbox_irq_handler, SA_INTERRUPT, "eth2_dma_isr", NULL);
    if ( ret )
        goto REQUEST_IRQ_PPE_MAILBOX_IGU1_INT_FAIL;

    init_chip(0);
    eth2_dev.rx_irq =  *MBOX_IGU1_IER & 0x0000FFFF;
    eth2_dev.tx_irq = (*MBOX_IGU1_IER & 0xFFFF0000) >> 16;

    ppe_bp_setup();

    ret = pp32_start();
    if ( ret )
        goto PP32_START_FAIL;

    /*  careful, PPE firmware may set some registers, recover them here */
    *MBOX_IGU1_IER = (eth2_dev.tx_irq << 16) | eth2_dev.rx_irq;

    /*  create proc file    */
    proc_file_create();

    printk("MBOX_IGU1_IER: %08X\n", *MBOX_IGU1_IER);

    printk("init succeeded (firmware version  %d.%d.%d.%d.%d.%d)\n", (int)FW_VER_ID->family, (int)FW_VER_ID->fwtype, (int)FW_VER_ID->interface, (int)FW_VER_ID->fwmode, (int)FW_VER_ID->major, (int)FW_VER_ID->minor);
    return 0;

PP32_START_FAIL:
    free_irq(PPE_MAILBOX_IGU1_INT, NULL);
REQUEST_IRQ_PPE_MAILBOX_IGU1_INT_FAIL:
REGISTER_NETDEV_FAIL:
    while ( i-- )
        unregister_netdev(eth2_net_dev + i);
    clear_eth2_dev();
INIT_ETH2_DEV_FAIL:
    printk(" init failed\n");
    return ret;
}

/*
 *  Description:
 *    Release memory, free IRQ, and deregister device.
 *  Input:
 *    none
 *  Output:
 *    none
 */
void __exit danube_ppe_ptm_exit(void)
{
    int i;

    proc_file_delete();

    pp32_stop();

    free_irq(PPE_MAILBOX_IGU1_INT, NULL);

    for ( i = sizeof(eth2_net_dev) / sizeof(*eth2_net_dev) - 1; i >= 0; i-- )
        unregister_netdev(eth2_net_dev + i);

    clear_eth2_dev();
}

#ifndef MODULE
static int __init danube_ethaddr_setup(char *line)
{
    char *ep;
    int i;
    
    memset(MY_ETHADDR, 0, sizeof(MY_ETHADDR));
    for ( i = 0; i < 6; i++ )
    {
        MY_ETHADDR[i] = line ? simple_strtoul(line, &ep, 16) : 0;
        if ( line )
            line = *ep ? ep + 1 : ep;
    }
    dbg("2nd eth mac address %02X-%02X-%02X-%02X-%02X-%02X\n",
        MY_ETHADDR[0],
        MY_ETHADDR[1],
        MY_ETHADDR[2],
        MY_ETHADDR[3],
        MY_ETHADDR[4],
        MY_ETHADDR[5]);

    return 0;
}
#endif


module_init(danube_ppe_ptm_init);
module_exit(danube_ppe_ptm_exit);
__setup("ethaddr=", danube_ethaddr_setup);
