/*
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
//-----------------------------------------------------------------------
//Description:	
// AMAZON_SE specific setup
//-----------------------------------------------------------------------
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/model.h>

#include <asm/cpu.h>
#include <asm/bootinfo.h>
#include <asm/irq.h>
#include <asm/time.h>
#include <asm/amazon_se/amazon_se.h>
#include <asm/amazon_se/irq.h>
#include <asm/amazon_se/serial.h>
#include <asm/amazon_se/emulation.h>
#include <asm/mipsregs.h>

static unsigned int r4k_offset; /* Amount to increment compare reg each time */
static unsigned int r4k_cur;    /* What counter should be at next timer irq */

extern void amazon_se_reboot_setup(void); 
extern int kgdb_serial_init(void);
void prom_printf(const char * fmt, ...);
void __init bus_error_init(void) { /* nothing */ }

int (*adsl_link_notify)(int);

/* modify amazon_se register with a small delay after writing to the register */
/* this delay is located in the same cache line as the write access, to prevent */
/* erroneous updates of the caches (seen with CGU_DIVCR register !) */
/* Input: reg = address of register */
/*        andmask = bits to clear in the register */
/*        ormask  = bits to set in the register */
static inline unsigned long modify_amazon_se_reg(volatile unsigned int *reg, 
                                            unsigned long andmask, 
                                            unsigned long ormask)
{
    int count=15000;
    unsigned long regValue = 0;

    /* equivalent C-Code: (*reg) = ((*reg) & (~andmask)) | (ormask); */
    asm volatile (" .set noreorder\n"
                  "\tlw\t%2,(%3)\n"
                  "\tand\t%2,%2,%5\n"
                  "\tor\t%2,%2,%6\n"
                  "\t.align\t4\n" /* align on cache line boundary */
                  "\tsw\t%2,(%3)\n"
                  "\t0:\n"
                  "\tbnez\t%4,0b\n"
                  "\taddiu\t%4,%4,-1\n" /* branch delay */
                  "\t.set\treorder\n"
                  :
                  "=r" (regValue), /* %0 - output */
                  "=r" (count)  /* %1 - count is modified */
                  :
                  "0" (regValue),    /* %2 - same register as %0 */
                  "r" (reg),    /* %3 - address to modify */
                  "1" (count),  /* %4 - number of delay cycles */
                  "r" (~andmask), /* %5 - value to 'and' with on-chip register */
                  "r" (ormask)  /* %6 - value to 'or' with on-chip register */
/*                    : */
                  /* nothing clobbered */
        );

    return regValue;
}
#if 0
unsigned int amazon_se_get_ddr_hz(void)
{
	switch((*AMAZON_SE_CGU_SYS) & 0x3){
		case 0:
			return CLOCK_167M;
		case 1:
			return CLOCK_133M;
		case 2:
			return CLOCK_111M;
	}
  return CLOCK_83M;
}
#endif

/* the CPU clock rate - lifted from u-boot */
unsigned int
amazon_se_get_cpu_hz(void)
{
#ifdef CONFIG_USE_EMULATOR
	return EMULATOR_CPU_SPEED;
#else //NOT CONFIG_USE_EMULATOR
//      unsigned int ddr_clock=amazon_se_get_ddr_hz();
        switch((*AMAZON_SE_CGU_SYS) & 0x20){
                case 0:
                        return CLOCK_133M;
                case 32:
                        return CLOCK_266M;
                default:
                        return CLOCK_133M;
        }
#endif
}

/* the FPI clock rate - lifted from u-boot */
unsigned int
amazon_se_get_fpi_hz(void)
{

#ifdef CONFIG_USE_EMULATOR
	unsigned int  clkCPU;
	clkCPU = amazon_se_get_cpu_hz();
	return clkCPU >> 2;
#else //NOT CONFIG_USE_EMULATOR
//	unsigned int ddr_clock=amazon_se_get_ddr_hz();
//	if ((*AMAZON_SE_CGU_SYS) & 0x40){
//		return ddr_clock >> 1;
//	}
	return CLOCK_133M;
#endif
}

/* get the CPU version number  - based on sysLib.c from VxWorks sources */
/* this doesn't really belong here, but it's a convenient location */
unsigned int
amazon_se_get_cpu_ver(void)
{
	static unsigned int cpu_ver = 0;

	if (cpu_ver == 0)
		cpu_ver = *AMAZON_SE_MCD_CHIPID & 0xFFFFF000;
	return cpu_ver;
}



EXPORT_SYMBOL(amazon_se_get_cpu_hz);
EXPORT_SYMBOL(amazon_se_get_fpi_hz);
EXPORT_SYMBOL(amazon_se_get_cpu_ver);

void amazon_se_time_init(void)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,31) 
	#ifdef CONFIG_USE_EMULATOR
        //we do some hacking here to give illusion we have a faster counter frequency so that
        // the time interrupt happends less frequently
        mips_hpt_frequency = amazon_se_get_cpu_hz()/2 * 25;
	#else //not CONFIG_USE_EMULATOR
        mips_hpt_frequency = amazon_se_get_cpu_hz()/2;
	#endif //not CONFIG_USE_EMULATOR
        r4k_offset = mips_hpt_frequency/HZ;
        printk("mips_hpt_frequency:%d\n",mips_hpt_frequency);
        printk("r4k_offset: %08x(%d)\n",r4k_offset,r4k_offset);
#else
	#ifdef CONFIG_USE_EMULATOR
	//we do some hacking here to give illusion we have a faster counter frequency so that
	// the time interrupt happends less frequently
	mips_counter_frequency = amazon_se_get_cpu_hz()/2 * 25;
	#else //not CONFIG_USE_EMULATOR
	mips_counter_frequency = amazon_se_get_cpu_hz()/2;
	#endif //not CONFIG_USE_EMULATOR
	r4k_offset = mips_counter_frequency/HZ;
	printk("mips_counter_frequency:%d\n",mips_counter_frequency);
	printk("r4k_offset: %08x(%d)\n",r4k_offset,r4k_offset);
#endif /* kernel version */
}




#ifdef CONFIG_HIGH_RES_TIMERS
extern int hr_time_resolution;

/* ISR GPTU Timer 6 for high resolution timer */
void amazon_se_timer6_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	timer_interrupt(AMAZON_SE_TIMER6_INT, NULL, regs);
}

static struct irqaction hrt_irqaction = {
	amazon_se_timer6_interrupt,
	SA_INTERRUPT,
	0,
	"hrt",
	NULL,
	NULL
};

#endif //CONFIG_HIGH_RES_TIMERS

/*
 * THe CPU counter for System timer, set to HZ
 * GPTU Timer 6 for high resolution timer, set to hr_time_resolution
 * Also misuse this routine to print out the CPU type and clock.
 */
//void amazon_se_timer_setup(struct irqaction *irq)
void __init plat_timer_setup(struct irqaction *irq)
{
	/* cpu counter for timer interrupts */
	irq->handler = no_action;     /* we use our own handler */
	setup_irq(MIPS_CPU_TIMER_IRQ, irq);
	 /* to generate the first timer interrupt */
	r4k_cur = (read_c0_count() + r4k_offset);
	write_c0_compare(r4k_cur);

#ifdef CONFIG_HIGH_RES_TIMERS	
	/* GPTU timer 6 */
	int retval;
	if ( hr_time_resolution > 200000000 || hr_time_resolution < 40) {
		prom_printf("hr_time_resolution is out of range, HIGH_RES_TIMER is diabled.\n");
		return;
	}
	
	/* enable the timer in the PMU */
        *(AMAZON_SE_PMU_PWDCR) = (*(AMAZON_SE_PMU_PWDCR))| AMAZON_SE_PMU_PWDCR_GPT|AMAZON_SE_PMU_PWDCR_FPI;
	/* setup the GPTU for timer tick  f_fpi == f_gptu*/
	*(AMAZON_SE_GPTU_CLC) = 0x100;

#ifdef CONFIG_USE_EMULATOR
	//reload value = fpi/(HZ * P), timer mode, Prescaler = 4 ( T6I = 000, T6BPS2 = 0)	
	*(AMAZON_SE_GPTU_CAPREL) = (amazon_se_get_fpi_hz()*hr_time_resolution/1000000000)>>2;
	*(AMAZON_SE_GPTU_T6CON) = 0x80C0;
#else //not CONFIG_USE_EMULATOR
	*(AMAZON_SE_GPTU_CAPREL) = 0xffff;
	*(AMAZON_SE_GPTU_T6CON) = 0x80C0;
#endif //not CONFIG_USE_EMULATOR
	retval = setup_irq(AMAZON_SE_TIMER6_INT,&hrt_irqaction);
	if (retval){
		prom_printf("reqeust_irq failed %d. HIGH_RES_TIMER is diabled\n",AMAZON_SE_TIMER6_INT);		
	}
#endif //CONFIG_HIGH_RES_TIMERS		

}

static char buf[1024];

void prom_printf(const char * fmt, ...)
{
	va_list args;
	int l;
	char *p, *buf_end;

	/* Low level, brute force, not SMP safe... */
	va_start(args, fmt);
	l = vsprintf(buf, fmt, args); /* hopefully i < sizeof(buf) */
	va_end(args);
	buf_end = buf + l;
	
	for (p = buf; p < buf_end; p++) {
		/* Wait for FIFO to empty */
#ifdef CONFIG_IFX_ASC_CONSOLE_ASC0
		while ((((*AMAZON_SE_ASC0_FSTAT)& ASCFSTAT_TXFFLMASK) >> ASCFSTAT_TXFFLOFF) != 0x00) ; 
		/* Crude cr/nl handling is better than none */
		if(*p == '\n') *AMAZON_SE_ASC0_TBUF=('\r');
		*AMAZON_SE_ASC0_TBUF=(*p);
#else
		while ((((*AMAZON_SE_ASC1_FSTAT)& ASCFSTAT_TXFFLMASK) >> ASCFSTAT_TXFFLOFF) != 0x00) ; 
		/* Crude cr/nl handling is better than none */
		if(*p == '\n') *AMAZON_SE_ASC1_TBUF=('\r');
		*AMAZON_SE_ASC1_TBUF=(*p);
#endif
	}

}

EXPORT_SYMBOL(prom_printf);

u32 chipid;

//void __init amazon_se_setup(void)
//void __init plat_setup(void)
void __init plat_mem_setup(void)
{	

	u32 status;
	//TODO
//	prom_printf("TODO: chip version\n");	
	/* clear RE bit*/
	status = read_c0_status();
	status &= (~(1<<25));
	write_c0_status(status);
	clear_c0_status(ST0_RE);
        chipid = *AMAZON_SE_MPS_CHIPID;

	*AMAZON_SE_BIU_B2S &= ~(1 << 1);
        *AMAZON_SE_MC_PRIO &= ~(3 << AMAZON_SE_MC_PRIO_AHB_SHIFT);
        *AMAZON_SE_MC_PRIO |= (AMAZON_SE_MC_PRIO_3 << AMAZON_SE_MC_PRIO_AHB_SHIFT);

	amazon_se_reboot_setup();
	board_time_init = amazon_se_time_init;
//	plat_timer_setup = amazon_se_timer_setup;
#ifdef CONFIG_KGDB
	kgdb_serial_init();
	prom_printf("\n===>Please connect GDB to console tty0\n");
#endif
}
