/*
 *  Gary Jennejohn (C) 2003 <gj@denx.de>
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * Routines for generic manipulation of the interrupts found on the 
 * AMAZON_SE boards.
 */
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/kernel_stat.h>
#include <linux/module.h>

#include <asm/bootinfo.h>
#include <asm/irq.h>
#include <asm/time.h>
#include <asm/amazon_se/amazon_se.h>
#include <asm/amazon_se/irq.h>
#include <asm/traps.h>
#ifdef CONFIG_KGDB
#include <asm/gdb-stub.h>
#endif

#if 0 
extern void do_IRQ(int irq);//, struct pt_regs *regs);
#endif
#undef AMAZON_SE_INT_DEBUG_MSG
#ifdef AMAZON_SE_INT_DEBUG_MSG
#define AMAZON_SE_INT_DMSG(fmt,args...) printk("%s: " fmt, __FUNCTION__ , ##args)
#else
#define AMAZON_SE_INT_DMSG(x...)
#endif 

void disable_amazon_se_irq(unsigned int irq_nr)
{
  AMAZON_SE_INT_DMSG("disable_amazon_se_irq: %d\n", irq_nr);
  /* have to access the correct register here */
  if (irq_nr <= INT_NUM_IM0_IRL31 && irq_nr >= INT_NUM_IM0_IRL0)
  {
    /* access IM0 */
    *AMAZON_SE_ICU_IM0_IER &= (~AMAZON_SE_ICU_IM0_IER_IR(irq_nr));

  }else if (irq_nr <= INT_NUM_IM1_IRL31 && irq_nr >= INT_NUM_IM1_IRL0)
  {
    /* access IM1 */
    *AMAZON_SE_ICU_IM1_IER &= (~AMAZON_SE_ICU_IM1_IER_IR(irq_nr - 32));
  }else if (irq_nr <= INT_NUM_IM2_IRL31 && irq_nr >= INT_NUM_IM2_IRL0)
  {
    /* access IM2 */
    *AMAZON_SE_ICU_IM2_IER &= (~AMAZON_SE_ICU_IM2_IER_IR(irq_nr - 64));
  }else if (irq_nr <= INT_NUM_IM3_IRL31 && irq_nr >= INT_NUM_IM3_IRL0)
  {
    /* access IM3 */
    *AMAZON_SE_ICU_IM3_IER &= (~AMAZON_SE_ICU_IM3_IER_IR((irq_nr - 96)));
  }
}

EXPORT_SYMBOL(disable_amazon_se_irq);

void mask_and_ack_amazon_se_irq(unsigned int irq_nr)
{
  AMAZON_SE_INT_DMSG("mask_and_ack_amazon_se_irq: %d\n", irq_nr);
  /* have to access the correct register here */
  if (irq_nr <= INT_NUM_IM0_IRL31 && irq_nr >= INT_NUM_IM0_IRL0)
  {
    /* access IM0 */
    /* mask */
    *AMAZON_SE_ICU_IM0_IER &= ~AMAZON_SE_ICU_IM0_IER_IR(irq_nr);
    /* ack */
    *AMAZON_SE_ICU_IM0_ISR = AMAZON_SE_ICU_IM0_ISR_IR(irq_nr);
  }else if (irq_nr <= INT_NUM_IM1_IRL31 && irq_nr >= INT_NUM_IM1_IRL0)
  {
    /* access IM1 */
    /* mask */
    *AMAZON_SE_ICU_IM1_IER &= ~AMAZON_SE_ICU_IM1_IER_IR(irq_nr - 32);
    /* ack */
    *AMAZON_SE_ICU_IM1_ISR = AMAZON_SE_ICU_IM1_ISR_IR(irq_nr - 32);
  }else if (irq_nr <= INT_NUM_IM2_IRL31 && irq_nr >= INT_NUM_IM2_IRL0)
  {
    /* access IM2 */
    /* mask */
    *AMAZON_SE_ICU_IM2_IER &= ~AMAZON_SE_ICU_IM2_IER_IR(irq_nr - 64);
    /* ack */
    *AMAZON_SE_ICU_IM2_ISR = AMAZON_SE_ICU_IM2_ISR_IR(irq_nr - 64);
  }else if (irq_nr <= INT_NUM_IM3_IRL31 && irq_nr >= INT_NUM_IM3_IRL0)
  {
    /* access IM3 */
    /* mask */
    *AMAZON_SE_ICU_IM3_IER &= ~AMAZON_SE_ICU_IM3_IER_IR(irq_nr - 96);
    /* ack */
    *AMAZON_SE_ICU_IM3_ISR = AMAZON_SE_ICU_IM3_ISR_IR(irq_nr - 96);
  }
}

EXPORT_SYMBOL(mask_and_ack_amazon_se_irq);

void enable_amazon_se_irq(unsigned int irq_nr)
{
  AMAZON_SE_INT_DMSG("enable_amazon_se_irq: %d\n", irq_nr);
  /* have to access the correct register here */
  if (irq_nr <= INT_NUM_IM0_IRL31 && irq_nr >= INT_NUM_IM0_IRL0){
    /* access IM0 */
    *AMAZON_SE_ICU_IM0_IER |= AMAZON_SE_ICU_IM0_IER_IR(irq_nr);
  }else if (irq_nr <= INT_NUM_IM1_IRL31 && irq_nr >= INT_NUM_IM1_IRL0)
  {
    /* access IM1 */
    *AMAZON_SE_ICU_IM1_IER |= AMAZON_SE_ICU_IM1_IER_IR(irq_nr - 32);
  }else if (irq_nr <= INT_NUM_IM2_IRL31 && irq_nr >= INT_NUM_IM2_IRL0)
  {
    /* access IM2 */
    *AMAZON_SE_ICU_IM2_IER |= AMAZON_SE_ICU_IM2_IER_IR(irq_nr - 64);
  }else if (irq_nr <= INT_NUM_IM3_IRL31 && irq_nr >= INT_NUM_IM3_IRL0)
  {
    /* access IM3 */
    *AMAZON_SE_ICU_IM3_IER |= AMAZON_SE_ICU_IM3_IER_IR((irq_nr - 96));
  }
}

EXPORT_SYMBOL(enable_amazon_se_irq);

static unsigned int startup_amazon_se_irq(unsigned int irq)
{
	enable_amazon_se_irq(irq);
	return 0; /* never anything pending */
}

#define shutdown_amazon_se_irq	disable_amazon_se_irq

static void end_amazon_se_irq(unsigned int irq)
{
  if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS))){
    enable_amazon_se_irq(irq);
  }
}

static struct irq_chip amazon_se_irq_type = {
	.name = "AMAZON_SE",
	.startup = startup_amazon_se_irq,
	.shutdown = shutdown_amazon_se_irq,
	.enable = enable_amazon_se_irq,
	.disable = disable_amazon_se_irq,
	.ack = mask_and_ack_amazon_se_irq,
	.mask = mask_and_ack_amazon_se_irq,
	.mask_ack = mask_and_ack_amazon_se_irq,
	.unmask = enable_amazon_se_irq,
	.end = end_amazon_se_irq,
};

#undef IVEC_BUG_FIX
#ifdef IVEC_BUG_FIX
/**
 * Find least significant bit.
 * This function searches for the least significant bit in 32bit value.
 *
 * \param  x - The value to be checked.
 * \return Position of the least significant bit (0..31)
 */
static inline int ls1bit32(unsigned int x)
{
  int b = 31, s;

  s = 16; if (x << 16 == 0) s = 0; b -= s; x <<= s;
  s =  8; if (x <<  8 == 0) s = 0; b -= s; x <<= s;
  s =  4; if (x <<  4 == 0) s = 0; b -= s; x <<= s;
  s =  2; if (x <<  2 == 0) s = 0; b -= s; x <<= s;
  s =  1; if (x <<  1 == 0) s = 0; b -= s;

  return b;
}
#endif /* IVEC_BUG_FIX */
/* Cascaded interrupts from IM0 */
void amazon_se_hw0_irqdispatch(void)
{
  u32 irq;

#ifdef IVEC_BUG_FIX
  irq = (*AMAZON_SE_ICU_IM0_IOSR); 
  /* if int_status == 0, then the interrupt has already been cleared */
  if (irq == 0) {
    return;
  }
  irq = ls1bit32(irq);
#else
  irq = (*AMAZON_SE_ICU_IM_VEC) & AMAZON_SE_ICU_IM0_VEC_MASK;
  if (irq ==0) return;
  irq--;
#endif /*IVEC_BUG_FIX*/
  if (irq == 22){
    /*clear EBU interrupt */
    *(AMAZON_SE_EBU_PCC_ISTAT) |= 0x10;
  }
  AMAZON_SE_INT_DMSG("amazon_se_hw0_irqdispatch: irq=%d\n", irq);

  do_IRQ((int)irq);

  return;
}

/* Cascaded interrupts from IM1 */
void amazon_se_hw1_irqdispatch(void)
{
  u32 irq;

#ifdef IVEC_BUG_FIX
  irq = (*AMAZON_SE_ICU_IM1_IOSR); 
  /* if int_status == 0, then the interrupt has already been cleared */
  if (irq == 0) {
    return;
  }
  irq = ls1bit32(irq)+32;
#else
  irq = ((*AMAZON_SE_ICU_IM_VEC) & AMAZON_SE_ICU_IM1_VEC_MASK) >>6;
  if (irq ==0) return;
  irq+=31;
#endif /*IVEC_BUG_FIX*/
  AMAZON_SE_INT_DMSG("amazon_se_hw1_irqdispatch: irq=%d\n", irq);

  do_IRQ((int)irq);

  return;
}
/* Cascaded interrupts from IM2 */
void amazon_se_hw2_irqdispatch(void)
{
  u32 irq;

#ifdef IVEC_BUG_FIX
  irq = (*AMAZON_SE_ICU_IM2_IOSR); 
  /* if int_status == 0, then the interrupt has already been cleared */
  if (irq == 0) {
    return;
  }
  irq = ls1bit32(irq)+64;
#else
  irq = ((*AMAZON_SE_ICU_IM_VEC) & AMAZON_SE_ICU_IM2_VEC_MASK) >>12;
  if (irq ==0) return;
  irq+=63;
#endif /*IVEC_BUG_FIX*/
  AMAZON_SE_INT_DMSG("amazon_se_hw2_irqdispatch: irq=%d\n", irq);

  do_IRQ((int)irq);

  return;
}
/* Cascaded interrupts from IM3 */
void amazon_se_hw3_irqdispatch(void)
{
  u32 irq;

#ifdef IVEC_BUG_FIX
  irq = (*AMAZON_SE_ICU_IM3_IOSR); 
  /* if int_status == 0, then the interrupt has already been cleared */
  if (irq == 0) {
    return;
  }
  irq = ls1bit32(irq)+96;
#else
  irq = ((*AMAZON_SE_ICU_IM_VEC) & AMAZON_SE_ICU_IM3_VEC_MASK) >>18;
  if (irq ==0) return;
  irq+=95;
#endif /*IVEC_BUG_FIX*/
  AMAZON_SE_INT_DMSG("amazon_se_hw3_irqdispatch: irq=%d\n", irq);

  do_IRQ((int)irq);

  return;
}

int amazon_se_be_handler(struct pt_regs *regs, int is_fixup)
{
  /*TODO: bus error */
  printk("TODO: BUS error\n");
	return MIPS_BE_FATAL;
}

asmlinkage void amazon_se_timer_interrupt(void)
{
	irq_enter();                                                    \
        __DO_IRQ_SMTC_HOOK(MIPS_CPU_TIMER_IRQ); 
        timer_interrupt(MIPS_CPU_TIMER_IRQ, NULL);
	irq_exit();
}

asmlinkage void plat_irq_dispatch(void)
{
        unsigned int pending = read_c0_status() & read_c0_cause();

        if (pending & CAUSEF_IP7)              /* int2 hardware line (timer) */
	amazon_se_timer_interrupt();
        else if (pending & CAUSEF_IP2)         /* int0 hardware line */
	amazon_se_hw0_irqdispatch();
        else if (pending & CAUSEF_IP3)         /* int3 hardware line */
        amazon_se_hw1_irqdispatch();
        else if (pending & CAUSEF_IP4)         /* int4 hardware line */
        amazon_se_hw2_irqdispatch();
        else if (pending & CAUSEF_IP5)         /* compare int */
        amazon_se_hw3_irqdispatch();
        else
                spurious_interrupt();
}

/* Function for careful CP0 interrupt mask access */

//void __init init_IRQ(void)
void __init arch_init_irq(void)
{
  int i;

  AMAZON_SE_INT_DMSG("init_IRQ\n");

  board_be_handler = &amazon_se_be_handler;

//  init_generic_irq();

  /* mask all interrupt sources */
  *AMAZON_SE_ICU_IM0_IER = 0;
  *AMAZON_SE_ICU_IM1_IER = 0;
  *AMAZON_SE_ICU_IM2_IER = 0;
  *AMAZON_SE_ICU_IM3_IER = 0;

  for (i = 0; i <= INT_NUM_IM3_IRL31; i++) {
    irq_desc[i].status	= IRQ_DISABLED;
    irq_desc[i].action	= 0;
    irq_desc[i].depth	= 1;
//    irq_desc[i].handler	= &amazon_se_irq_type;
    set_irq_chip_and_handler(i, &amazon_se_irq_type, handle_level_irq);
  }

  set_c0_status(IE_IRQ0 | IE_IRQ1 | IE_IRQ2 | IE_IRQ3 | IE_IRQ4 | IE_IRQ5);

#ifdef CONFIG_KGDB
  set_debug_traps();
  breakpoint();
#endif
}
