/******************************************************************************
**
** FILE NAME    : dma-core.c
** PROJECT      : Amazon SE
** MODULES      : Central DMA
**
** DATE         : 14 SEP 2006
** AUTHOR       : Reddy Mallikarjuna
** DESCRIPTION  : Central DMA Driver 
** COPYRIGHT    :       Copyright (c) 2006
**                      Infineon Technologies AG
**                      Am Campeon 1-12, 85579 Neubiberg, Germany
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
** HISTORY
** $Date        $Author                 $Comment
** 14 SEP 2006  Reddy Mallikarjuna      Initiate Version 
					Modification based on Amazon-SE chip
** 16 May 2007  Reddy Mallikarjuna	Declare g_amazon_se_dma_in_process and
**					g_amazon_se_dma_int_status as volatile 
**					object and fix problem caused by 
**					compiler optimization
*******************************************************************************/

#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#if defined(MODVERSIONS) && !defined(__GENKSYMS__)
//#include <linux/modversions.h>
#endif

#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB		/* need this one 'cause we export symbols */
#endif

#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/selection.h>
#include <linux/kmod.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <linux/errno.h>
#include <asm/io.h>
//#include <linux/wrapper.h>
#include <asm-mips/semaphore.h>

#include <asm/amazon_se/amazon_se.h>
#include <asm/amazon_se/irq.h>
#include <asm/amazon_se/amazon_se_dma.h>

#include "dma-core.h"

#if defined(CONFIG_IFX_PPA_A4)
  #define ENABLE_IFX_PPA_SUPPORT            1
#endif

/*****************definitions for the macros used in dma-core.c***************/

#define IFX_SUCCESS 1
#define IFX_ERROR   0

#define AMAZON_SE_DMA_EMSG(fmt, args...) printk( KERN_ERR  "%s: " fmt,__FUNCTION__, ## args)
#define AMAZON_SE_DMA_WRREG_PROT(chan,addr,value) \
	      *AMAZON_SE_DMA_CS=chan;  \
	      *addr=value;

#define AMAZON_SE_DMA_RDREG_PROT(chan,addr,value) \
	      *AMAZON_SE_DMA_CS=chan;  \
	      value=*addr;

#define AMAZON_SE_DMA_ORREG_PROT(chan,addr,value) \
	      *AMAZON_SE_DMA_CS=chan;  \
	      *addr|=value;

#define AMAZON_SE_DMA_ANDREG_PROT(chan,addr,value) \
	      *AMAZON_SE_DMA_CS=chan;  \
	      *addr &=value;

#define MAX_DMA_DEVICE_NUM  3	/*max ports connecting to dma */
#define MAX_DMA_CHANNEL_NUM 10	/*max dma channels */
#define DMA_INT_BUDGET      100	/*budget for interrupt handling */
#define DMA_POLL_COUNTER    4	/*fix me, set the correct counter value here! */

//#define DMA_DESCRIPTOR_OFFSET 50  
#define DMA_DESCRIPTOR_WIDTH	8	// 64-bit

#define DMA_CCTRL_P2PCPY_BIT  24

//#define DMA_DESCRIPTOR_OFFSET (PAGE_SIZE/(DMA_DESCRIPTOR_WIDTH*MAX_DMA_CHANNEL_NUM))
#define DMA_DESCRIPTOR_OFFSET  IFX_DMA_DESCRIPTOR_OFFSET

//Max descripters per channel 255 (8-bit)

#define NO_DESCRIPTORS_PER_CAHHNEL	DMA_DESCRIPTOR_OFFSET

//#define AMAZON_SE_DMA_CH_INT(n) AMAZON_SE_DMA_CH##n##_INT   
/*****************************************************************************/

extern void mask_and_ack_amazon_se_irq (unsigned int irq_nr);
extern void enable_amazon_se_irq (unsigned int irq_nr);
extern void disable_amazon_se_irq (unsigned int irq_nr);

/******************Global variables ******************************************/

struct proc_dir_entry *g_amazon_se_dma_dir;
u64 *g_desc_list;
//u64* g_desc_list = NULL;
_dma_device_info dma_devs[MAX_DMA_DEVICE_NUM];
_dma_channel_info dma_chan[MAX_DMA_CHANNEL_NUM];
char global_device_name[MAX_DMA_DEVICE_NUM][15] =
	{ {"PPE"}, {"SDIO"}, {"MCTRL"} };
//device name, dir, priority, irq, rel_chanel_no
_dma_chan_map default_dma_map[MAX_DMA_CHANNEL_NUM] =
	{ {"PPE", AMAZON_SE_DMA_RX, 0, AMAZON_SE_DMA_CH0_INT, 0},
{"PPE", AMAZON_SE_DMA_TX, 0, AMAZON_SE_DMA_CH1_INT, 0},
{"PPE", AMAZON_SE_DMA_RX, 1, AMAZON_SE_DMA_CH2_INT, 1},
{"PPE", AMAZON_SE_DMA_TX, 1, AMAZON_SE_DMA_CH3_INT, 1},
{"PPE", AMAZON_SE_DMA_RX, 2, AMAZON_SE_DMA_CH4_INT, 2},
{"PPE", AMAZON_SE_DMA_RX, 3, AMAZON_SE_DMA_CH5_INT, 3},
{"SDIO", AMAZON_SE_DMA_RX, 0, AMAZON_SE_DMA_CH6_INT, 0},
{"SDIO", AMAZON_SE_DMA_TX, 0, AMAZON_SE_DMA_CH7_INT, 0},
{"MCTRL", AMAZON_SE_DMA_RX, 0, AMAZON_SE_DMA_CH8_INT, 0},
{"MCTRL", AMAZON_SE_DMA_TX, 0, AMAZON_SE_DMA_CH9_INT, 0}
};

_dma_chan_map *chan_map = default_dma_map;
volatile u32 g_amazon_se_dma_int_status = 0;
volatile int g_amazon_se_dma_in_process = 0;	/*0=not in process,1=in process */
struct semaphore *amazon_se_dma_sem;

/******************************************************************************/

/***********function definitions***********************************************/
void do_dma_tasklet (unsigned long);
DECLARE_TASKLET (dma_tasklet, do_dma_tasklet, 0);
irqreturn_t dma_interrupt(int irq, void *dev_id);

/******************************************************************************/

int select_chan (int chan_num)
{
	*AMAZON_SE_DMA_CS = chan_num;

	return IFX_SUCCESS;
}

int enable_chan (int chan_num)
{
	*AMAZON_SE_DMA_CS = chan_num;
	*AMAZON_SE_DMA_CCTRL |= 1;
	return IFX_SUCCESS;
}

int disable_chan (int chan_num)
{
	*AMAZON_SE_DMA_CS = chan_num;
	*AMAZON_SE_DMA_CCTRL &= ~1;
	return IFX_SUCCESS;
}

u8* common_buffer_alloc (int len, int *byte_offset, void **opt)
{
	u8 *buffer = (u8 *) kmalloc (len * sizeof (u8), GFP_ATOMIC);
	*byte_offset = 0;
	return buffer;

}

int common_buffer_free (u8 * dataptr, void *opt)
{
	if (dataptr)
		kfree (dataptr);
	return 0;
}

int enable_ch_irq (_dma_channel_info * pCh)
{
	int chan_no = (int) (pCh - dma_chan);
	unsigned long flag;

	local_irq_save(flag);
	*AMAZON_SE_DMA_CS = chan_no;
	*AMAZON_SE_DMA_CIE = 0x4a;
	*AMAZON_SE_DMA_IRNEN |= 1 << chan_no;
	local_irq_restore(flag);

	return IFX_SUCCESS;
}

int  disable_ch_irq (_dma_channel_info * pCh)
{
	unsigned long flag;
	int chan_num = (int) (pCh - dma_chan);

	local_irq_save(flag);
	g_amazon_se_dma_int_status &= ~(1 << chan_num);
	*AMAZON_SE_DMA_CS = chan_num;
	*AMAZON_SE_DMA_CIE = 0;
	*AMAZON_SE_DMA_IRNEN &= ~(1 << chan_num);

	local_irq_restore(flag);

	return IFX_SUCCESS;
}

int open_chan (_dma_channel_info * pCh)
{
	unsigned long flag;
	int chan_num = (int) (pCh - dma_chan);

	local_irq_save(flag);
	*AMAZON_SE_DMA_CS = chan_num;
	*AMAZON_SE_DMA_CCTRL |= 1;	//channel turn On
	if (pCh->dir == AMAZON_SE_DMA_RX)
		enable_ch_irq (pCh);
	local_irq_restore(flag);
	return IFX_SUCCESS;
}

int close_chan (_dma_channel_info * pCh)
{
	unsigned long flag;
	int chan_num = (int) (pCh - dma_chan);

	local_irq_save(flag);
	*AMAZON_SE_DMA_CS = chan_num;
	*AMAZON_SE_DMA_CCTRL &= ~1;	//channel turn off
	disable_ch_irq (pCh);
	local_irq_restore(flag);
	return IFX_SUCCESS;
}

int reset_chan (_dma_channel_info * pCh)
{
	int chan_num = (int) (pCh - dma_chan);
	*AMAZON_SE_DMA_CS = chan_num;
	*AMAZON_SE_DMA_CCTRL |= 2;	//Reset the channel
	return IFX_SUCCESS;
}
void rx_chan_intr_handler (int chan_no)
{
	_dma_device_info *pDev = (_dma_device_info *) dma_chan[chan_no].dma_dev;
	_dma_channel_info *pCh = &dma_chan[chan_no];
	struct rx_desc *rx_desc_p;
	int tmp;
	unsigned long flag;

	/*handle command complete interrupt */
	rx_desc_p = (struct rx_desc *) pCh->desc_base + pCh->curr_desc;
	if (rx_desc_p->status.field.OWN == CPU_OWN
	    && rx_desc_p->status.field.C) {
		/*Every thing is correct, then we inform the upper layer */
		pDev->current_rx_chan = pCh->rel_chan_no;
		if (pDev->intr_handler)
			pDev->intr_handler (pDev, RCV_INT);
		pCh->weight--;
	}
	else {
		local_irq_save(flag);
		tmp = *AMAZON_SE_DMA_CS;
		*AMAZON_SE_DMA_CS = chan_no;
		*AMAZON_SE_DMA_CIS |= 0x7e;
		*AMAZON_SE_DMA_CS = tmp;
		if (rx_desc_p->status.field.OWN != CPU_OWN)
			g_amazon_se_dma_int_status &= ~(1 << chan_no);

		/*
		* Enable this channel interrupt again after processing all packets available
		* Placing this line at the last line will avoid interrupt coming again
		* in heavy load to great extent
		*/
		*AMAZON_SE_DMA_IRNEN |= (1 << chan_no);
		    local_irq_restore(flag);
	}
}

inline void
tx_chan_intr_handler (int chan_no)
{
	_dma_device_info *pDev = (_dma_device_info *) dma_chan[chan_no].dma_dev;
	_dma_channel_info *pCh = &dma_chan[chan_no];
	unsigned long flag;
	int tmp;

	local_irq_save(flag);
	tmp = *AMAZON_SE_DMA_CS;
	*AMAZON_SE_DMA_CS = chan_no;
	*AMAZON_SE_DMA_CIS |= 0x7e;
	*AMAZON_SE_DMA_CS = tmp;
	g_amazon_se_dma_int_status &= ~(1 << chan_no);
	local_irq_restore(flag);

	pDev->current_tx_chan = pCh->rel_chan_no;
	if (pDev->intr_handler)
		pDev->intr_handler (pDev, TRANSMIT_CPT_INT);
}

void  do_dma_tasklet (unsigned long unused)
{
	int i;
	int chan_no = 0;
	int budget = DMA_INT_BUDGET;
	int weight = 0;
	unsigned long flag;

	while (g_amazon_se_dma_int_status) {
		if (budget-- < 0) {
			tasklet_schedule (&dma_tasklet);
			return;
		}
		chan_no = -1;
		weight = 0;
		/*WFQ algorithm to select the channel */
		for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {
			if ((g_amazon_se_dma_int_status & (1 << i)) && dma_chan[i].weight > 0) {
				if (dma_chan[i].weight > weight) {
					chan_no = i;
					weight = dma_chan[chan_no].weight;
				}
			}
		}

		if (chan_no >= 0) {
			if (chan_map[chan_no].dir == AMAZON_SE_DMA_RX)
				rx_chan_intr_handler (chan_no);
			else
				tx_chan_intr_handler (chan_no);
		}
		else {		/*reset all the channels */

			for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {
				dma_chan[i].weight =
					dma_chan[i].default_weight;
			}
		}
	}			//while(g_amazon_se_dma_int_status)

	local_irq_save(flag);
		g_amazon_se_dma_in_process = 0;
	if (g_amazon_se_dma_int_status) {
		g_amazon_se_dma_in_process = 1;
		tasklet_schedule (&dma_tasklet);
	} 
	//else
	//	g_amazon_se_dma_in_process = 0;
	   local_irq_restore(flag);
}


irqreturn_t dma_interrupt (int irq, void *dev_id)
{
	_dma_channel_info *pCh;
	int chan_num = 0;
	int tmp;// flag;

	pCh = (_dma_channel_info *) dev_id;
	chan_num = (int) (pCh - dma_chan);
	if (chan_num < 0 || chan_num > (MAX_DMA_CHANNEL_NUM - 1))
		printk (KERN_INFO "dma_interrupt irq=%d chan_num=%d\n", irq, chan_num);


/*
     * Disable interrupt on this channel, later tasklet will enable it after
	 * processing all available packets on this channel.
*/
	*AMAZON_SE_DMA_IRNEN &= ~(1 << chan_num);

  /* Record this channel interrupt for tasklet */
      g_amazon_se_dma_int_status |= (1 << chan_num);


#if defined(ENABLE_NAPI) && ENABLE_NAPI
    if ( pCh->dir == IFX_DMA_RX )
	{
		    _dma_device_info* pDev=(_dma_device_info*)pCh->dma_dev;

			if ( pDev->activate_poll )
			{
				//  handled by polling rather than tasklet
				g_amazon_se_dma_int_status &= ~(1 << chan_num);
				g_dma_poll_int_status |= 1 << chan_num;
				pDev->activate_poll(pDev);
					return
			IRQ_RETVAL(1);
			}
	}
#endif

	// if(!g_amazon_se_dma_in_process)/*if not in process, then invoke the tasklet*/
	if (!g_amazon_se_dma_in_process) {
		g_amazon_se_dma_in_process = 1;
		tasklet_schedule (&dma_tasklet);
	}
    return IRQ_RETVAL(1);
}

_dma_device_info *
dma_device_reserve (char *dev_name)
{
	int i;

	for (i = 0; i < MAX_DMA_DEVICE_NUM; i++) {	/*may put some hash function here in the future:) */
		if (strcmp (dev_name, dma_devs[i].device_name) == 0) {
			if (dma_devs[i].reserved)
				return NULL;
			dma_devs[i].reserved = 1;
#if defined(ENABLE_NAPI) && ENABLE_NAPI
			dma_devs[i].activate_poll = NULL;
			dma_devs[i].inactivate_poll = NULL;
#endif
	return &dma_devs[i];
									       
		}
	}
	return NULL;
//	return &dma_devs[i];
}

int dma_device_release (_dma_device_info * dev)
{
	dev->reserved = 0;

	return IFX_SUCCESS;
}

int
dma_device_register (_dma_device_info * dev)
{
	int result = IFX_SUCCESS;
	int i, j;
	int chan_no = 0;
	u8 *buffer;
	int byte_offset;
	unsigned long flag;
	_dma_device_info *pDev;
	_dma_channel_info *pCh;
	struct rx_desc *rx_desc_p;
	struct tx_desc *tx_desc_p;

	for (i = 0; i < dev->max_tx_chan_num; i++) {
		pCh = dev->tx_chan[i];
		if (pCh->control == AMAZON_SE_DMA_CH_ON) {
			chan_no = (int) (pCh - dma_chan);
			for (j = 0; j < pCh->desc_len; j++) {
				tx_desc_p =(struct tx_desc *) pCh->desc_base + j;
				memset (tx_desc_p, 0,sizeof (struct tx_desc));

			}
			 local_irq_save(flag);
			*AMAZON_SE_DMA_CS = chan_no;
			/*check if the descriptor base is changed*/
			if ( *AMAZON_SE_DMA_CDBA != (u32)CPHYSADDR(pCh->desc_base) )
				*AMAZON_SE_DMA_CDBA = (u32)CPHYSADDR(pCh->desc_base);
				 /*check if the descriptor length is changed*/
			if (*AMAZON_SE_DMA_CDLEN != pCh->desc_len)
				*AMAZON_SE_DMA_CDLEN = pCh->desc_len;

			*AMAZON_SE_DMA_CCTRL &= ~1;
			*AMAZON_SE_DMA_CCTRL |= 2;
			while (*AMAZON_SE_DMA_CCTRL & 2);
			//disable_amazon_se_irq(pCh->irq);

			//*AMAZON_SE_DMA_CIE=0x0a;
			*AMAZON_SE_DMA_IRNEN |= 1 << chan_no;
			*AMAZON_SE_DMA_CCTRL = 0x30100;	/*reset and enable channel,enable channel later */
#if defined(ENABLE_IFX_PPA_SUPPORT) && ENABLE_IFX_PPA_SUPPORT
			if ( i == 1 )
				*AMAZON_SE_DMA_CCTRL = 0x01030100;
#endif
			local_irq_restore(flag);

		}
	}

	for (i = 0; i < dev->max_rx_chan_num; i++) {
		pCh = dev->rx_chan[i];
		if (pCh->control == AMAZON_SE_DMA_CH_ON) {
			chan_no = (int) (pCh - dma_chan);

			for (j = 0; j < pCh->desc_len; j++) {
				rx_desc_p =	(struct rx_desc *) pCh->desc_base + j;
				pDev = (_dma_device_info *) (pCh->dma_dev);
				buffer = pDev->buffer_alloc (pCh->packet_size,&byte_offset, (void *) &(pCh->opt[j]));
				if (!buffer) {
					break;
				}

#ifndef CONFIG_MIPS_UNCACHED
				dma_cache_wback_inv ((unsigned long) buffer, pCh->packet_size);	/* dma_device_register */
#endif

				rx_desc_p->Data_Pointer = (u32) CPHYSADDR ((u32) buffer);
				rx_desc_p->status.word = 0;
				rx_desc_p->status.field.byte_offset = byte_offset;
				rx_desc_p->status.field.OWN = DMA_OWN;
				rx_desc_p->status.field.data_length =pCh->packet_size;
			}

			 local_irq_save(flag);
			*AMAZON_SE_DMA_CS = chan_no;

  /*check if the descriptor base is changed*/
			if ( *AMAZON_SE_DMA_CDBA != (u32)CPHYSADDR(pCh->desc_base) )
				*AMAZON_SE_DMA_CDBA = (u32)CPHYSADDR(pCh->desc_base);
				   /*check if the descriptor length is changed*/
			if (*AMAZON_SE_DMA_CDLEN != pCh->desc_len)
				*AMAZON_SE_DMA_CDLEN = pCh->desc_len;

			*AMAZON_SE_DMA_CCTRL &= ~1;
			*AMAZON_SE_DMA_CCTRL |= 2;
			while (*AMAZON_SE_DMA_CCTRL & 2);
			*AMAZON_SE_DMA_CIE = 0x0A;	/*fix me, should enable all the interrupts here? */
			*AMAZON_SE_DMA_IRNEN |= 1 << chan_no;
			*AMAZON_SE_DMA_CCTRL = 0x30000;

  			local_irq_restore(flag);

		}
	}
//Debug
//        *AMAZON_SE_DMA_PS=0;
//      printk(" DMA_PCTRL: %08X", (*AMAZON_SE_DMA_PCTRL));

	return result;
}

int dma_device_unregister (_dma_device_info * dev)
{
	int result = IFX_SUCCESS;
	int i, j;
	int chan_num;
	_dma_channel_info *pCh;
	struct rx_desc *rx_desc_p;
	struct tx_desc *tx_desc_p;
	unsigned long flag;

	for (i = 0; i < dev->max_tx_chan_num; i++) {
		pCh = dev->tx_chan[i];
		if (pCh->control == AMAZON_SE_DMA_CH_ON) {
			chan_num = (int) (dev->tx_chan[i] - dma_chan);
			//down(amazon_se_dma_sem);
			local_irq_save(flag);
			*AMAZON_SE_DMA_CS = chan_num;
			pCh->curr_desc = 0;
			pCh->prev_desc = 0;
			pCh->control = AMAZON_SE_DMA_CH_OFF;

			*AMAZON_SE_DMA_CIE = 0;	/*fix me, should disable all the interrupts here? */
			*AMAZON_SE_DMA_IRNEN &= ~(1 << chan_num);	/*disable interrupts */
			*AMAZON_SE_DMA_CCTRL &= ~1;

			while (*AMAZON_SE_DMA_CCTRL & 1);

			local_irq_restore(flag);

			//up(amazon_se_dma_sem);
			for (j = 0; j < pCh->desc_len; j++) {
				tx_desc_p =	(struct tx_desc *) pCh->desc_base + j;
				if ((tx_desc_p->status.field.OWN == CPU_OWN && tx_desc_p->status.field.C) || \
					(tx_desc_p->status.field.OWN == DMA_OWN && tx_desc_p->status.field.data_length > 0)) 
				{
					dev->buffer_free ((u8 *) __va (tx_desc_p->Data_Pointer),(void *) pCh->opt[j]);
				}
				tx_desc_p->status.field.OWN = CPU_OWN;
				memset (tx_desc_p, 0, sizeof (struct tx_desc));
			}
			/*fix me: should free buffer that is not transferred by dma */
		}
	}

	for (i = 0; i < dev->max_rx_chan_num; i++) {
		pCh = dev->rx_chan[i];
		chan_num = (int) (dev->rx_chan[i] - dma_chan);
		local_irq_save(flag);
//		if (chan_num == 5)
//			printk ("52");
		g_amazon_se_dma_int_status &= ~(1 << chan_num);
		pCh->curr_desc = 0;
		pCh->prev_desc = 0;
		pCh->control = AMAZON_SE_DMA_CH_OFF;

		*AMAZON_SE_DMA_CS = chan_num;
		if (*AMAZON_SE_DMA_CS != chan_num)
			printk(__FILE__ ":%d:%s: *DMA_CS (%d) != chan_num (%d)", __LINE__, __FUNCTION__, *AMAZON_SE_DMA_CS,chan_num);


		*AMAZON_SE_DMA_CIE = 0;	/*fix me, should disable all the interrupts here? */
		*AMAZON_SE_DMA_IRNEN &= ~(1 << chan_num);	/*disable interrupts */
		*AMAZON_SE_DMA_CCTRL &= ~1;

		while (*AMAZON_SE_DMA_CCTRL & 1) ;

		local_irq_restore(flag);

		for (j = 0; j < pCh->desc_len; j++) {
			rx_desc_p = (struct rx_desc *) pCh->desc_base + j;
			if ((rx_desc_p->status.field.OWN == CPU_OWN  && rx_desc_p->status.field.C) || \
				(rx_desc_p->status.field.OWN == DMA_OWN && rx_desc_p->status.field.data_length > 0)) 
			{
				dev->buffer_free ((u8 *) __va (rx_desc_p->Data_Pointer), (void *) pCh->opt[j]);
			}
		}
	}

	return result;
}

/**
	* This function applies to the DMA synchronization case. That is, when DMA client
	* driver actively requires data, instead of interrupt/event driven. The basic flow
	* as follows,
    * 1) DMA client driver prepares DMA descriptor
	* 2) Enable DMA client driver control function
	* 3) Wait for DMA reception interrupt is coming
	* In this way, there is no prealloc memory, no memory copy. All buffers are prepared
	* by DMA client driver
*/
int
dma_device_desc_setup(_dma_device_info *dma_dev, char *buf, size_t len)
{
    int byte_offset = 0;
	struct rx_desc *rx_desc_p;

	_dma_channel_info *pCh = dma_dev->rx_chan[dma_dev->current_rx_chan];

	rx_desc_p = (struct rx_desc *) pCh->desc_base + pCh->curr_desc;

#ifndef CONFIG_MIPS_UNCACHED
	dma_cache_inv ((unsigned long) buf, len);
#endif
	pCh->opt[pCh->curr_desc] = NULL;
	rx_desc_p->Data_Pointer = (u32)CPHYSADDR((u32)buf);
	rx_desc_p->status.word = (DMA_OWN << 31) |((byte_offset) << 23)| len;
	wmb();
/* Increase descriptor index and process wrap around */
	pCh->curr_desc++;
	if (pCh->curr_desc == pCh->desc_len) {
		pCh->curr_desc = 0;
	}
	return 0;
}
EXPORT_SYMBOL(dma_device_desc_setup);

/**
	* This function applies to the DMA synchronization case.
	* All buffers are provided by DMA client. DMA client driver
	* has no method to modify descriptor state. The only method
	* is to change interrupt status so that DMA tasklet don't
	* need to run again and again.
*/
int dma_device_clear_int(_dma_device_info *dma_dev, int dir)
{
    int chan_no;
	_dma_channel_info *pCh;
	unsigned long flag;

	if (dir == 0) {
		pCh = dma_dev->rx_chan[dma_dev->current_rx_chan];
	}
	else {
		pCh = dma_dev->tx_chan[dma_dev->current_tx_chan];
	}
	chan_no = (int)(pCh - dma_chan);
	local_irq_save(flag);
	g_amazon_se_dma_int_status  &= ~(1 << chan_no);
	local_irq_restore(flag);
	return 0;
}
EXPORT_SYMBOL(dma_device_clear_int);


int
dma_device_read (struct dma_device_info *dma_dev, u8 ** dataptr, void **opt)
{
	u8 *buf;
	int len;
	int byte_offset = 0;
	void *p = NULL;

	_dma_channel_info *pCh = dma_dev->rx_chan[dma_dev->current_rx_chan];

	struct rx_desc *rx_desc_p;

	/*get the rx data first */
	rx_desc_p = (struct rx_desc *) pCh->desc_base + pCh->curr_desc;
	if (!(rx_desc_p->status.field.OWN == CPU_OWN && rx_desc_p->status.field.C)) {
		return 0;
	}
	buf = (u8 *) __va (rx_desc_p->Data_Pointer);
	*(u32 *) dataptr = (u32) buf;
	len = rx_desc_p->status.field.data_length;
	if (opt)
		*(int *) opt = (int) pCh->opt[pCh->curr_desc];

	/*replace with a new allocated buffer */
	buf = dma_dev->buffer_alloc (pCh->packet_size, &byte_offset, &p);
	if (buf) {
		pCh->opt[pCh->curr_desc] = p;
#ifndef CONFIG_MIPS_UNCACHED
		//       dma_cache_inv ((unsigned long) buf, pCh->packet_size);
//         dma_cache_inv ((unsigned long) buf, 64); /* dma_device_read: new allocated buffer */
		dma_cache_wback_inv ((unsigned long) buf, 64);	/* dma_device_read: new allocated buffer */
#endif

		rx_desc_p->Data_Pointer = (u32) CPHYSADDR ((u32) buf);
		wmb();
//      byte_offset = byte_offset & 0x3;  
		rx_desc_p->status.word = (DMA_OWN << DESCRIPTER_OWN_BIT_OFFSET) \
								| ((byte_offset) << DESCRIPTER_RX_BYTE_OFFSET) \
								| pCh->packet_size;
      wmb();
	}
	else {
		*(u32 *) dataptr = 0;
		if (opt)
			*(int *) opt = 0;
		len = 0;
		rx_desc_p->status.word = (DMA_OWN << DESCRIPTER_OWN_BIT_OFFSET) \
								| ((byte_offset) << DESCRIPTER_RX_BYTE_OFFSET) \
								| pCh->packet_size;

	}

	/*increase the curr_desc pointer */
	pCh->curr_desc++;
	if (pCh->curr_desc == pCh->desc_len)
		pCh->curr_desc = 0;
	/*return the length of the received packet */
	return len;
}

int
dma_device_write (struct dma_device_info *dma_dev, u8 * dataptr, int len, void *opt)
{
	unsigned long flag;
	u32 tmp, byte_offset,tcs;
	_dma_channel_info *pCh;
	int chan_no;
	struct tx_desc *tx_desc_p;

	pCh = dma_dev->tx_chan[dma_dev->current_tx_chan];
	chan_no = (int) (pCh - (_dma_channel_info *) dma_chan);

	tx_desc_p = (struct tx_desc *) pCh->desc_base + pCh->prev_desc;
	while (tx_desc_p->status.field.OWN == CPU_OWN && tx_desc_p->status.field.C) {

		dma_dev->buffer_free ((u8 *) __va (tx_desc_p->Data_Pointer),pCh->opt[pCh->prev_desc]);
		memset (tx_desc_p, 0, sizeof (struct tx_desc));
		pCh->prev_desc = (pCh->prev_desc + 1) % (pCh->desc_len);
		tx_desc_p = (struct tx_desc *) pCh->desc_base + pCh->prev_desc;

	}

	tx_desc_p = (struct tx_desc *) pCh->desc_base + pCh->curr_desc;
	/*Check whether this descriptor is available */
	if (tx_desc_p->status.field.OWN == DMA_OWN || tx_desc_p->status.field.C) {
		/*if not , the tell the upper layer device */
		dma_dev->intr_handler (dma_dev, TX_BUF_FULL_INT);

		printk (KERN_INFO "%s : failed to write!\n", __func__);

		return 0;
	}
	pCh->opt[pCh->curr_desc] = opt;
	/*byte offset----to adjust the starting address of the data buffer, should be multiple of the burst length. */
	byte_offset = ((u32) CPHYSADDR ((u32) dataptr)) % ((dma_dev->tx_burst_len) * 4);
#ifndef	CONFIG_MIPS_UNCACHED
//      dma_cache_wback((unsigned long) dataptr,len); /* dma_device_write */
	dma_cache_wback_inv ((unsigned long) dataptr, len);	/* dma_device_write */
//      wmb();
#endif //CONFIG_MIPS_UNCACHED

	byte_offset = byte_offset & 0x1f;

	tx_desc_p->Data_Pointer = (u32) CPHYSADDR ((u32) dataptr) - byte_offset;
    wmb();

	tx_desc_p->status.word = (DMA_OWN << DESCRIPTER_OWN_BIT_OFFSET) \
							| DMA_DESC_SOP_SET \
							| DMA_DESC_EOP_SET \
							| ((byte_offset) << DESCRIPTER_TX_BYTE_OFFSET) \
							| len;
    wmb();

	pCh->curr_desc++;
	if (pCh->curr_desc == pCh->desc_len)
		pCh->curr_desc = 0;

	/*Check whether this descriptor is available */
	tx_desc_p = (struct tx_desc *) pCh->desc_base + pCh->curr_desc;
	if (tx_desc_p->status.field.OWN == DMA_OWN) {
		/*if not , the tell the upper layer device */
		dma_dev->intr_handler (dma_dev, TX_BUF_FULL_INT);

	}
//	AMAZON_SE_DMA_RDREG_PROT (chan_no, AMAZON_SE_DMA_CCTRL, tmp);
	
	local_irq_save(flag);
    tcs = *AMAZON_SE_DMA_CS;
	*AMAZON_SE_DMA_CS = chan_no;
	tmp = *AMAZON_SE_DMA_CCTRL;
	*AMAZON_SE_DMA_CS = tcs;
	local_irq_restore(flag);
					

	if (!(tmp & 1))
		pCh->open (pCh);


	return len;
}

int
desc_list_proc_read (char *buf, char **start, off_t offset,
		     int count, int *eof, void *data)
{
	int i, j;
	int len = 0;
	len += sprintf (buf + len, "descriptor list:\n");
	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {
		sprintf (buf + len, "channel %d\n", i);
		for (j = 0; j < dma_chan[i].desc_len; j++) {
			sprintf (buf + len, "%08x\n%08x\n\n",
				 (u32) ((u64) dma_chan[i].desc_base + j),
				 (u32) dma_chan[i].desc_base + j * 2 + 1);

		}
		sprintf (buf + len, "\n\n\n");
	}
	return len;
}

int
channel_weight_proc_read (char *buf, char **start, off_t offset,
			  int count, int *eof, void *data)
{

	int i = 0;
	int len = 0;

	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {
		len += sprintf (buf + len, "channel %d  %08x\n", i,
				dma_chan[i].weight);

	}
	return len;
}

int
dma_register_proc_read (char *buf, char **start, off_t offset,
			int count, int *eof, void *data)
{
	/*fix me, do I need to implement this? */
	int len = 0;;
	len += sprintf (buf + len, "register of DMA:\n");
	return len;
}

static int
dma_open (struct inode *inode, struct file *file)
{
//	MOD_INC_USE_COUNT;
	return 0;
}

static int
dma_release (struct inode *inode, struct file *file)
{
	/*release the resources */
//	MOD_DEC_USE_COUNT;
	return 0;
}

static int
dma_ioctl (struct inode *inode, struct file *file, unsigned int cmd,
	   unsigned long arg)
{
	int result = 0;
	/*TODO: add some user controled functions here */
	return result;
}

static struct file_operations dma_fops = {
      owner:THIS_MODULE,
      open:dma_open,
      release:dma_release,
      ioctl:dma_ioctl,
};

int
map_dma_chan (_dma_chan_map * map)
{
	int i, j;
	int result;
	for (i = 0; i < MAX_DMA_DEVICE_NUM; i++) {
		strcpy (dma_devs[i].device_name, global_device_name[i]);

	}
	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {

		dma_chan[i].irq = map[i].irq;
#if defined(ENABLE_IFX_PPA_SUPPORT) && ENABLE_IFX_PPA_SUPPORT
		if ( map[i].irq == AMAZON_SE_DMA_CH2_INT || map[i].irq == AMAZON_SE_DMA_CH4_INT )
			continue;
#endif
		result = request_irq (dma_chan[i].irq, dma_interrupt,
				      IRQF_DISABLED /*SA_INTERRUPT*/, "dma-core",
				      (void *) &dma_chan[i]);
		if (result) {
			AMAZON_SE_DMA_EMSG ("error, cannot get dma_irq!\n");
			free_irq (dma_chan[i].irq, (void *) &dma_interrupt);
//              free_irq(dma_chan[i].irq,(void*)&dma_chan[i]); //
			return -EFAULT;
		}

	}

	for (i = 0; i < MAX_DMA_DEVICE_NUM; i++) {
		dma_devs[i].num_tx_chan = 0;	/*set default tx channel number to be one */
		dma_devs[i].num_rx_chan = 0;	/*set default rx channel number to be one */
		dma_devs[i].max_rx_chan_num = 0;
		dma_devs[i].max_tx_chan_num = 0;
		dma_devs[i].buffer_alloc = &common_buffer_alloc;
		dma_devs[i].buffer_free = &common_buffer_free;
		dma_devs[i].intr_handler = NULL;
		dma_devs[i].tx_burst_len = 4;
		dma_devs[i].rx_burst_len = 4;

		if (i == 0) {
			*AMAZON_SE_DMA_PS = 0;
			*AMAZON_SE_DMA_PCTRL |= (0xf << 8) | (1 << 6);	/*enable dma drop */

//        *AMAZON_SE_DMA_PCTRL=0x1F68;/*tx and rx burst length both set to be 4*/
			//  *AMAZON_SE_DMA_PCTRL=0x1F7C;/*tx and rx burst length both set to be 8*/
//       *AMAZON_SE_DMA_PCTRL|=(0x7<<12);

		}
		if ( i == 1 ) {
			*AMAZON_SE_DMA_PS = 1;
			*AMAZON_SE_DMA_PCTRL |= (0xf << 8) ;
		}
                
        *AMAZON_SE_DMA_PCTRL=0x1F68;/*tx and rx burst length both set to be 4*/
				
		for (j = 0; j < MAX_DMA_CHANNEL_NUM; j++) {
			dma_chan[j].byte_offset = 0;
			dma_chan[j].open = &open_chan;
			dma_chan[j].close = &close_chan;
			dma_chan[j].reset = &reset_chan;
			dma_chan[j].enable_irq = &enable_ch_irq;
			dma_chan[j].disable_irq = &disable_ch_irq;
			dma_chan[j].rel_chan_no = map[j].rel_chan_no;
			dma_chan[j].control = AMAZON_SE_DMA_CH_OFF;
			dma_chan[j].default_weight = DEF_DMA_CH_WEIGHT;
			dma_chan[j].weight = dma_chan[j].default_weight;
			dma_chan[j].curr_desc = 0;
			dma_chan[j].prev_desc = 0;

		}

		for (j = 0; j < MAX_DMA_CHANNEL_NUM; j++) {
			if (strcmp (dma_devs[i].device_name, map[j].dev_name)
			    == 0) {

				if (map[j].dir == AMAZON_SE_DMA_RX) {
					dma_chan[j].dir = AMAZON_SE_DMA_RX;
					dma_devs[i].max_rx_chan_num++;
					dma_devs[i].rx_chan[dma_devs[i].
							    max_rx_chan_num -
							    1] = &dma_chan[j];
					dma_devs[i].rx_chan[dma_devs[i].
							    max_rx_chan_num -
							    1]->pri =
						map[j].pri;
					dma_chan[j].dma_dev =
						(void *) &dma_devs[i];

					/*have to program the class value into the register later, fix me */
				}
				else if (map[j].dir == AMAZON_SE_DMA_TX) {	/*TX direction */
					dma_chan[j].dir = AMAZON_SE_DMA_TX;
					dma_devs[i].max_tx_chan_num++;
					dma_devs[i].tx_chan[dma_devs[i].
							    max_tx_chan_num -
							    1] = &dma_chan[j];
					dma_devs[i].tx_chan[dma_devs[i].
							    max_tx_chan_num -
							    1]->pri =
						map[j].pri;
					dma_chan[j].dma_dev =
						(void *) &dma_devs[i];
				}
				else
					printk ("WRONG MAP!\n");

			}
		}

	}

	return IFX_SUCCESS;
}

int
dma_chip_init (void)
{
//	int result = 0;
	int i;

	printk (KERN_INFO "%s\n", __FUNCTION__);

	// *AMAZON_SE_PMU_PWDCR &=~(1<<AMAZON_SE_PMU_DMA_SHIFT);/*enable DMA from PMU*/
	*AMAZON_SE_PMU_PWDCR &= ~(1 << 5);	/*enable DMA from PMU */
	/*reset DMA, necessary? */
	*AMAZON_SE_DMA_CTRL |= 1;	//Resets the complete DMA module
	*AMAZON_SE_DMA_IRNEN = 0;	/*disable all the interrupts first */

	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {
		*AMAZON_SE_DMA_CS = i;
		*AMAZON_SE_DMA_CCTRL = 0x2;	/*fix me, need to reset this channel first? */
		*AMAZON_SE_DMA_CPOLL = 0x80000040;

	}
#if 0
	/*TODO: add the port settings, ENDIAN conversion here */
	for (i = 0; i < MAX_DMA_DEVICE_NUM; i++) {
		/*select the port */
		*AMAZON_SE_DMA_PS = i;
		/*set port parameters */
		*AMAZON_SE_DMA_PCTRL = 0x1028;	/*tx and rx burst length both set to be 4 */

	}
#endif
    /****************************************************/
	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {
		disable_chan (i);	/*disable all the dma channel first */
	}
	return IFX_SUCCESS;
}

int __init
amazon_se_dma_init (void)
{
	int result = 0;
	int i;
	printk (KERN_INFO "%s\n", __FUNCTION__);
	result = register_chrdev (DMA_MAJOR, "dma-core", &dma_fops);
	if (result) {
		AMAZON_SE_DMA_EMSG ("cannot register device dma-core!\n");
		return result;
	}
	amazon_se_dma_sem =
		(struct semaphore *) kmalloc (sizeof (struct semaphore),
					      GFP_KERNEL);
	if (amazon_se_dma_sem == NULL) {
		AMAZON_SE_DMA_EMSG ("Memory  allocation fail!\n");
		return -ENOMEM;
	}

	init_MUTEX (amazon_se_dma_sem);
	dma_chip_init ();
	map_dma_chan (default_dma_map);
	g_desc_list = (u64 *) KSEG1ADDR (__get_free_page (GFP_DMA));
	if (g_desc_list == NULL) {
		AMAZON_SE_DMA_EMSG ("no memory for desriptor\n");
		return -ENOMEM;
	}

//      printk("g_desc_list address: %08X, page size:%08X \n", (u32)g_desc_list, PAGE_SIZE);

	dma_cache_wback_inv ((unsigned long)g_desc_list, PAGE_SIZE);	/* amazon_se_dma_init */
	g_desc_list = (u64 *)KSEG1ADDR (g_desc_list);
	memset (g_desc_list, 0, PAGE_SIZE);
	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++) {

		dma_chan[i].desc_base =
			(u32) g_desc_list +
			i * DMA_DESCRIPTOR_OFFSET * DMA_DESCRIPTOR_WIDTH;
		dma_chan[i].curr_desc = 0;
		dma_chan[i].desc_len = DMA_DESCRIPTOR_OFFSET;
		select_chan (i);
		*AMAZON_SE_DMA_CDBA = (u32) CPHYSADDR (dma_chan[i].desc_base);
		*AMAZON_SE_DMA_CDLEN = dma_chan[i].desc_len;
//      printk("channel %d-> address : %08X \n", i,(u32)dma_chan[i].desc_base);
	}

	g_amazon_se_dma_dir = proc_mkdir ("amazon_se_dma", NULL);

	create_proc_read_entry ("dma_register",
				0,
				g_amazon_se_dma_dir,
				dma_register_proc_read, NULL);

	create_proc_read_entry ("g_desc_list",
				0,
				g_amazon_se_dma_dir,
				desc_list_proc_read, NULL);

	create_proc_read_entry ("channel_weight",
				0,
				g_amazon_se_dma_dir,
				channel_weight_proc_read, NULL);
	return 0;
}

void
dma_cleanup (void)
{
	int i;
	unregister_chrdev (DMA_MAJOR, "dma-core");
	free_page (KSEG0ADDR ((unsigned long) g_desc_list));
	remove_proc_entry ("channel_weight", g_amazon_se_dma_dir);
	remove_proc_entry ("dma_list", g_amazon_se_dma_dir);
	remove_proc_entry ("dma_register", g_amazon_se_dma_dir);
	remove_proc_entry ("amazon_se_dma", NULL);
	/*release the resources */
	for (i = 0; i < MAX_DMA_CHANNEL_NUM; i++)
		free_irq (dma_chan[i].irq, (void *) &dma_interrupt);
}
__initcall(amazon_se_dma_init);

EXPORT_SYMBOL(dma_cleanup);
//EXPORT_SYMBOL(amazon_se_dma_init);

EXPORT_SYMBOL (dma_device_reserve);
EXPORT_SYMBOL (dma_device_release);
EXPORT_SYMBOL (dma_device_register);
EXPORT_SYMBOL (dma_device_unregister);
EXPORT_SYMBOL (dma_device_read);
EXPORT_SYMBOL (dma_device_write);

MODULE_LICENSE ("GPL");
