/* vi: set sw=4 ts=4: */
/*
 * Mini syslogd implementation for busybox
 *
 * Copyright (C) 1999,2000 by Lineo, inc. and Erik Andersen
 * Copyright (C) 1999,2000,2001 by Erik Andersen <andersee@debian.org>
 *
 * Copyright (C) 2000 by Karl M. Hegbloom <karlheg@debian.org>
 *
 * "circular buffer" Copyright (C) 2001 by Gennady Feldman <gfeldman@cachier.com>
 *
 * Maintainer: Gennady Feldman <gena01@cachier.com> as of Mar 12, 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <paths.h>
#include <signal.h>
#include <stdarg.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/sysinfo.h>
#include "sysklogd.h"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
/* SYSLOG_NAMES defined to pull some extra junk from syslog.h */
#define SYSLOG_NAMES
#include <sys/syslog.h>
#include <sys/uio.h>
#include <sys/uio.h>
/* Path for the file where all log messages are written */
#define __LOG_FILE "/var/log/messages"
#define __CONF_FILE "/etc/syslog.conf"

/* Path to the unix socket */
static char lfile[MAXPATHLEN]="";

static char *logFilePath = __LOG_FILE;

#define dprintf(msg,...)
struct syslog_conf conf;

#define ALERT_MAX_INTERVAL 3*60
static time_t last_send_mail=0;

/* interval between marks in seconds */
static int MarkInterval = 10 * 60;

#ifdef SHOW_HOSTNAME
/* localhost's name */
static char LocalHostName[64]="";
#endif

#ifdef BB_FEATURE_REMOTE_LOG
#include <netinet/in.h>
/* udp socket for logging to remote host */
static int remotefd = -1;
/* where do we log? */
static char *RemoteHost=NULL;
/* what port to log to? */
static int RemotePort = 514;
/* To remote log or not to remote log, that is the question. */
static int doRemoteLog = FALSE;
static int local_logging = FALSE;
#endif


#define MAXLINE         1024            /* maximum line length */


/* circular buffer variables/structures */
#ifdef BB_FEATURE_IPC_SYSLOG
#if __GNU_LIBRARY__ < 5
#error Sorry.  Looks like you are using libc5.
#error libc5 shm support isnt good enough.
#error Please disable BB_FEATURE_IPC_SYSLOG
#endif

#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>

/* our shared key */
static const long KEY_ID = 0x414e4547; /*"GENA"*/

// Semaphore operation structures
static struct shbuf_ds {
	int size;               // size of data written
	int head;               // start of message list
	int tail;               // end of message list
	/* can't use char *data */
	char data[1];           // data/messages
} *buf = NULL;                  // shared memory pointer

static struct sembuf SMwup[1] = {{1, -1, IPC_NOWAIT}}; // set SMwup
static struct sembuf SMwdn[3] = {{0, 0}, {1, 0}, {1, +1}}; // set SMwdn

static int      shmid = -1;     // ipc shared memory id
static int      s_semid = -1;   // ipc semaphore id
int     data_size = 16000; // data size
int     shm_size = 16000 + sizeof(*buf); // our buffer size
static int circular_logging = TRUE;

/* Ron */
static void clear_signal(int sig);
static void reload_signal(int sig);
static char last_log[1024]="";

void logMessage (int pri, char *msg);
/*
 * sem_up - up()'s a semaphore.
 */
static inline void sem_up(int semid)
{
	if ( semop(semid, SMwup, 1) == -1 )
		perror_msg_and_die("semop[SMwup]");
}

/*
 * sem_down - down()'s a semaphore
 */
static inline void sem_down(int semid)
{
	if ( semop(semid, SMwdn, 3) == -1 )
		perror_msg_and_die("semop[SMwdn]");
}


void ipcsyslog_cleanup(void){
	dprintf("Exiting Syslogd!\n");
	if (shmid != -1)
		shmdt(buf);

	if (shmid != -1)
		shmctl(shmid, IPC_RMID, NULL);
	if (s_semid != -1)
		semctl(s_semid, 0, IPC_RMID, 0);
}

void ipcsyslog_init(void){
	if (buf == NULL){
	    if ((shmid = shmget(KEY_ID, shm_size, IPC_CREAT | 1023)) == -1)
			perror_msg_and_die("shmget");

	    if ((buf = shmat(shmid, NULL, 0)) == NULL)
			perror_msg_and_die("shmat");


	    buf->size=data_size;
	    buf->head=buf->tail=0;

	    // we'll trust the OS to set initial semval to 0 (let's hope)
	    if ((s_semid = semget(KEY_ID, 2, IPC_CREAT | IPC_EXCL | 1023)) == -1){
		if (errno == EEXIST){
		   if ((s_semid = semget(KEY_ID, 2, 0)) == -1)
		    perror_msg_and_die("semget");
		}else
			perror_msg_and_die("semget");
	    }
	}else{
		dprintf("Buffer already allocated just grab the semaphore?");
	}
}

static void send_mail_signal(int sig)
{
	sem_down(s_semid);

	if(conf.mail_enable==1){
		char cmd[1024]="";
		if(conf.mail_enable_auth==1)
		sprintf(cmd,"/usr/sbin/smtpc -t %s -d %d -m -h %s -r %s -f %s -s \"%s\" -U %s -P %s </var/log/messages "			,conf.TZ
				,atoi(conf.daylight)
				,conf.mail_server
				,conf.mail_receiver
                ,conf.mail_sender
				,conf.mail_subject
				,conf.mail_login
				,conf.mail_password);
		else
		sprintf(cmd,"/usr/sbin/smtpc -t %s -d %d -m -h %s -r %s -f %s -s \"%s\" </var/log/messages "
				,conf.TZ
				,atoi(conf.daylight)
				,conf.mail_server
				,conf.mail_receiver
                ,conf.mail_sender
				,conf.mail_subject);

		if(system(cmd)==0){
			buf->head=0;
			buf->tail=0;
		}
	}

	sem_up(s_semid);
}

/* write message to buffer */
void circ_message(const char *msg){
	int l=strlen(msg); /* count the whole message w/ '\0' included */

	sem_down(s_semid);

	if ( (buf->tail + l) < buf->size ){
		if ( buf->tail < buf->head){
			if ( (buf->tail + l) >= buf->head ){
				int k= buf->tail + l - buf->head;
				char *c=memchr(buf->data+buf->head + k,'\n',buf->size - (buf->head + k));
			   	buf->head=(c != NULL)?( c - buf->data + 1):0;

			}
		}
		strncpy(buf->data + buf->tail,msg,l); /* append our message */
		buf->tail+=l;
	}else{
		char *c;
		int k=buf->tail + l - buf->size;

		c=memchr(buf->data + k ,'\n', buf->size - k);

		if (c != NULL) {
				buf->head=c-buf->data+1;
				strncpy(buf->data + buf->tail, msg, l - k - 1);
				strcpy(buf->data, &msg[l-k-1]);
				buf->tail = k + 1;
		}else{
				buf->head = buf->tail = 0;
		}

	}
	sem_up(s_semid);
}
#endif  /* BB_FEATURE_IPC_SYSLOG */

/* try to open up the specified device */
int device_open(char *device, int mode)
{
	int m, f, fd = -1;

	m = mode | O_NONBLOCK;

	/* Retry up to 5 times */
	for (f = 0; f < 5; f++)
		if ((fd = open(device, m, 0600)) >= 0)
			break;
	if (fd < 0)
		return fd;
	/* Reset original flags. */
	if (m != mode)
		fcntl(fd, F_SETFL, mode);
	return fd;
}
int vdprintf(int d, const char *format, va_list ap)
{
	char buf[BUF_SIZE];
	int len;

	len = vsnprintf(buf, sizeof(buf), format, ap);
	return write(d, buf, len);
}
/* Note: There is also a function called "message()" in init.c */
/* Print a message to the log file. */
static void message (char *fmt, ...) __attribute__ ((format (printf, 1, 2)));
static void message (char *fmt, ...)
{
	int fd;
	struct flock fl;
	va_list arguments;

	fl.l_whence = SEEK_SET;
	fl.l_start  = 0;
	fl.l_len    = 1;
#ifdef BB_FEATURE_IPC_SYSLOG
	if ((circular_logging == TRUE) && (buf != NULL)){
			char b[1024];
			va_start (arguments, fmt);
			vsnprintf (b, sizeof(b)-1, fmt, arguments);
			va_end (arguments);
			circ_message(b);
#ifdef DEBUG
			printf("head=%d tail=%d\n",buf->head,buf->tail);
#endif
/* print_circ_buf */
			if((fd=open(logFilePath,O_WRONLY|O_TRUNC| O_CREAT|O_NONBLOCK))<0)
				return;
			fl.l_type = F_WRLCK;
			fcntl(fd, F_SETLKW, &fl);
			if(buf->tail > buf->head){
				write(fd,buf->data,buf->tail);
				write(fd,"\0",1);
			}else {

				write(fd,buf->data+buf->head,buf->size-buf->head-1);
				write(fd,buf->data,buf->tail);
				write(fd,"\0",1);

				if(conf.mail_log_full==1){//brayg add
					send_mail_signal(0);
					write(fd,EOF,0);
				}
#if 0
				char tmp[buf->size+1];
				char *pt=tmp;
				/* we don't +1 because last bit is '\0'*/
				memcpy(pt,buf->data+buf->head,buf->size-buf->head);
				strncat(pt+buf->size-buf->head-1,buf->data,buf->tail);
				write(fd,tmp,data_size-buf->head+buf->tail);

				if(conf.mail_enable==1 && conf.mail_log_full==1){
					if(send_mail(conf.mail_server
							,conf.mail_sender
							,conf.mail_receiver
							,conf.mail_subject
							,tmp)==TRUE){
						buf->head=0;
						buf->tail=0;
					}
				}
#endif
			}
			fl.l_type = F_UNLCK;
			fcntl(fd, F_SETLKW, &fl);
			close(fd);
	}else
#endif
	if ((fd = device_open (logFilePath,
						   O_WRONLY | O_CREAT | O_NOCTTY | O_APPEND |
						   O_NONBLOCK)) >= 0) {
		fl.l_type = F_WRLCK;
		fcntl (fd, F_SETLKW, &fl);
		va_start (arguments, fmt);
		vdprintf (fd, fmt, arguments);
		va_end (arguments);
		fl.l_type = F_UNLCK;
		fcntl (fd, F_SETLKW, &fl);
		close (fd);
	} else {
		/* Always send console messages to /dev/console so people will see them. */
		if ((fd = device_open (_PATH_CONSOLE,
							   O_WRONLY | O_NOCTTY | O_NONBLOCK)) >= 0) {
			va_start (arguments, fmt);
			vdprintf (fd, fmt, arguments);
			va_end (arguments);
			close (fd);
		} else {
			fprintf (stderr, "Bummer, can't print: ");
			va_start (arguments, fmt);
			vfprintf (stderr, fmt, arguments);
			fflush (stderr);
			va_end (arguments);
		}
	}
}


int check_log(int fac)
{
	int i=0;
	while(conf.log_list[i]>=0 && i < 20){
	   if(conf.log_list[i++]==fac)
	   		return TRUE;
	}
	return FALSE;
}
#if 0
void strccpy(char *dst, char *src,char c)
{
	char *pt=src;
	if(pt==NULL){
		printf("pt==NULL");
		dst[0]='\0';
		return ;
	}
	for(;*pt!=c && *pt!='\0';*dst++=*pt++);
	*dst='\0';

}
#endif
void strccpy2(char *dst, char *src,char *key,char c)
{
	char *pt=strstr(src,key);
	if(pt==NULL){
		dst[0]='\0';
		return ;
	}
	pt+=strlen(key);
	for(;*pt!=c && *pt!='\0';*dst++=*pt++);
	*dst='\0';

}
/*
 * smtp server:
 *      smtp.domain.xxx
 *      smtp.mail.domian.xxx    ==> username@domain.xxx
 * because some smtp server not allowed to send mail by anonymous
 */
void fillsender(void)
{
    char *smtp_prefix[] = {
                            "smtp.mail.",
                            "smtp.",
                            NULL,
                          };
    int i, len, pos;

    /* If the sender's e-mail address and the recipient's e-mail addresses are the same
     * strato.de mail server can not send e-mail */
    pos = strlen(conf.mail_server) - strlen("strato.de");
    if(pos >= 0 && strcmp(conf.mail_server+pos,"strato.de") == 0)
    {
    	strcpy(conf.mail_sender,"\"\"");
    	return;
    }

    for (i = 0; smtp_prefix[i] != NULL; i++) {
        len = strlen(smtp_prefix[i]);
        if(strncmp(smtp_prefix[i], conf.mail_server, len) == 0) {
            sprintf(conf.mail_sender, "%s@%s", conf.mail_login, conf.mail_server + len);
            return;
        }
    }
    strcpy(conf.mail_sender ,conf.mail_receiver);
    return;
}

void logMessage (int pri, char *msg)
{
	time_t now;
	char *wday[7]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
	struct tm *st;
	//char *timestamp;
	char timestamp[32]="";
	char res[20] = "";
	CODE *c_fac;
	int fac=0;
	int send_mail=0;
	int do_log=1;

	if (pri!=0){
		for (c_fac = facilitynames;
			c_fac->c_name && !(c_fac->c_val == LOG_FAC(pri) << 3); c_fac++);
		fac=c_fac->c_val;
		//for (c_pri = prioritynames;
		//	c_pri->c_name && !(c_pri->c_val == LOG_PRI(pri)); c_pri++);
	}

	/* check this log  */
	if(check_log(fac)==FALSE)
		return;

	/* if msg had time stamp ,remove it*/
	if (strlen(msg) > 16 && msg[3] == ' ' && msg[6] == ' ' &&
			msg[9] == ':' && msg[12] == ':' && msg[15] == ' ')
		msg+=16;

	/* we don't need log from pppd */
	if(strstr(msg,"ppp0 -> "))
		return;

#ifdef DEBUG_CALL_TRACE
	/* log Call Trace for debug */
	if(strstr(msg,"Call Trace")){
		//message("<DEBUG> - %s\n",msg);
		system("/bin/dmesg > /var/log/messages");
		exit(0);
	}
#endif
	while(*msg!='\0' && *msg++!=':');
	/* handle kern message*/
	if(fac==LOG_LOCAL0){

		char proto[8]="",src[16]="",dst[128]="",spt[8]="",dpt[8]="";
		char prefix[32]="";
		char *pt;
		int i;

		if(((pt=strchr(msg,'['))==NULL) || (strchr(msg,']')==NULL))
			return;

		for(i=0,pt;*pt!=']';prefix[i++]=*pt++);
		prefix[i++]=']';
		prefix[i]='\0';

		strccpy2(proto,msg,"PROTO=",0x20);
		if(strcmp(proto,"47")==0){
			strcpy(proto,"GRE");
		}else if(strcmp(proto,"50")==0){
			strcpy(proto,"ESP");
		}else if(strcmp(proto,"51")==0){
			strcpy(proto,"AH");
		}else if(strcmp(proto,"115")==0){
			strcpy(proto,"L2TP");
		}

		strccpy2(src,msg,"SRC=",0x20);
		strccpy2(dst,msg,"DST=",0x20);
		strccpy2(spt,msg,"SPT=",0x20);
		strccpy2(dpt,msg,"DPT=",0x20);

		if(strcmp(prefix,"[BLOCK]")==0){
			struct hostent *host;
			struct in_addr addr;
			addr.s_addr=inet_addr(dst);
			host=gethostbyaddr((char *)&addr,4,AF_INET);
			if(host){
				strcpy(dst,host->h_name);
			}
		}
		pt=msg;
		if(spt[0]!='\0' && strcmp(prefix,"[PORT SCAN]")!=0)
		{
			sprintf(pt,"%s Packet - Source:%s,%s Destination:%s,%s - %s"
			,proto,src,spt,dst,dpt,prefix);
		}
		else if(strcmp(prefix,"[WLAN ACL]") == 0)
		{
			msg += strlen(prefix)+1;
		}
		else if(!strcmp(prefix, "[SITE BLOCK]") || !strcmp(prefix, "[SITE ALLOW]"))
		{
			/* when Keyword Blocking is enabled, we will show "site block" or "site allow" logs */
			msg ++;//message from printk start with space, so we should backward
		}
		else
		{
			sprintf(pt,"%s Packet - Source:%s Destination:%s - %s"
			,proto,src,dst,prefix);
		}
#if 0
		pt=msg;

		sprintf(pt,"%s Packet - Source:%s Destination:%s - %s",proto,src,dst,prefix);
#endif
		if(strstr(conf.mail_keyword,prefix)){
			send_mail=1;
		}

		if(strstr(conf.log_keyword,prefix)==NULL){
			do_log=0;
		}
		if(strstr(prefix,"match")){
			do_log=1;
		}

	}else
		msg++;

	if(strcmp(last_log,msg)==0){
       return ;
       }else{
		strcpy(last_log,msg);
	}

	/* time stamp */
	//setenv("TZ",conf.TZ,1);
	time(&now);
	//now+=atoi(conf.daylight)*3600;
	unsetenv("TZ");
	st=localtime(&now);
	sprintf(timestamp,"%s, %d-%02d-%02d %02d:%02d:%02d"
			,wday[st->tm_wday]
			,st->tm_year+1900
			,st->tm_mon+1
			,st->tm_mday
			,st->tm_hour
			,st->tm_min
			,st->tm_sec);

	//if(st)
	//	free(st);

	if(send_mail==1 && conf.mail_enable==1){
				char cmd[1024]="";
				FILE *fp;
				unsigned int flen=0;
				sem_down(s_semid);
				fp=fopen("/var/log/alert","a+");

				if(fp==NULL){
					sem_up(s_semid);
					return ;
				}
				fprintf(fp,"%s\n",msg);
				fseek(fp,0,SEEK_END);
				flen=ftell(fp);
				fclose(fp);

				if(conf.mail_enable_auth==1)
				sprintf(cmd,"/usr/sbin/smtpc -t %s -d %d -m -h %s -r %s -f %s -s \"%s\" -U %s -P %s </var/log/alert "
				,conf.TZ
				,atoi(conf.daylight)
				,conf.mail_server
				,conf.mail_receiver
                ,conf.mail_sender
				,conf.mail_subject_alert
				,conf.mail_login
				,conf.mail_password);
				else
				sprintf(cmd,"/usr/sbin/smtpc -t %s -d %d -m -h %s -r %s -f %s -s \"%s\" </var/log/alert "
				,conf.TZ
				,atoi(conf.daylight)
				,conf.mail_server
				,conf.mail_receiver
                ,conf.mail_sender
				,conf.mail_subject_alert);


				if((now-last_send_mail) > ALERT_MAX_INTERVAL
						|| flen > 1500 ){
					system(cmd);
					last_send_mail=now;
					unlink("/var/log/alert");
				}
				sem_up(s_semid);
	}

	if(do_log==0)
		return ;

	/* todo: supress duplicates */
#ifdef BB_FEATURE_REMOTE_LOG
	/* send message to remote logger */
	if ( (-1 != remotefd) && (doRemoteLog==TRUE)){
static const int IOV_COUNT = 2;
		struct iovec iov[IOV_COUNT];
		struct iovec *v = iov;

		memset(&res, 0, sizeof(res));
		snprintf(res, sizeof(res), "<%d>", pri);
		v->iov_base = res ;
		v->iov_len = strlen(res);
		v++;

		v->iov_base = msg;
		v->iov_len = strlen(msg);
writev_retry:
		if ( -1 == writev(remotefd,iov, IOV_COUNT)){
			if (errno == EINTR) goto writev_retry;
			error_msg_and_die("cannot write to remote file handle on"
					"%s:%d",RemoteHost,RemotePort);
		}
	}


//	if (local_logging == TRUE)
#endif
#ifdef SHOW_HOSTNAME
		/* now spew out the message to wherever it is supposed to go */
		message("%s %s %s %s\n", timestamp, LocalHostName, res, msg);
#else
		message("%s - %s\n", timestamp, msg);
#endif
}

static void quit_signal(int sig)
{
	//logMessage(LOG_SYSLOG | LOG_INFO, "System log daemon exiting.");
	unlink(lfile);
#ifdef BB_FEATURE_IPC_SYSLOG
	ipcsyslog_cleanup();
#endif
	exit(TRUE);
}
static int log_start=0;

static inline void router_start()
{
	struct sysinfo info;
	time_t now;
	char *wday[7]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
	struct tm *st;
	char timestamp[128];

	if(log_start==1)
		return ;

	sysinfo(&info);
	/* time stamp */
//	setenv("TZ",conf.TZ,1);
	time(&now);
	now-=info.uptime;
	unsetenv("TZ");
	st=localtime(&now);
	sprintf(timestamp,"%s, %d-%02d-%02d %02d:%02d:%02d"
			,wday[st->tm_wday]
			,st->tm_year+1900
			,st->tm_mon+1
			,st->tm_mday
			,st->tm_hour
			,st->tm_min
			,st->tm_sec);

	//if(st)
	//	free(st);

	log_start=1;
	message("%s - %s\n", timestamp, "Router start up");
}

static void domark(int sig)
{
	if(log_start==0){
		router_start();
	}
	last_log[0]='\0';
	alarm(MarkInterval);
}

/* This must be a #define, since when DODEBUG and BUFFERS_GO_IN_BSS are
 * enabled, we otherwise get a "storage size isn't constant error. */
static int serveConnection (char* tmpbuf, int n_read)
{
	char *p = tmpbuf;

	while (p < tmpbuf + n_read) {

		int           pri = (LOG_USER | LOG_NOTICE);
		char          line[ MAXLINE + 1 ];
		unsigned char c;
		int find=0;

		char *q = line;

		while ( (c = *p) && q < &line[ sizeof (line) - 1 ]) {
			if (c == '<' && find==0) {
			/* Parse the magic priority number. */
				pri = 0;
				find= 1;
				while (isdigit (*(++p))) {
					pri = 10 * pri + (*p - '0');
				}
				if (pri & ~(LOG_FACMASK | LOG_PRIMASK)){
					pri = (LOG_USER | LOG_NOTICE);
				}
			} else if (c == '\n') {
				*q++ = ' ';
			} else if (iscntrl (c) && (c < 0177)) {
				*q++ = '^';
				*q++ = c ^ 0100;
			} else {
				*q++ = c;
			}
			p++;
		}
		*q = '\0';
		p++;
		/* Now log it */
		logMessage (pri, line);
	}
	return n_read;
}


#ifdef BB_FEATURE_REMOTE_LOG
static void init_RemoteLog (void)
{

  struct sockaddr_in remoteaddr;
  struct hostent *hostinfo;
  int len = sizeof(remoteaddr);
  int so_bc=1;

  memset(&remoteaddr, 0, len);

  remotefd = socket(AF_INET, SOCK_DGRAM, 0);

  if (remotefd < 0) {
    error_msg_and_die("cannot create socket");
  }

  remoteaddr.sin_family = AF_INET;

  /* Ron */
  /* allow boardcast */
  setsockopt(remotefd,SOL_SOCKET,SO_BROADCAST,&so_bc,sizeof(so_bc));
  hostinfo = gethostbyname(RemoteHost);
  remoteaddr.sin_addr = *(struct in_addr *) *hostinfo->h_addr_list;
  remoteaddr.sin_port = htons(RemotePort);

  /*
     Since we are using UDP sockets, connect just sets the default host and port
     for future operations
  */
  if ( 0 != (connect(remotefd, (struct sockaddr *) &remoteaddr, len))){
     error_msg_and_die("cannot connect to remote host %s:%d", RemoteHost, RemotePort);
  }
}
#endif

static void doSyslogd (void) __attribute__ ((noreturn));
static void doSyslogd (void)
{
	struct sockaddr_un sunx;
	socklen_t addrLength;

	int sock_fd;
	fd_set fds;

	/* Set up signal handlers. */
	signal (SIGINT,  quit_signal);
	signal (SIGTERM, quit_signal);
	signal (SIGQUIT, quit_signal);
	signal (SIGHUP,  send_mail_signal);
	signal (SIGUSR1, clear_signal);
	signal (SIGUSR2, reload_signal);
	signal (SIGCHLD,  SIG_IGN);
#ifdef SIGCLD
	signal (SIGCLD,  SIG_IGN);
#endif
	signal (SIGALRM, domark);
	//wait ntp get correct time
	alarm (MarkInterval);

	/* Create the syslog file so realpath() can work. */
	if (realpath (_PATH_LOG, lfile) != NULL)
		unlink (lfile);

	memset (&sunx, 0, sizeof (sunx));
	sunx.sun_family = AF_UNIX;
	strncpy (sunx.sun_path, lfile, sizeof (sunx.sun_path));
	if ((sock_fd = socket (AF_UNIX, SOCK_DGRAM, 0)) < 0)
		perror_msg_and_die ("Couldn't get file descriptor for socket " _PATH_LOG);

	addrLength = sizeof (sunx.sun_family) + strlen (sunx.sun_path);
	if (bind(sock_fd, (struct sockaddr *) &sunx, addrLength) < 0)
		perror_msg_and_die ("Could not connect to socket " _PATH_LOG);

	if (chmod (lfile, 0666) < 0)
		perror_msg_and_die ("Could not set permission on " _PATH_LOG);


#ifdef BB_FEATURE_IPC_SYSLOG
	if (circular_logging == TRUE ){
	   ipcsyslog_init();
	}
#endif

#if 0
#ifdef BB_FEATURE_REMOTE_LOG
	if (doRemoteLog == TRUE){
	  init_RemoteLog();
	}
#endif
#endif

	for (;;) {

		FD_ZERO (&fds);
		FD_SET (sock_fd, &fds);

		if (select (sock_fd+1, &fds, NULL, NULL, NULL) < 0) {
			if (errno == EINTR) {
				/* alarm may have happened. */
				continue;
			}
			perror_msg_and_die ("select error");
		}

		if (FD_ISSET (sock_fd, &fds)) {
		       int   i;
		       RESERVE_BB_BUFFER(tmpbuf, BUFSIZ + 1);

		       memset(tmpbuf, '\0', BUFSIZ+1);
		       if ( (i = recv(sock_fd, tmpbuf, BUFSIZ, 0)) > 0) {
			       serveConnection(tmpbuf, i);
		       } else {
			       perror_msg_and_die ("UNIX socket error");
		       }
		       RELEASE_BB_BUFFER (tmpbuf);
		}/* FD_ISSET() */
	} /* for main loop */
}

char *config_file_path;

int parse_config(char *conf_path);

static void clear_signal(int sig)
{
	buf->head=0;
	buf->tail=0;
}

static void reload_signal(int sig)
{
	parse_config(config_file_path);
}

int parse_config(char *conf_path)
{
		FILE *fp;
		char buf[1024];
		int i=0;
		char rhip[16]="";
#ifdef DEBUG
		printf("conf_path==%s\n",conf_path);
#endif
		if(conf_path==NULL)
			fp=fopen(__CONF_FILE,"r");
		else
			fp=fopen(conf_path,"r");

		if(fp==NULL)
			return FALSE;

		fread(buf,1024,1,fp);
		fclose(fp);

		/* initial conf */
		bzero(&conf,sizeof(conf));
		memset(&conf.log_list,-1,sizeof(conf.log_list));
		/* initial conf */

		if(strstr(buf,"mail_enable=1")) conf.mail_enable=1;
		if(strstr(buf,"mail_enable_auth=1")) conf.mail_enable_auth=1;

		strccpy2(conf.TZ,buf,"TZ=",'\n');

		//setenv("TZ",conf.TZ,1);
		unsetenv("TZ");

		strccpy2(conf.daylight,buf,"daylight=",'\n');
		strccpy2(rhip,buf,"log_remote_ip=",'\n');

		if(strlen(rhip)>2){
				if(RemoteHost) free(RemoteHost);
				RemoteHost=strdup(rhip);
				doRemoteLog = TRUE;
				if(remotefd > 0) close(remotefd);
	  			init_RemoteLog();
		}else
				doRemoteLog = FALSE;

		strccpy2(conf.log_keyword,buf,"log_keyword=",'\n');

		/* if email is not enable ,we don't need to parser those config*/
		if(conf.mail_enable==1){
			if(strstr(buf,"mail_log_full=1")) conf.mail_log_full=1;

			strccpy2(conf.mail_server,buf,"mail_server=",'\n');
			strccpy2(conf.mail_receiver,buf,"mail_receiver=",'\n');
			strccpy2(conf.mail_subject,buf,"mail_subject=",'\n');
			strccpy2(conf.mail_subject_alert,buf,"mail_subject_alert=",'\n');
			strccpy2(conf.mail_keyword,buf,"mail_keyword=",'\n');
			strccpy2(conf.mail_login,buf,"mail_login=",'\n');
			strccpy2(conf.mail_password,buf,"mail_password=",'\n');
	        fillsender();
		}

		i=0;
		if(strstr(buf,"log_syslog=1")) conf.log_list[i++]=LOG_SYSLOG;
		if(strstr(buf,"log_local0=1")) conf.log_list[i++]=LOG_LOCAL0;
		if(strstr(buf,"log_auth=1")) conf.log_list[i++]=LOG_AUTH;

		conf.log_list[i]=0;

		return TRUE;
}

int syslogd_main(int argc, char **argv)
//int main(int argc, char **argv)
{
	int opt;
#if ! defined(__uClinux__)
	int doFork = TRUE;
#endif

	char *p;


	/* do normal option parsing */
	while ((opt = getopt(argc, argv, "m:nO:Rf:LC")) > 0) {
		switch (opt) {
			case 'm':
				MarkInterval = atoi(optarg) * 60;
				break;
#if ! defined(__uClinux__)
			case 'n':
				doFork = FALSE;
				break;
#endif
			case 'O':
				logFilePath = strdup(optarg);
				break;
#ifdef BB_FEATURE_REMOTE_LOG
			case 'R':
				if(RemoteHost!=NULL) free(RemoteHost);
				RemoteHost = strdup(optarg);
				if ( (p = strchr(RemoteHost, ':'))){
					RemotePort = atoi(p+1);
					*p = '\0';
				}
				doRemoteLog = TRUE;
				break;
			case 'L':
				local_logging = TRUE;
				break;
#endif
#ifdef BB_FEATURE_IPC_SYSLOG
			case 'C':
				circular_logging = TRUE;
				break;
#endif
			case 'f':
				config_file_path=optarg;
				if(parse_config(optarg)==FALSE)
					show_usage();
				break;

			default:
				show_usage();
		}
	}
#if 0
#ifdef BB_FEATURE_REMOTE_LOG
	/* If they have not specified remote logging, then log locally */
	if (doRemoteLog == FALSE)
		local_logging = TRUE;
#endif
#endif

#ifdef SHOW_HOSTNAME
//	}
	/* Store away localhost's name before the fork */
	gethostname(LocalHostName, sizeof(LocalHostName));
	if ((p = strchr(LocalHostName, '.'))) {
		*p++ = '\0';
	}
#endif
	umask(0);

#if ! defined(__uClinux__)
	if (doFork == TRUE) {
		if (daemon(0, 1) < 0)
			perror_msg_and_die("daemon");
	}
#endif
	doSyslogd();

	return EXIT_SUCCESS;
}
#if 1
extern int klogd_main (int argc ,char **argv);

int main(int argc ,char **argv)
{
	int ret = 0;
	char *base = strrchr(argv[0], '/');

	if (strstr(base ? (base + 1) : argv[0], "syslogd"))
		ret = syslogd_main(argc,argv);
	else if (strstr(base ? (base + 1) : argv[0], "klogd"))
		ret = klogd_main(argc,argv);
	else
		show_usage();

	return ret;
}
#endif
/*
Local Variables
c-file-style: "linux"
c-basic-offset: 4
tab-width: 4
End:
*/
