#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <signal.h>
#include <string.h>
#include <syslog.h>
#include <atm.h>
#include <linux/atmdev.h>
#include <linux/atmbr2684.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/if_ether.h>
#ifndef BR2684_FLAG_ROUTED
#warning "Kernel missing routed support for br2684"
#define BR2684_FLAG_ROUTED    (1<<16) /* payload is routed, not bridged */
#endif

/* Written by Marcell GAL <cell@sch.bme.hu> to make use of the */
/* ioctls defined in the br2684... kernel patch */
/* Compile with cc -o br2684ctl br2684ctl.c -latm */

/*
  Modified feb 2001 by Stephen Aaskov (saa@lasat.com)
  - Added daemonization code
  - Added syslog
  
  6/6/2008	Arvind		Fixed the code to handle QoS parameters correctly.
				Fixed the condition where if the PVC creation fails, then the netdevice is not created.
  23/07/2008	Lin Mars	Support removal of interface once process is killed.
				Fix incorrect handle of opened socket
*/


#define LOG_NAME       "br2684ctl"
#define LOG_OPTION     LOG_PERROR|LOG_PID
#define LOG_FACILITY   LOG_LOCAL2

struct br2684_params {
  int itfnum;
  int encap;
  int sndbuf;
  int payload;
  char *astr; /* temporary */
  struct atm_qos reqqos;
  int is_addr;
  struct sockaddr addr;
};


int lastsock, lastitf;


void fatal(const char *str, int err)
{
  syslog (LOG_ERR,"Fatal: %s; %s", str, strerror(err));
  //exit(-2);
}

int remove_brif(int fd, int itfnum)
{
  struct atm_backend_br2684 be;
  int err = -1;

  if (fd >= 0) {
    be.backend_num = ATM_BACKEND_BR2684;
    be.ifspec.method = BR2684_FIND_BYIFNAME;
    sprintf(be.ifspec.spec.ifname, "nas%d", itfnum);
    err = ioctl (fd, ATM_DELBACKENDIF, &be);
    if (err == 0)
        syslog (LOG_INFO,"Interface nas%d removed", itfnum);
    else {
        syslog (LOG_ERR,"Could not remove interface nas%d :%s", itfnum, strerror(errno));
        close(fd);
        return(-1);
    }
    close(fd);
  }
  return 0;
}

int create_pidfile(int num)
{
  FILE *pidfile = NULL;
  char name[32];

  if (num < 0) return -1;

  snprintf(name, 32, "/var/run/br2684ctl-nas%d.pid", num);
  pidfile = fopen(name, "w");
  if (pidfile == NULL) return -1;
  fprintf(pidfile, "%d", getpid());
  fclose(pidfile);

  return 0;
}

int remove_pidfile(int num)
{
  FILE *pidfile = NULL;
  char name[32];

  if (num < 0) return -1;

  snprintf(name, 32, "/var/run/br2684ctl-nas%d.pid", num);
  pidfile = fopen(name, "r");
  if (pidfile == NULL) 
	  return -1;
  fclose(pidfile);
  remove(name);

  return 0;
}

void int_signal(int dummy)
{
  syslog (LOG_ERR,"Killed by a signal");
  remove_brif(lastsock, lastitf);
  remove_pidfile(lastitf);
  exit(0);
}

void exitFunc(void)
{
  syslog (LOG_ERR,"Daemon terminated");
  remove_brif(lastsock, lastitf);
  remove_pidfile(lastitf);
}


/* AV: 
	Moved to assign_vcc as this causes the creation of a ethernet (netdevice) which is cannot be used if the vcc
	creation fails.
*/
#if 0
int create_br(int itfnum, int payload)
{
  int err;
  
  if(lastsock < 0) 
  {
    lastsock = socket(PF_ATMPVC, SOCK_DGRAM, ATM_AAL5);
  }
  if (lastsock < 0) 
  {
    syslog(LOG_ERR, "socket creation failed: %s",strerror(errno));
  } 
  else 
  {
    /* create the device with ioctl: */
    if(itfnum >=0 && itfnum < 1234567890)
	{
      struct atm_newif_br2684 ni;
      ni.backend_num = ATM_BACKEND_BR2684;
      ni.media = BR2684_MEDIA_ETHERNET;
#ifdef BR2684_FLAG_ROUTED
      if (payload == 0)
        ni.media |= BR2684_FLAG_ROUTED;
#endif
      ni.mtu = 1500;
      sprintf(ni.ifname, "nas%d", itfnum);
      err = ioctl (lastsock, ATM_NEWBACKENDIF, &ni);
  
      if (err == 0)
		syslog(LOG_NOTICE, "Interface \"%s\" created sucessfully",ni.ifname);
      else
		syslog(LOG_INFO, "Interface \"%s\" could not be created, reason: %s",
				ni.ifname,
				strerror(errno));
      lastitf = itfnum;	/* even if we didn't create, because existed, 
			assign_vcc will want to know it! */
    } 
	else 
	{
      syslog(LOG_ERR,"err: strange interface number %d", itfnum );
    }
  }
  return lastsock;
}
#endif //#if 0

int assign_vcc(char *astr, int encap, int payload,
               int bufsize, struct atm_qos qos, int itfnum, struct sockaddr *hwaddr)
{
    int err;
    struct sockaddr_atmpvc addr;
    int fd, skfd;
    struct atm_backend_br2684 be;
    struct ifreq ifr;

    if(itfnum < 0 || itfnum > 1234567890)
    {
        syslog(LOG_ERR,"Invalid interface number requested %d",itfnum);
        return(-1);
    }

    memset(&addr, 0, sizeof(addr));
    err = text2atm(astr,(struct sockaddr *)(&addr), sizeof(addr), T2A_PVC);
    if (err!=0)
    {
        syslog(LOG_ERR,"Could not parse ATM parameters (error=%d)",err);
        return(-2);
    }
    
    syslog(LOG_NOTICE,"Communicating over ATM %d.%d.%d, encapsulation: %s",
	   addr.sap_addr.itf,
	   addr.sap_addr.vpi,
	   addr.sap_addr.vci,
	   encap?"VC mux":"LLC");
    
    if ((fd = socket(PF_ATMPVC, SOCK_DGRAM, ATM_AAL5)) < 0)
    {
        syslog(LOG_ERR,"failed to create socket %d, reason: %s", errno,strerror(errno));
        return(-3);
    }

	
    if (qos.aal == 0) {
        qos.aal                     = ATM_AAL5;
        /* AV: This should not overide the default QoS. */
        //qos.txtp.traffic_class      = ATM_UBR;
        qos.txtp.max_sdu            = 1524;
        /* AV: This should not overide the default QoS parameters. */
        //qos.txtp.pcr                = ATM_MAX_PCR;
        qos.rxtp = qos.txtp;
    }

    if ( (err = setsockopt(fd,SOL_SOCKET,SO_SNDBUF, &bufsize ,sizeof(bufsize))) )
    {
        syslog(LOG_ERR,"setsockopt SO_SNDBUF: (%d) %s",err, strerror(err));
        close(fd);
        return(-4);
    }
    
    if (setsockopt(fd, SOL_ATM, SO_ATMQOS, &qos, sizeof(qos)) < 0)
    {
        syslog(LOG_ERR,"setsockopt SO_ATMQOS %s", strerror(errno));
        close(fd);
        return(-5);
    }

    err = connect(fd, (struct sockaddr*)&addr, sizeof(struct sockaddr_atmpvc));
    
    if (err < 0)
    {
        syslog(LOG_ERR,"failed to connect on socket %s", strerror(errno));
        close(fd);
        return(-6);
    }
	
    /* AV: moved here from create_br() */
    /* attach the vcc to device after creating the device with ioctl: */
    if(itfnum >=0 && itfnum < 1234567890)
    {
        struct atm_newif_br2684 ni;
        ni.backend_num = ATM_BACKEND_BR2684;
        ni.media = BR2684_MEDIA_ETHERNET;
#ifdef BR2684_FLAG_ROUTED
        if (payload == 0)
            ni.media |= BR2684_FLAG_ROUTED;
#endif
        ni.mtu = 1500;
        sprintf(ni.ifname, "nas%d", itfnum);
        err = ioctl (fd, ATM_NEWBACKENDIF, &ni);

        if (err == 0)
            syslog(LOG_NOTICE, "Interface \"%s\" created sucessfully",ni.ifname);
        else
            syslog(LOG_INFO, "Interface \"%s\" could not be created, reason: %s",
                ni.ifname, strerror(errno));
        lastitf = itfnum;   /* even if we didn't create, because existed, 
                               someone else will want to know it! */
    }
    else 
    {
        syslog(LOG_ERR,"err: strange interface number %d", itfnum);
        return(-7);
    }
		
    be.backend_num = ATM_BACKEND_BR2684;
    be.ifspec.method = BR2684_FIND_BYIFNAME;
    sprintf(be.ifspec.spec.ifname, "nas%d", itfnum);
    be.fcs_in = BR2684_FCSIN_NO;
    be.fcs_out = BR2684_FCSOUT_NO;
    be.fcs_auto = 0;
    be.encaps = encap ? BR2684_ENCAPS_VC : BR2684_ENCAPS_LLC;
    be.has_vpiid = 0;
    be.send_padding = 0;
    be.min_size = 0;
    if (hwaddr != NULL) {
        be.has_addr = 1;
	memcpy(&(be.addr), hwaddr, sizeof(struct sockaddr));
    }
    err = ioctl (fd, ATM_SETBACKEND, &be);
    if (err == 0)
    {
        syslog (LOG_INFO,"Interface configured");
    }
    else 
    {
        syslog (LOG_ERR,"Could not configure interface:%s",strerror(errno));
        close(fd);
        return(-8);
    }

    /* AV: Bring up the interface by default. */
    skfd = socket(AF_INET, SOCK_DGRAM, 0);
    sprintf(ifr.ifr_name, "nas%d", itfnum);
    err = ioctl(skfd, SIOCGIFFLAGS, &ifr);
    if(err)
        fprintf(stderr, "SIOCGIFFLAGS:%d\n",err);
    sprintf(ifr.ifr_name, "nas%d", itfnum);
    ifr.ifr_flags |= IFF_UP | IFF_RUNNING;
    err = ioctl(skfd, SIOCSIFFLAGS, &ifr);
    if(err)
        fprintf(stderr, "SIOCGIFFLAGS:%d\n",err);
    close(skfd);

    return fd ;
}

int start_interface(struct br2684_params* params)
{
    int vcc_sock = 0;

    if (params->astr==NULL) {
        syslog(LOG_ERR, "Required ATM parameters not specified.");
        exit(1);
    }

    /* AV: Removing this for now as it is moved into the assign_vcc(). */
    /*br_sock = create_br(params->itfnum, params->payload);
    if(br_sock)
    {*/
        if (params->is_addr == 1)
            vcc_sock = assign_vcc(params->astr, params->encap, params->payload, params->sndbuf, params->reqqos, params->itfnum, &(params->addr));
        else
            vcc_sock = assign_vcc(params->astr, params->encap, params->payload, params->sndbuf, params->reqqos, params->itfnum, NULL);
    //}
	
    return vcc_sock;
}

void usage(char *s)
{
  printf("usage: %s [-b] [[-c number] [-e 0|1] [-m hwaddr] [-s sndbuf] [-q qos] [-p 0|1] "
	 "[-a [itf.]vpi.vci]*]*\n", s);
  printf("  encapsulations: 0=llc, 1=vcmux\n  payloads: 0=routed, 1=bridged\n");
  exit(1);
}

/* in_ether() is copied from busybox's ifconfig.c */
static int in_ether(const char *bufp, struct sockaddr *sap)
{
	char *ptr;
	int i, j;
	unsigned char val;
	unsigned char c;

	sap->sa_family = ARPHRD_ETHER;
	ptr = sap->sa_data;

	i = 0;
	do {
		j = val = 0;

		/* We might get a semicolon here - not required. */
		if (i && (*bufp == ':')) {
			bufp++;
		}

		do {
			c = *bufp;
			if (((unsigned char)(c - '0')) <= 9) {
				c -= '0';
			} else if (((unsigned char)((c|0x20) - 'a')) <= 5) {
				c = (c|0x20) - ('a'-10);
			} else if (j && (c == ':' || c == 0)) {
				break;
			} else {
				return -1;
			}
			++bufp;
			val <<= 4;
			val += c;
		} while (++j < 2);
		*ptr++ = val;
	} while (++i < ETH_ALEN);

	return *bufp; /* Error if we don't end at end of string. */
}

int main (int argc, char **argv)
{
  int c, background=0;
  int vcc_sock = 0;

  struct br2684_params params;
  params.itfnum=-1;
  params.encap=0;
  params.sndbuf=8192;
  params.payload=1;
  params.astr=NULL;
  memset(&params.reqqos, 0, sizeof(params.reqqos));
  
  lastsock = -1;
  lastitf = 0;
  
  /* st qos to 0 */

  openlog (LOG_NAME,LOG_OPTION,LOG_FACILITY);
  if (argc > 1)
    while ((c = getopt(argc, argv,"q:a:bc:e:s:p:m:?h")) !=EOF)
      switch (c) {
      case 'q':
	//printf ("optarg : %s",optarg);
	if (text2qos(optarg,&params.reqqos,0))
	  fprintf(stderr,"QOS parameter invalid\n");
	break;
      case 'a':
	params.astr=optarg;
	break;
      case 'b':
	background=1;
	break;
      case 'c':
	/* temporary, to make it work with multiple interfaces: */
	/* AV: Hack breaks things. */
	/*if (params.itfnum >= 0) 
	  {
		vcc_sock = start_interface(&params);
		if(vcc_sock < 0)
		  {
			 close(lastsock);
			 lastsock = -1;
		  }
	  }*/
	params.itfnum= atoi(optarg);
	break;
      case 'e':
	params.encap=(atoi(optarg));
	if(params.encap<0){
	  syslog (LOG_ERR, "invalid encapsulation: %s:",optarg);
	  params.encap=0;
	}
	break;
      case 's':
	params.sndbuf=(atoi(optarg));
	if(params.sndbuf<0){
	  syslog(LOG_ERR, "Invalid sndbuf: %s, using size of 8192 instead", 
		 optarg);
	  params.sndbuf=8192;
	}
	break;
      case 'p':	/* payload type: routed (0) or bridged (1) */
#ifdef BR2684_FLAG_ROUTED
	params.payload = atoi(optarg);
	break;
#else
	syslog(LOG_ERR, "payload option not supported.");
#endif
      case 'm':
	in_ether(optarg, &(params.addr));
	params.is_addr = 1;
        break;
      case '?':
      case 'h':
      default:
	usage(argv[0]);
      }
  else
    usage(argv[0]);


  if (argc != optind) 
	  usage(argv[0]);
  
  if(vcc_sock <= 0)
	vcc_sock = start_interface(&params);  

  if(lastsock >= 0) 
	  close(lastsock);

  /* AV: Please do not fork if the PVC creation/configuration has failed. */
  if(vcc_sock <=0)
  {
	syslog(LOG_ERR,"creating PVC failed = %d \n", vcc_sock);
	return -1;
  }
  lastsock = vcc_sock;
 
  if (background) {
    pid_t pid;
    
    pid = fork();
    if (pid < 0) {
      fprintf(stderr,"Error detaching\n");
	  return 2;
    } else if (pid) 
	  {
		// This is the parent
		return 0;
	  }
    
    // Become a process group and session group leader
    if (setsid()<0) {
      fprintf (stderr,"Could not set process group\n");
	  return 2;
    }
    
    // Fork again to let process group leader exit
    pid = fork();
    if (pid < 0) {
      fprintf(stderr,"Error detaching during second fork\n");
	  return 2;
    } else if (pid)
      {
		return 0;
	  }
    
    // Now we're ready for buisness
    chdir("/");            // Don't keep directories in use
    close(0); close(1); close(2);  // Close stdin, -out and -error
    /*
      Note that this implementation does not keep an open 
      stdout/err.
      If we need them they can be opened now
    */
    
  }
  
  create_pidfile(params.itfnum);
  signal(SIGINT, int_signal);
  signal(SIGTERM, int_signal);

  syslog (LOG_INFO, "RFC 1483/2684 bridge daemon started");
  atexit (exitFunc);
  
  while (1) pause();	/* to keep the sockets... */
  return 0;
}

